/*
 * Nightshade (TM) astronomy simulation and visualization
 *
 * Copyright (C) 2005 Robert Spearman
 * Copyright (C) 2009 Digitalis Education Solutions, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Nightshade is a trademark of Digitalis Education Solutions, Inc.
 * See the TRADEMARKS file for trademark usage requirements.
 *
 */

// manage an audio track (SDL mixer music track)

#include <iostream>
#include "audio.h"
#include "nightshade.h"
#include "translator.h"

#include <sstream>


#ifdef HAVE_SDL_MIXER_H

float Audio::master_volume = 1.0;

bool Audio::is_disabled = 0;


Audio::Audio(std::string filename, std::string name, long int output_rate)
{

	if (output_rate < 1000) output_rate = 22050;

	// initialize audio
	//	if(Mix_OpenAudio(output_rate, MIX_DEFAULT_FORMAT, 2, 4096)) {
	if (Mix_OpenAudio(output_rate, MIX_DEFAULT_FORMAT, 2, 2048)) {
		printf("Unable to open audio output!\n");
		track = NULL;
		return;
	}

	// Mix_VolumeMusic(int(MIX_MAX_VOLUME*master_volume));
	Mix_VolumeMusic(MIX_MAX_VOLUME);

	track = Mix_LoadMUS(filename.c_str());
	if (track == NULL) {
		is_playing = 0;
		std::cout << "Could not load audio file " << filename << "\n";
	} else is_playing = 1;

	if (is_disabled) disable();

	track_name = name;

}

Audio::~Audio()
{
	Mix_HaltMusic(); // stop playing
	Mix_FreeMusic(track);  // free memory

	// stop audio use
	Mix_CloseAudio();
}

void Audio::play(bool loop)
{

	// TODO check for load errors

	if (track) {
		is_playing = 1;
		elapsed_seconds = 0;

		if (loop) Mix_PlayMusic(track, -1);
		else Mix_PlayMusic(track, 0);
		//		std::cout << "now playing audio\n";

		if (is_disabled) disable();
	}
}

// used solely to track elapsed seconds of play
void Audio::update(int delta_time)
{

	if (track && is_playing) elapsed_seconds += delta_time/1000.f;

}

// sychronize with elapsed time
// no longer starts playback if paused or disabled
void Audio::sync()
{
	if (track==NULL) return;

	// Mix_RewindMusic();

	if (is_playing && !is_disabled) {
		Mix_SetMusicPosition(elapsed_seconds);  // ISSUE doesn't work for all audio formats
		Mix_ResumeMusic();
	}
	//	printf("Synced audio to %f seconds\n", elapsed_seconds);

}

void Audio::pause()
{
	if (is_playing) Mix_PauseMusic();
	is_playing=0;
}

void Audio::resume()
{
	is_playing=1;
	sync();
}

void Audio::stop()
{
	Mix_HaltMusic();
	is_playing=0;
}

// _volume should be between 0 and 1
void Audio::set_volume(float _volume)
{
	if (_volume >= 0 && _volume <= 1) {
		master_volume = _volume;

#ifdef DESKTOP
		Mix_VolumeMusic(int(MIX_MAX_VOLUME*master_volume));
#else
		std::ostringstream oss;
		oss << "amixer set PCM " << int(0.5 + master_volume*100) << "% >> /dev/null";
		std::string comm = oss.str();
		system(comm.c_str());
#endif

	}
}

void Audio::increment_volume()
{
	master_volume += 0.02f;

	if (master_volume > 1) master_volume = 1;

#ifdef DESKTOP
	Mix_VolumeMusic(int(MIX_MAX_VOLUME*master_volume));
#else
	system("amixer set PCM 2%+ >> /dev/null");
#endif

}

void Audio::decrement_volume()
{
	master_volume -= 0.02f;

	if (master_volume < 0) master_volume = 0;

#ifdef DESKTOP
	Mix_VolumeMusic(int(MIX_MAX_VOLUME*master_volume));
#else
	system("amixer set PCM 2%- unmute >> /dev/null");
#endif

}

// used when ffwd/resume playing a script with audio
void Audio::disable()
{
	if (is_playing) Mix_PauseMusic();
	is_disabled = 1;
}

void Audio::enable()
{

	is_disabled=0;
	sync();
}


#else
// SDL_mixer is not available - no audio
#endif
