#ifndef FILE_SMOOTHER
#define FILE_SMOOTHER

/*********************************************************************/
/* File:   smoother.hh                                               */
/* Author: Joachim Schoeberl                                         */
/* Date:   20. Apr. 2000                                             */
/*********************************************************************/

namespace ngmg
{

  /** 
      Smoothing iteration for multigrid method.
      Pure virtual base class.
  */
  class NGS_DLL_HEADER Smoother
  {
  protected:
    /// additive or multiplicative smooting
    int additive; 
    /// should coarse levels be updated, too ?
    int updateall;
    /// 
    Flags flags;
  public: 
    /// Constructor
    Smoother();
    /// Constructor
    Smoother(const Flags & aflags); 
    /// Destructor
    virtual ~Smoother();
  
    /// Update smoother (fine level or all levels)
    virtual void Update (bool force_update = 0) = 0;

    /// Do steps iterations of pre-smoothing
    virtual void PreSmooth (int level, ngla::BaseVector & u, 
			    const ngla::BaseVector & f, int steps) const = 0;


    /// Do steps iterations of pre-smoothing
    virtual void PreSmoothResiduum (int level, ngla::BaseVector & u, 
				    const ngla::BaseVector & f, 
				    ngla::BaseVector & res, 
				    int steps) const 
    {
      PreSmooth (level, u, f, steps);
      Residuum (level, u, f, res);
    }


    /// Do steps iterations of post-smoothing
    virtual void PostSmooth (int level, ngla::BaseVector & u, 
			     const ngla::BaseVector & f, int steps) const = 0;
    /// Apply the preconditioning action (additive or multiplicative)
    virtual void Precond (int level, const ngla::BaseVector & f, ngla::BaseVector & u) const;

    /// 
    virtual void Residuum (int level, ngla::BaseVector & u, 
			   const ngla::BaseVector & f, 
			   ngla::BaseVector & d) const = 0;

    ///
    void SetAdditive () { additive = 1; }
    ///
    void SetUpdateAll (int ua) { updateall = ua; }
    ///
    void SetMultiplicative () { additive = 0; }
    ///
    int Additive () const { return additive; }

    ///
    virtual ngla::BaseVector * CreateVector(int level) const = 0;

    virtual void MemoryUsage (Array<MemoryUsageStruct*> & mu) const
    { ; }
  };


  /**
     Gauss-Seidel smoother.
     Common relaxation of unknwons in node.
  */
  class GSSmoother : public Smoother
  {
    ///
    const MeshAccess & ma;
    ///
    const BilinearForm & biform;
    ///
    Array<ngla::BaseJacobiPrecond*> jac;
  
  public:
    ///
    GSSmoother (const MeshAccess & ama,
		const BilinearForm & abiform);
    ///
    virtual ~GSSmoother();
  
    ///
    virtual void Update (bool force_update = 0);
    ///
    virtual void PreSmooth (int level, ngla::BaseVector & u, 
			    const ngla::BaseVector & f, int steps) const;
    ///
    virtual void PostSmooth (int level, ngla::BaseVector & u, 
			     const ngla::BaseVector & f, int steps) const;

    virtual void PreSmoothResiduum (int level, ngla::BaseVector & u, 
				    const ngla::BaseVector & f, 
				    ngla::BaseVector & res, 
				    int steps) const;

    ///
    virtual void Residuum (int level, ngla::BaseVector & u, 
			   const ngla::BaseVector & f, ngla::BaseVector & d) const;
    ///
    virtual ngla::BaseVector * CreateVector(int level) const;
  };


  /**
     Anisotropic smoother.
     Common relaxation of vertically aligned nodes.
  */
  class AnisotropicSmoother : public Smoother
  {
    ///
    const MeshAccess & ma;
    ///
    const BilinearForm & biform;
    ///
    Array<BaseBlockJacobiPrecond*> jac;
  
  public:
    ///
    AnisotropicSmoother (const MeshAccess & ama,
			 const BilinearForm & abiform);
    ///
    virtual ~AnisotropicSmoother();
  
    ///
    virtual void Update (bool forace_update = 0);
    ///
    virtual void PreSmooth (int level, ngla::BaseVector & u, 
			    const ngla::BaseVector & f, int steps) const;
    ///
    virtual void PostSmooth (int level, ngla::BaseVector & u, 
			     const ngla::BaseVector & f, int steps) const;
    ///
    virtual void Residuum (int level, ngla::BaseVector & u, 
			   const ngla::BaseVector & f, ngla::BaseVector & d) const;
    ///
    virtual ngla::BaseVector * CreateVector(int level) const;
  };





  /**
     Block-Gauss-Seidel smoother.
     Blocks are defined by underlying FESpace.
  */
  class BlockSmoother : public Smoother
  {
    ///
    const MeshAccess & ma;
    ///
    const BilinearForm & biform;
    ///
    const LinearForm * constraint;
    ///
    Array<ngla::BaseBlockJacobiPrecond*> jac;
    ///
    Array<BaseMatrix*> inv;
    ///
    Array<int> * direct;
  
  public:
    ///
    BlockSmoother (const MeshAccess & ama,
		   const BilinearForm & abiform, const Flags & aflags);
    ///
    BlockSmoother (const MeshAccess & ama,
		   const BilinearForm & abiform,
		   const LinearForm & aconstraint, const Flags & aflags);
    ///
    virtual ~BlockSmoother();
  
    ///
    virtual void Update (bool forace_update = 0);
    ///
    virtual void PreSmooth (int level, BaseVector & u, 
			    const BaseVector & f, int steps) const;
    ///
    virtual void PreSmoothResiduum (int level, ngla::BaseVector & u, 
				    const ngla::BaseVector & f, 
				    ngla::BaseVector & res, 
				    int steps) const;

    ///
    virtual void PostSmooth (int level, ngla::BaseVector & u, 
			     const ngla::BaseVector & f, int steps) const;
    ///
    virtual void Precond (int level, const ngla::BaseVector & f, ngla::BaseVector & u) const;
    ///
    virtual void Residuum (int level, ngla::BaseVector & u, 
			   const ngla::BaseVector & f, ngla::BaseVector & d) const;
    ///
    virtual ngla::BaseVector * CreateVector(int level) const;

    virtual void MemoryUsage (Array<MemoryUsageStruct*> & mu) const;
  };






  /**
     Matrix - vector multiplication by smoothing step.
  */
  class SmoothingPreconditioner : public ngla::BaseMatrix
  {
    ///
    const Smoother & smoother;
    ///
    int level;
  public:
    ///
    SmoothingPreconditioner (const Smoother & asmoother,
			     int alevel = 0);
    ///
    virtual void Mult (const ngla::BaseVector & f, ngla::BaseVector & u) const;
    ///
    virtual ngla::BaseVector * CreateVector () const;
  };

}

#endif
