#include <stdlib.h>
#include "newsraft.h"

static WINDOW *counter_window = NULL;

// We take 999999999 as the maximum value for count variable to avoid overflow
// of the uint32_t integer. The width of this number is hardcoded into the
// terminal width limit, so when changing it here, consider changing the limit.
// 9 (max length of input) + 1 (terminator) = 10
static char count_buf[10];
static uint8_t count_buf_len = 0;
static const struct timespec input_polling_period = {0, 30000000}; // 0.03 seconds

static volatile bool they_want_us_to_break_input = false;

static inline void
counter_update_unprotected(void)
{
	werase(counter_window);
	if (count_buf_len != 0) {
		mvwaddnstr(counter_window, 0, 0, count_buf, count_buf_len);
	}
	wrefresh(counter_window);
}

bool
counter_recreate_unprotected(void)
{
	if (counter_window != NULL) {
		delwin(counter_window);
	}
	counter_window = newwin(1, 9, list_menu_height, list_menu_width - 9);
	if (counter_window == NULL) {
		write_error("Failed to create counter window!\n");
		return false;
	}
	INFO("Created counter window.");
	nodelay(counter_window, TRUE);
	if (keypad(counter_window, TRUE) == ERR) {
		WARN("Can't enable keypad and function keys reading support for counter window!");
	}
	counter_update_unprotected();
	return true;
}

void
tell_program_to_terminate_safely_and_quickly(int dummy)
{
	(void)dummy;
	they_want_us_to_terminate = true;
}

input_id
get_input(struct input_binding *ctx, uint32_t *count, const struct wstring **macro_ptr)
{
	static wint_t c = 0;
	static size_t queued_action_index = 0;
	if (queued_action_index > 0) {
		input_id next = get_action_of_bind(ctx, keyname(c), queued_action_index, macro_ptr);
		if (next != INPUT_ERROR) {
			queued_action_index += 1;
			return next;
		} else {
			queued_action_index = 0;
		}
	}
	while (they_want_us_to_terminate == false) {
		// We have to read input from a child window because
		// reading it from the main window will bring stdscr
		// on top of other windows and overlap them.
		pthread_mutex_lock(&interface_lock);
		int status = wget_wch(counter_window, &c);
		pthread_mutex_unlock(&interface_lock);
		if (status == ERR) {
			if (they_want_us_to_break_input == true) {
				they_want_us_to_break_input = false;
				return INPUT_ERROR;
			}
			nanosleep(&input_polling_period, NULL);
			continue;
		} else if (c == KEY_RESIZE) {
			return resize_handler();
		}
		const char *key = keyname(c);
		INFO("Read key %d (\'%lc\', \"%s\")", c, c, key ? key : "ERROR");
		if (search_mode_is_enabled == true) {
			if (c == '\n' || c == 27) {
				search_mode_is_enabled = false;
				status_clean();
				if (c == '\n') return INPUT_APPLY_SEARCH_MODE_FILTER;
			} else if (c == KEY_BACKSPACE || c == KEY_DC || c == 127) {
				if (search_mode_text_input->len > 0) {
					search_mode_text_input->len -= 1;
					search_mode_text_input->ptr[search_mode_text_input->len] = L'\0';
					update_status_window_content();
				} else {
					search_mode_is_enabled = false;
					status_clean();
				}
			} else {
				wcatcs(search_mode_text_input, c);
				update_status_window_content();
			}
		} else if (ISDIGIT(c)) {
			count_buf_len %= 9;
			count_buf[count_buf_len++] = c;
			pthread_mutex_lock(&interface_lock);
			counter_update_unprotected();
			pthread_mutex_unlock(&interface_lock);
		} else {
			input_id cmd = get_action_of_bind(ctx, key, 0, macro_ptr);
			if (cmd == INPUT_START_SEARCH_INPUT) {
				wstr_set(&search_mode_text_input, L"", 0, 100);
				search_mode_is_enabled = true;
				update_status_window_content();
			} else {
				count_buf[count_buf_len] = '\0';
				if (count != NULL && sscanf(count_buf, "%" SCNu32, count) != 1) {
					*count = 1;
				}
				count_buf_len = 0;
				pthread_mutex_lock(&interface_lock);
				counter_update_unprotected();
				pthread_mutex_unlock(&interface_lock);
				queued_action_index = 1;
				return cmd;
			}
		}
	}
	INFO("Received signal requesting termination of program.");
	return INPUT_QUIT_HARD;
}

void
break_getting_input_command(void)
{
	they_want_us_to_break_input = true;
}

void
counter_delete(void)
{
	if (counter_window != NULL) {
		INFO("Freeing counter window.");
		delwin(counter_window);
	}
}
