/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <nd_tcp.h>
#include <nd_tcp_callbacks.h>
#include <nd_dialog.h>

void    
nd_tcp_sport_cb(LND_Packet  *packet,
		guchar      *header,
		guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  cb_data = nd_proto_cb_data_new(nd_tcp_get(), data, 0, FALSE);

  nd_dialog_number(_("Enter source port:"),
		   ND_BASE_DEC,
		   ntohs(tcphdr->th_sport), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, cb_data);
}


void    
nd_tcp_dport_cb(LND_Packet  *packet,
		guchar      *header,
		guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  cb_data = nd_proto_cb_data_new(nd_tcp_get(), data, data-header, TRUE);

  nd_dialog_number(_("Enter destination port:"),
		   ND_BASE_DEC,
		   ntohs(tcphdr->th_dport), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, cb_data);
}


static void
tcp_seq_ok_cb(LND_Packet *packet, void *user_data, guint value)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  struct tcphdr        *tcphdr;
  
  if (! (trace = libnd_packet_get_trace(packet)))
    return;
  
  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) libnd_packet_get_data(libnd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
      
      tcphdr->th_seq = htonl(value);
      libnd_tcp_update_state(libnd_pit_get(&pit), -1);
      libnd_packet_modified(libnd_pit_get(&pit));
    }
  
  return;
  TOUCH(user_data);
}


void    
nd_tcp_seq_cb(LND_Packet  *packet,
	      guchar      *header,
	      guchar      *data)
{
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;

  nd_dialog_number(_("Enter sequence number:"),
		   ND_BASE_DEC,
		   ntohl(tcphdr->th_seq), (guint) -1,
		   tcp_seq_ok_cb,
		   NULL, packet, data);

  return;
  TOUCH(data);
}


static void
tcp_ack_ok_cb(LND_Packet *packet, void *user_data, guint value)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  struct tcphdr        *tcphdr;
  
  if (! (trace = libnd_packet_get_trace(packet)))
    return;
  
  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) libnd_packet_get_data(libnd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
      
      tcphdr->th_ack = htonl(value);
      libnd_tcp_update_state(libnd_pit_get(&pit), -1);
      libnd_packet_modified(libnd_pit_get(&pit));
    }
  
  return;
  TOUCH(user_data);
}


void    
nd_tcp_ack_cb(LND_Packet  *packet,
	      guchar      *header,
	      guchar      *data)
{
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;

  nd_dialog_number(_("Enter acknowledgement number:"),
		   ND_BASE_DEC,
		   ntohl(tcphdr->th_ack), (guint) -1,
		   tcp_ack_ok_cb,
		   NULL, packet, data);

  return;
  TOUCH(data);
}


void    
nd_tcp_off_cb(LND_Packet  *packet,
	      guchar      *header,
	      guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  cb_data = nd_proto_cb_data_new(nd_tcp_get(), data, data-header, FALSE);

  nd_dialog_number(_("Enter header size in 32-bit words:"),
		   ND_BASE_DEC,
		   tcphdr->th_off, 15,
		   nd_proto_iterate_4bit_hi_cb,
		   NULL, packet, cb_data);
}


static void
tcp_x2_ok_cb(LND_Packet *packet, void *user_data, guint value)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  struct tcphdr        *tcphdr;
  
  if (! (trace = libnd_packet_get_trace(packet)))
    return;
  
  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) libnd_packet_get_data(libnd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
      
      tcphdr->th_flags &= 0x3f;
      tcphdr->th_flags |= ((value & 0x03) << 6);
      tcphdr->th_x2 = ((value & 0x3c) >> 2);

      libnd_packet_modified(libnd_pit_get(&pit));
    }
  
  return;
  TOUCH(user_data);
}


void    
nd_tcp_unused_cb(LND_Packet  *packet,
		 guchar      *header,
		 guchar      *data)
{
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  nd_dialog_number(_("Enter 6-bit unused value:"),
		   ND_BASE_DEC,
		   ((tcphdr->th_x2 << 2) | ((tcphdr->th_flags & 0xc0) >> 6)),
		   63,
		   tcp_x2_ok_cb,
		   NULL, packet, data);
}


static void    
tcp_flag_impl_cb(LND_Packet  *packet,
		 int          flag)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  struct tcphdr        *tcphdr;
  gboolean              set;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;
  
  tcphdr = (struct tcphdr *) libnd_packet_get_data(packet, nd_tcp_get(), 0);
  if (!tcphdr)
    return;

  set = ! (tcphdr->th_flags & flag);

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) libnd_packet_get_data(libnd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
      
      if (set)
	tcphdr->th_flags |= flag;
      else
	tcphdr->th_flags &= ~flag;

      libnd_packet_modified(libnd_pit_get(&pit));
    }
}

void    
nd_tcp_u_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  tcp_flag_impl_cb(packet, TH_URG);

  return;
  TOUCH(header);
  TOUCH(data);
}

void    
nd_tcp_a_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  tcp_flag_impl_cb(packet, TH_ACK);

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_p_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  tcp_flag_impl_cb(packet, TH_PUSH);

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_r_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  tcp_flag_impl_cb(packet, TH_RST);

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_s_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  tcp_flag_impl_cb(packet, TH_SYN);

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_f_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  tcp_flag_impl_cb(packet, TH_FIN);

  return;
  TOUCH(header);
  TOUCH(data);
}


void    
nd_tcp_win_cb(LND_Packet  *packet,
	      guchar      *header,
	      guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct tcphdr *tcphdr;

  tcphdr = (struct tcphdr *) header;
  cb_data = nd_proto_cb_data_new(nd_tcp_get(), data, data-header, FALSE);

  nd_dialog_number(_("Enter window size:"),
		   ND_BASE_DEC,
		   ntohs(tcphdr->th_win), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, cb_data);
}


void    
nd_tcp_cksum_cb(LND_Packet  *packet,
		guchar      *header,
		guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct tcphdr *tcphdr;
  
  tcphdr = (struct tcphdr *) header;
  cb_data = nd_proto_cb_data_new(nd_tcp_get(), data, data-header, FALSE);
  
  nd_dialog_number(_("Enter checksum value:"),
		   ND_BASE_HEX,
		   ntohs(tcphdr->th_sum), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, cb_data);
}


void    
nd_tcp_urp_cb(LND_Packet  *packet,
	      guchar      *header,
	      guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct tcphdr *tcphdr;
  
  tcphdr = (struct tcphdr *) header;
  cb_data = nd_proto_cb_data_new(nd_tcp_get(), data, data-header, FALSE);

  nd_dialog_number(_("Enter urgent pointer value:"),
		   ND_BASE_DEC,
		   ntohs(tcphdr->th_urp), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL, packet, cb_data);
}


void    
nd_tcp_cksum_fix_cb(LND_Packet  *packet,
		    guchar      *header,
		    int          value)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  struct tcphdr        *tcphdr;
  guint16               correct_sum;
  
  if (! (trace = libnd_packet_get_trace(packet)))
    return;
  
  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      tcphdr = (struct tcphdr *) libnd_packet_get_data(libnd_pit_get(&pit), nd_tcp_get(), 0);
      if (!tcphdr)
	continue;
      
      if (!libnd_tcp_csum_correct(libnd_pit_get(&pit), &correct_sum))
	{
	  tcphdr->th_sum = correct_sum;
	  libnd_packet_modified(libnd_pit_get(&pit));
	}
    }
  
  return;
  TOUCH(header);
  TOUCH(value);
}
