/*
 * Copyright (c) 1998 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

/**********	main.c	***********/

#include <sys/time.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>
#include <syslog.h>
#include <math.h>
#include <pwd.h>
#include <unistd.h>
#include <errno.h>
#include <netdb.h>
#include <fcntl.h>
#include <dirent.h>
#include <signal.h>

#include "nefu.h"

char			*bin_name;
char			*nefu_html_dir;
char			*nefu_script_dir;

/* these are here because yacc documentation blows */
extern int yyparse( void );
extern int yy_infile( char * );

struct log_facility log_list[] = {
    { "USER",		LOG_USER },
    { "DAEMON",		LOG_DAEMON },
    { "LOCAL0",		LOG_LOCAL0 },
    { "LOCAL1",		LOG_LOCAL1 },
    { "LOCAL2",		LOG_LOCAL2 },
    { "LOCAL3",		LOG_LOCAL3 },
    { "LOCAL4",		LOG_LOCAL4 },
    { "LOCAL5",		LOG_LOCAL5 },
    { "LOCAL6",		LOG_LOCAL6 },
    { "LOCAL7",		LOG_LOCAL7 },
    { NULL,		0 },
};


    /* make an empty directory */

    int
directory( char *dir )
{
    DIR			*dirp;
    struct dirent	*entry;
    char		oldfile[ MAXPATHLEN ];

    if ( mkdir( dir, S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH ) != 0 ) {
	/* if it's already there, delete everything inside of it */
	if ( errno != EEXIST ) {
	    fprintf( stderr, "mkdir: %s: ", dir );
	    perror( NULL );
	    return( 1 );

	} else {
	    /* blow away the old deptree */
	    if (( dirp = opendir( dir )) == NULL ) {
		fprintf( stderr, "opendir: %s: ", dir );
		perror( NULL );
		return( 1 );
	    }

	    /* clear errno before trying to read */
	    errno = 0;

	    /* blow everything away */
	    while (( entry = readdir( dirp )) != NULL ) {

		/* '.' and '..' are safe */
		if ( entry->d_name[ 0 ] == '.' ) {
		    if ( entry->d_name[ 1 ] == '\0' ) {
			continue;
		    } else if ( entry->d_name[ 1 ] == '.' ) {
			if ( entry->d_name[ 2 ] == '\0' ) {
			    continue;
			}
		    }
		}

		sprintf( oldfile, "%s/%s", dir, entry->d_name );
		if ( unlink( oldfile ) != 0 ) {
		    fprintf( stderr, "unlink: %s: ", oldfile );
		    perror( NULL );
		    return( 1 );
		}
	    }

	    /* did readdir finish, or encounter an error? */
	    if ( errno != 0 ) {
		perror( "readdir" );
		return( 1 );
	    }
	}
    }

    return( 0 );
}


    int
main( int argc, char *argv[] )
{
    extern char		*optarg;
    extern int		optind;
    int			err = 0;
    int			c;
    int			quit_stdout = 0;
    int			quit_html = 0;
    int			no_html = 0;
    int			i;
    int			dt;
    int			time;
    int			nefu_log_facility;
    struct protoent	*proto;
    struct passwd	*nefu_pw;
    struct log_facility	*log;
    struct schedule	s;
    char		localhostname[ MAXHOSTNAMELEN + 1 ];
    char		*config_fname = NULL;

    nefu_version = VERSION;
    nefu_sendmail = NEFU_SENDMAIL;
    nefu_html_dir = NEFU_HTML_DIR;
    nefu_script_dir = NEFU_SCRIPT_DIR;

    memset( &s, 0, sizeof( struct schedule ));
    s.s_tv_pass_minimum.tv_sec = DEFAULT_PASS_TIME;
    s.s_tv_service_minimum.tv_sec = DEFAULT_TEST_TIME;

    rcode_stab = NULL;

    machine_stab = NULL;

    nefu_redundant_paging = 0;

    machines_no_dns = NULL;
    dns_pass = DEFAULT_DNS_PASSES;

    root_nodes = NULL;

    bin_name = argv[ 0 ];

    nefu_log_facility = LOG_DAEMON;

    if (( *bin_name == '.' ) && ( *(bin_name+1) == '/' )) {
	bin_name += 2;
    }

    nefu_cmask = 022;
    nefu_html_reload = -1;

    /* don't log html history pages by default */
    nefu_html_history = 0;

    nefu_uname = DEFAULT_UNAME;
    nefu_from_addr = nefu_uname;

    nefu_localdomain = NULL;

    while (( c =
	    getopt( argc, argv, "d:D:f:F:hHl:m:M:Nnp:r:RSs:u:VW:" )) != EOF ) {
	switch ( c ) {

	case 'd':
	    dns_pass = atoi( optarg );
	    break;

	case 'D':
	    if (( nefu_localdomain = optarg ) == NULL ) {
		err++;
	    }
	    break;

	case 'f':
	    if ( strcmp( optarg, "-") != 0 ) {
		config_fname = optarg;
	    }
	    break;

	case 'F':
	    if (( nefu_from_addr = optarg ) == NULL ) {
		err++;
	    }

	case 'h':
	    no_html = 1;
	    break;

	case 'H':
	    nefu_html_history = 1;
	    break;

	case 'l':
	    for ( log = log_list; log->log_name != NULL; log++ ) {
		if ( strcmp( optarg, log->log_name ) == 0 ) {
		    nefu_log_facility = log->log_facility;
		    break;
		}
	    }

	    if ( log->log_name == NULL ) {
		fprintf( stderr, "Invalid log facility: %s\n", optarg );
		err++;
	    }
	    break;

	case 'm':
	    nefu_cmask = atoi( optarg );
	    break;

	case 'M':
	    nefu_master_dns_domain = optarg;
	    break;

	case 'n':
	    quit_stdout = 1;
	    break;

	case 'N':
	    quit_html = 1;
	    break;

	case 'p':
	    /* XXX atoi sucks */
	    if (( time = atoi( optarg )) < 0 ) {
		fprintf( stderr, "Invalid time: %s\n", optarg );
		err++;

	    } else {
		s.s_tv_pass_minimum.tv_sec = time;
	    }
	    break;

	case 'r':
	    /* XXX atoi sucks */
	    if (( time = atoi( optarg )) < 0 ) {
		fprintf( stderr, "Invalid time: %s\n", optarg );
		err++;

	    } else {
		nefu_html_reload = time;
	    }
	    break;

	case 'R':
	    nefu_redundant_paging = 1;
	    break;

	case 'S':
	    nefu_suppress_reports = 1;
	    break;

	case 's':
	    if (( time = atoi( optarg )) < 0 ) {
		fprintf( stderr, "Invalid time: %s\n", optarg );
		err++;

	    } else {
		s.s_tv_service_minimum.tv_sec = time;
	    }
	    break;

	case 'u':
	    if (( nefu_uname = optarg ) == NULL ) {
		err++;
	    }
	    break;

	case 'V':
	    printf( "Version %s\n", nefu_version );
	    exit( 0 );

	case 'W':
	    if (( nefu_test_maxwait = atoi( optarg )) < 0 ) {
		fprintf( stderr, "Invalid wait time: %s\n", optarg );
		err++;
	    }
	    break;

	default:
	    err++;
	    break;
	}
    }

    if (( argc - optind != 0 ) || ( err != 0 )) {
	printf( "Usage:\t%s ", argv[ 0 ] );
	printf( "[ -d passes ] " );
	printf( "[ -D domain ] " );
	printf( "[ -f file ] " );
	printf( "[ -h ] " );
	printf( "[ -H ]\n" );
	printf( "\t[ -l facility ] " );
	printf( "[ -m mask ] " );
	printf( "[ -n ] " );
	printf( "[ -N ] " );
	printf( "[ -p seconds ] " );
	printf( "[ -r seconds ]\n" );
	printf( "\t[ -R ] " );
	printf( "[ -s seconds ] " );
	printf( "[ -u uname ] " );
	printf( "[ -V ]\n" );
	fflush( stdout );
	exit( 1 );
    }

    if ( nefu_localdomain == NULL ) {
	if ( gethostname( localhostname, MAXHOSTNAMELEN ) != 0 ) {
	    perror( "gethostname" );
	    exit( 1 );
	}

	nefu_localdomain = localhostname;
    }

    /* parse the config file */
    if ( nefu_read_config( config_fname ) != 0 ) {
	fprintf( stderr, "Error in config file, nefu not starting.\n" );
	exit( 1 );
    }

    /* prepare the graph */
    if ( order_tree() != 0 ) {
	exit( 1 );
    }

    /* print a dep map instead of monitoring */
    if ( quit_stdout != 0 ) {
	printf( "\nnefu nefu_version %s\n", nefu_version );
	depmap_stdout();
	exit( 0 );
    }

    /* get our user info from /etc/passwd */
    if (( nefu_pw = getpwnam( nefu_uname )) == NULL ) {
	fprintf( stderr, "getpwnam %s: user not found\n", nefu_uname );
	exit( 1 );
    }

    /* error check to make sure we got a directory entry */
    if (( nefu_pw->pw_dir == NULL ) || ( *nefu_pw->pw_dir == '\0' )) {
	fprintf( stderr, "getpwnam %s: directory not found\n", nefu_uname );
	exit( 1 );
    }

    /* save our directory path */
    if (( nefu_user_dir = strdup( nefu_pw->pw_dir )) == NULL ) {
	perror( "strdup" );
	exit( 1 );
    }

    /* cd to our directory */
    if ( chdir( nefu_user_dir ) != 0 ) {
	perror( nefu_user_dir );
	exit( 1 );
    }

    /* set our umask */
    umask( nefu_cmask );

    /* nefu needs a raw socket */
    if (( proto = getprotobyname( "ICMP" )) == NULL ) {
	fprintf( stderr, "getprotobyname: ICMP: not found" );
	exit( 1 );
    }

    if (( nefu_raw_socket = socket( AF_INET, SOCK_RAW, proto->p_proto )) < 0 ) {
	perror( "socket" );
	exit( 1 );
    }

    /* set our gid */
    if ( setgid( nefu_pw->pw_gid ) != 0 ) {
	perror( "setgid" );
	exit( 1 );
    }

    /* set our uid */
    if ( setuid( nefu_pw->pw_uid ) != 0 ) {
	perror( "setuid" );
	exit( 1 );
    }

    /* do we output html? */
    if ( no_html == 0 ) {
	if (( nefu_html_dir == NULL ) || ( *nefu_html_dir == '\0' )) {
	    no_html = 1;

	} else if ( nefu_html_reload < 0 ) {
	    if (( nefu_html_reload = s.s_tv_pass_minimum.tv_sec ) < 1 ) {
		nefu_html_reload = 1;
	    }
	}
    }

    /* initalize html pages */
    html_init( &s );

    /* output the html dependancy map */
    depmap_html();

    /* rcode html database */
    rcode_html_pages();

    /* suicide? */
    if ( quit_html != 0 ) {
	exit( 0 );
    }
    
    /* ignore SIGPIPE */
    signal( SIGPIPE, SIG_IGN );
    /*
    if ( signal( SIGPIPE, SIG_IGN ) != 0 ) {
	perror( "sigignore" );
	exit( 1 );
    }
    */

    /* disassociate from controlling tty */
    switch ( fork()) {
    case 0 :
	if ( setsid() < 0 ) {
	    perror( "setsid" );
	    exit( 1 );
	}
	dt = getdtablesize();

	/* close all sockets except the raw one */
	for ( i = 0; i < dt; i++ ) {
	    if ( i != nefu_raw_socket ) {
		close( i );
	    }
	}

	/* neuter stdin, stdout, stderr as we don't need them as a daemon */
	if (( i = open( "/", O_RDONLY, 0 )) == 0 ) {
	    dup2( i, 1 );
	    dup2( i, 2 );
	}
	break;

    case -1 :
	perror( "fork" );
	exit( 1 );

    default :
	exit( 0 );
    }

    openlog( "nefu", LOG_NDELAY, nefu_log_facility );

    syslog( LOG_DEBUG, "nefu starting %s %d %d\n", nefu_version, nodes_total,
	    nodes_leaf );

    monitor_network( &s );

    return( 1 );
}
