#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <errno.h>
#include <unistd.h>
#include <stdio.h>					/* perror */
#include <signal.h>
#include <strings.h>		
#include <string.h>		
#include <stdlib.h>
#include <syslog.h>

#include "nefu.h"
#include "snet.h"

#define	PARENT_DIRECTORY	"../"

extern char **environ;


    char *
init_shell( struct test *t )
{
    if ( t->t_argc < 1 ) {
	return( "useage: '!' scriptname [ args ]" );
    }

    if ( strstr( t->t_argv[ 0 ], PARENT_DIRECTORY ) != NULL ) {
	return( "illegal shell path: '../'" );
    }

    if (( nefu_script_dir == NULL ) || ( *nefu_script_dir == '\0' )) {
	return( "no script directory defined" );
    }

    return( NULL );
}


    int
test_shell( struct test *t_node, struct report *r )
{
    int			fds[ 2 ];
    int			status;
    pid_t		pid;
    char 		*line;
    char 		prev[ 256 ];
    char		path[ MAXPATHLEN ];
    SNET		*n;
    struct timeval	tv;
    size_t		len;
    static char		*nefu_machine = NULL;
    static char		*nefu_addr;

    if ( pipe( fds ) < 0 ) {
	report_printf( r, "pipe: %m" );
	return( T_MAYBE_DOWN );
    }
    
    pid = fork( );

    switch( pid ) {
    case 0:
	/* good child -  dup2 first for perror */
	if ( dup2( fds[ 1 ], 2 ) < 0 ) {
	    syslog( LOG_ERR, "dup2: %m" );
	    exit( T_MAYBE_DOWN);
	}

	if ( dup2( fds[ 1 ], 1 ) < 0 ) {
	    perror( "dup2" );
	    exit( T_MAYBE_DOWN);
	}

	if (( close( fds[ 0 ] ) < 0 ) || ( close( fds[ 1 ] ) < 0 )) {
	    perror( "close fds" );
	    exit( T_MAYBE_DOWN);
	}

	/* set up environment variables passed to shell */
	len = 14 + strlen( t_node->t_machine->m_name );

	if (( nefu_machine = malloc( len )) == NULL ) {
	    perror( "malloc" );
	    return( T_MAYBE_DOWN );
	}

	sprintf( nefu_machine, "NEFU_MACHINE=%s", t_node->t_machine->m_name );

	if ( putenv( nefu_machine ) != 0 ) {
	    perror( "putenv" );
	    return( T_MAYBE_DOWN );
	}

	len = 11 + strlen( inet_ntoa( t_node->t_sin.sin_addr ));

	if (( nefu_addr = malloc( len )) == NULL ) {
	    perror( "malloc" );
	    return( T_MAYBE_DOWN );
	}

	sprintf( nefu_addr, "NEFU_ADDR=%s",
		inet_ntoa( t_node->t_sin.sin_addr ));

	if ( putenv( nefu_addr ) != 0 ) {
	    perror( "putenv" );
	    return( T_MAYBE_DOWN );
	}

	/* dissallow the Swedish hacker port */
	if ( strstr( t_node->t_argv[ 0 ], PARENT_DIRECTORY ) != NULL ) {
	    report_printf( r, "illegal path: %s", t_node->t_argv[ 0 ]);
	    return( T_MAYBE_DOWN );
	}
	sprintf( path, "%s/%s", nefu_script_dir, t_node->t_argv[ 0 ]);

	/* execve only returns if it fails, stdout gets captured as an error */
	execve( path, t_node->t_argv, environ ); 
	printf( "execve: %s: %s\n", path, strerror( errno ));
	exit( T_MAYBE_DOWN );

    case -1:				/* bad, this is parent */
	close( fds[ 0 ] );
	close( fds[ 1 ] );
	report_printf( r, "fork: %m" );
	return(  T_MAYBE_DOWN );

    default:				/* good, parent */
	if ( close( fds[ 1 ] ) < 0 ) {
	    report_printf( r, "close: %m" );
	    return( T_MAYBE_DOWN );
	}

	if (( n = snet_attach( fds[ 0 ], 1024 * 1024 )) == NULL ) {
	    report_printf( r, "snet_attach: %m" );
	    return( T_MAYBE_DOWN );
	}

	tv.tv_sec = 10;
	tv.tv_usec = 0;

	if (( nefu_test_maxwait >= 0 ) && ( tv.tv_sec > nefu_test_maxwait )) {
	    tv.tv_sec = nefu_test_maxwait;
	}

	*prev = '\0';

	while (( line = snet_getline( n, &tv )) != NULL ) {
	    strncpy( prev, line, sizeof( prev ));
	    prev[ sizeof( prev) - 1 ] = '\0';
	}

	if ( errno == ETIMEDOUT ) {
	    kill( pid, SIGKILL );
	}

	if ( waitpid( pid, &status, 0 ) == -1 ) {
	    report_printf( r, "waitpid: %m" );
	    return( T_MAYBE_DOWN );
	}

	if ( snet_close( n ) != 0 ) {
	    report_printf( r, "snet_close: %m" );
	    return( T_MAYBE_DOWN );
	}

	if ( WIFEXITED( status )) {
	    switch( WEXITSTATUS( status )) {
	    case T_UP:
		return( T_UP );
	    case( T_DOWN ):
		/* make a report */
		report_printf( r, "%s",
			( *prev == '\0' ) ? "no report" : prev );
		return( T_DOWN );
	    case( T_MAYBE_DOWN ):
	    default:
		/* make a report */
		report_printf( r, "%s",
			( *prev == '\0' ) ? "no report" : prev );
		return( T_MAYBE_DOWN );
	    }
	} else if ( WIFSIGNALED( status )) {
	    report_printf( r, "killed with %d", WTERMSIG( status ));
	    return( T_MAYBE_DOWN );
	} else {
	    report_printf( r, "died");
	    return( T_MAYBE_DOWN );
	}
    }
}
