#include <cctype>
#include <cstdlib>
#include <sstream>

#include "nds_parameter_block.hh"

#include "nds_param_str.h"

namespace NDS
{
  parameter_block::parameter_block(): \
    _max_nds1_command_count( 1000 ),
  	_gap_handler_string( "ABORT_HANDLER" ),
    _gap_handler( new abort_gap_handler() ),
    _allow_data_on_tape(false),
    _iterate_uses_gap_handler(false) {}

  void parameter_block::
  initialize_from_env()
  {
  	std::vector<std::string> possible_keys = parameters();
  	const std::string prefix (env_prefix());

  	for (std::vector<std::string>::iterator cur = possible_keys.begin();
  		cur != possible_keys.end();
  		++cur)
  	{
  	  std::string key = prefix + *cur;
  	  char *tmp = ::getenv(key.c_str());
  	  if (tmp)
  	  {
  	    set(*cur, tmp);
  	  }
  	}
  }

  bool parameter_block::
  set_gap_handler(const std::string& handler_str)
  {
  	  if ( handler_str.compare("ABORT_HANDLER") == 0 )
      {
        _gap_handler.reset( new abort_gap_handler() );
      }
      else if ( handler_str.compare("STATIC_HANDLER_ZERO") == 0 )
      {
        _gap_handler.reset( new fixed_point_gap_handler(fixed_point_gap_handler::static_val::ZERO_VAL) );
      }
      else if ( handler_str.compare("STATIC_HANDLER_ONE") == 0 )
      {
        _gap_handler.reset( new fixed_point_gap_handler(fixed_point_gap_handler::static_val::ONE_VAL) );
      }
      else if ( handler_str.compare("STATIC_HANDLER_NAN") == 0 )
      {
        _gap_handler.reset( new fixed_point_gap_handler(fixed_point_gap_handler::static_val::NAN_VAL) ); 
      }
      else if ( handler_str.compare("STATIC_HANDLER_POS_INF") == 0 )
      {
        _gap_handler.reset( new fixed_point_gap_handler(fixed_point_gap_handler::static_val::POS_INF_VAL) ); 
      }
      else if ( handler_str.compare("STATIC_HANDLER_NEG_INF") == 0 )
      {
        _gap_handler.reset( new fixed_point_gap_handler(fixed_point_gap_handler::static_val::NEG_INF_VAL) ); 
      }
      else
      {
      	return false;
      }
      _gap_handler_string = handler_str;
      return true;
  }

  bool parameter_block::
  extract_bool_value(const std::string &str, bool &dest)
  {
  	std::string lower_str(str);

  	{
  		std::string::iterator dest = lower_str.begin();
  		for (std::string::const_iterator cur=str.begin();
	  		cur != str.end();
	  		++cur, ++dest)
	  	{
	  		*dest = std::tolower(*cur);
	  	}
  	}

  	if (lower_str.compare("true") == 0 || lower_str.compare("1") == 0)
  	{
  		dest = true;
  		return true;
	} else if (lower_str.compare("false") == 0 || lower_str.compare("0") == 0)
  	{
  		dest = false;
  		return true;
  	}
  	return false;
  }

  std::string parameter_block::
  get(const std::string &parameter) const 
  {
    std::string result;
    if (parameter.compare(NDS_PARAM_ALLOW_DATA_ON_TAPE) == 0)
    {
    	result = (allow_data_on_tape() ? "true" : "false");
    }
    else if (parameter.compare(NDS_PARAM_GAP_HANDLER) == 0)
    {
    	result = _gap_handler_string;
    }
    else if (parameter.compare(NDS_PARAM_ITERATE_GAPS) == 0)
    {
      result = (iterate_uses_gap_handler() ? "true" : "false");
    }
    else if (parameter.compare(NDS_CYCLE_NDS1_AFTER_N_COMMANDS) == 0)
    {
        std::ostringstream os;
        os << _max_nds1_command_count;
        result = os.str();
    }
    return result;
  }

  bool parameter_block::
  set(const std::string &parameter, const std::string &value)
  {
  	if (parameter.compare(NDS_PARAM_ALLOW_DATA_ON_TAPE) == 0)
  	{
  		bool bool_val = true;
  		if (extract_bool_value(value, bool_val))
  		{
  			_allow_data_on_tape = bool_val;
  			return true;
  		}
  	}
  	else if (parameter.compare(NDS_PARAM_GAP_HANDLER) == 0)
  	{
  		return set_gap_handler(value);
  	}
    else if (parameter.compare(NDS_PARAM_ITERATE_GAPS) == 0)
    {
      bool bool_val = true;
      if (extract_bool_value(value, bool_val))
      {
        _iterate_uses_gap_handler = bool_val;
        return true;
      }
    }
    else if (parameter.compare(NDS_CYCLE_NDS1_AFTER_N_COMMANDS) == 0)
    {
        try {
            std::istringstream is(value);
            is >> _max_nds1_command_count;
        }
        catch (...) { return false; }
        return true;
    }
  	return false;
  }

  std::vector<std::string> parameter_block::
  parameters() const
  {
  	std::vector<std::string> results;
  	results.push_back(NDS_PARAM_ALLOW_DATA_ON_TAPE);
  	results.push_back(NDS_PARAM_GAP_HANDLER);
    results.push_back(NDS_PARAM_ITERATE_GAPS);
    results.push_back(NDS_CYCLE_NDS1_AFTER_N_COMMANDS);
  	return results;
  }

  std::string parameter_block::
  env_prefix() const
  {
  	return std::string(NDS_PARAM_ENV_PREFIX);
  }

}
