#include <fcntl.h>
#include <sqlite3.h>
#ifdef _WIN32
#include <Shlobj.h>
#else
#include <unistd.h>
#endif

#include "nds_os.h"

#include "nds_errno.hh"

#include <sys/stat.h>
#include <sys/types.h>

#include <cstring>
#include <cstddef>
#include <cstdlib>

#include <iostream>
#include <memory>
#include <sstream>

extern "C" {
  /* } */
#include "bash_pattern.h"
  /* { */
}

#include "nds_connection.hh"
#include "nds_connection_ptype.hh"
#include "nds_db_sqlite3.hh"

#include "debug_stream.hh"

/// \todo Need to reduce the times the create_tables gets executed

namespace
{
  static const char *create_tables =
    "CREATE TABLE IF NOT EXISTS servers ("
    "	protocol INTEGER NOT NULL DEFAULT 3,"
    "	timestamp INTEGER NOT NULL DEFAULT 0,"
    "	hash BLOB NOT NULL DEFAULT \'????????\'"
    ");"
    "CREATE TABLE IF NOT EXISTS channels ("
    "	name TEXT NOT NULL,"
    "	channel_type INTEGER NOT NULL,"
    "	data_type INTEGER NOT NULL,"
    "	sample_rate DOUBLE NOT NULL,"
    "	signal_gain FLOAT NOT NULL,"
    "	signal_slope FLOAT NOT NULL,"
    "	signal_offset FLOAT NOT NULL,"
    "	signal_units TEXT NOT NULL"
    ");"
    "CREATE INDEX IF NOT EXISTS channel_name_index ON channels (name);"
    ;

  /* typedef std::vector< daq_channel_t > daq_channels_type; */

  class transaction
  {
  public:
    typedef NDS::ChannelCache::sqlite3db::handle_type handle_type;
    typedef enum {
      ROLLBACK,
      COMMIT
    } exit_type;

    transaction( handle_type Handle );

    ~transaction( );

    void commit( );

    int exec( const char* Statement );

  private:
    handle_type	db;
    exit_type	exit_action;
    bool	bad;
  };

  transaction::
  transaction( handle_type Handle )
    : db( Handle ),
      exit_action( ROLLBACK ),
      bad( false )
  {
    try
    {
      exec( "BEGIN" );
    }
    catch( ... )
    {
      bad = true;
      throw;
    }
  }

  transaction::
  ~transaction( )
  {
    switch ( exit_action )
    {
    case ROLLBACK:
      exec( "ROLLBACK" );
      break;
    case COMMIT:
      exec( "COMMIT" );
      break;
    }
  }

  void transaction::
  commit( )
  {
    exit_action = COMMIT;
  }

  int transaction::
  exec( const char* Statement )
  {
  	int retval = SQLITE_OK;
    if ( ( retval = sqlite3_exec( db, Statement, NULL, NULL, NULL) ) != SQLITE_OK )
    {
      // throw 
    }
    return retval;
  }
  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  typedef struct {
    char *expr;
    bash_pattern *pattern;
  } _bash_pattern_auxdata;

  void _bash_pattern_free_auxdata(void *arg);
  void _bash_pattern_matches_func( sqlite3_context *ctx, int nargs, sqlite3_value **args );
  int _bash_pattern_matches_register_func( sqlite3 *db );
}

namespace NDS
{
  namespace ChannelCache
  { 
    class daq_channels_type
      : public std::vector< daq_channel_t >
    {
    };

    const sqlite3db::handle_type
    sqlite3db::NULL_HANDLE =
      (sqlite3db::handle_type)NULL;

    //-------------------------------------------------------------------
    /// Establish the interval to revalidate the channel hash.
    /// This value should be close to the frequency for which the
    /// list of channels is updated.
    /// Currently it is set for 24 hours (24 (hr) * 60(min) * 60(sec) )
    //-------------------------------------------------------------------
    const sqlite3db::timeout_type sqlite3db::HASH_CHECK_INTERVAL =
      (24 * 60 * 60);

    sqlite3db::
    sqlite3db( const connection& Server )
      : server( Server ),
	cache( NULL_HANDLE ),
	readonly( false )
    {
    }

    sqlite3db::
    ~sqlite3db( )
    {
      if ( cache != NULL_HANDLE )
      {
	sqlite3_close( cache );
	cache = NULL_HANDLE;
      }
    }

    void sqlite3db::
    channels( daq_channels_type& Channels /*,
	      daq_channel_t *channels,
	      size_t count_channels,
	      const time_t *t,
	      const void *hash,
	      int hash_size */ )
    {
      int		rc;
      sqlite3_stmt*	stmt;
      transaction	trans( cache );

      /* Wipe cache */
      trans.exec( "DELETE FROM servers; DELETE FROM channels;" );

      /* Write server metadata (protocol and timestamp) */
      rc = sqlite3_prepare_v2( cache,
			       "INSERT INTO servers (protocol, timestamp, hash) VALUES (?, ?, ?)",
			       -1,
			       &stmt,
			       NULL)
	|| sqlite3_bind_int( stmt, 1, server.get_protocol( ) )
	|| sqlite3_bind_int64( stmt, 2, Timestamp( ) )
	|| sqlite3_bind_blob( stmt,
			      3,
			      server.hash( ).ref( ),
			      static_cast<int>( server.hash( ).size( ) ),
			      SQLITE_STATIC );
      if ( rc == SQLITE_OK )
      {
	rc = sqlite3_step(stmt);
      }
      sqlite3_finalize(stmt);
      if ( rc != SQLITE_DONE )
      {
	return;
      }

      /* Write channel list */
      rc = sqlite3_prepare_v2( cache,
			       "INSERT INTO channels VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
			       -1,
			       &stmt,
			       NULL );
      if ( rc != SQLITE_OK )
      {
	sqlite3_finalize(stmt);
	return;
      }
      for ( daq_channels_type::const_iterator
	      cur = Channels.begin( ),
	      last = Channels.end( );
	    cur != last;
	    ++cur )
      {
	channel::channel_type channel_type = channel::convert_daq_chantype( cur->type );
	channel::data_type data_type = channel::convert_daq_datatype( cur->data_type );
	std::ostringstream	channel_name( cur->name );

	switch ( channel_type )
	{
	case channel::CHANNEL_TYPE_RDS:
	  channel_name << ",rds";
	  break;
	case channel::CHANNEL_TYPE_STREND:
	  channel_name << ",s-trend";
	  break;
	case channel::CHANNEL_TYPE_MTREND:
	  channel_name << ",m-trend";
	  break;
	default:
	  break;
	}

	rc = sqlite3_bind_text(stmt,
			       1,
			       channel_name.str( ).c_str( ),
			       -1, SQLITE_TRANSIENT)
	  || sqlite3_bind_int(stmt, 2, (int) channel_type)
	  || sqlite3_bind_int(stmt, 3, (int) data_type)
	  || sqlite3_bind_double(stmt, 4, cur->rate )
	  || sqlite3_bind_double(stmt, 5, cur->s.signal_gain )
	  || sqlite3_bind_double(stmt, 6, cur->s.signal_slope )
	  || sqlite3_bind_double(stmt, 7, cur->s.signal_offset )
	  || sqlite3_bind_text(stmt, 8, cur->s.signal_units, -1, SQLITE_TRANSIENT);
	if ( rc == SQLITE_OK )
	{
	  rc = sqlite3_step(stmt);
	}
	if ( rc != SQLITE_DONE )
	{
	  sqlite3_finalize(stmt);
	  goto rollback;
	}
	rc = sqlite3_reset(stmt);
	if ( rc != SQLITE_OK )
	{
	  sqlite3_finalize(stmt);
	  goto rollback;
	}
      }
      sqlite3_finalize(stmt);

      //-----------------------------------------------------------------
      // Everything has completed as expected so commit the transaction
      //-----------------------------------------------------------------
      trans.commit( );
    rollback:
      //-----------------------------------------------------------------
      // Time to return back to the caller
      // Got here either by:
      //   1) rollback - not everything succeeded
      //   2) finished the transaction - everything completed correctly
      //-----------------------------------------------------------------
      return;
    }

    void sqlite3db::
    update( )
    {
      if ( readonly )
      {
	//---------------------------------------------------------------
	// Don't waste our time trying to update a readonly database
	//---------------------------------------------------------------
	return;
      }
      int 				rc;

      /* Fail if there is another transfer already in progress. */
      if ( server.request_in_progress( ) )
      {
	throw connection::transfer_busy_error( );
      }

      /* If hash database is not up to date, fetch channel list. */
      if ( update_hash( ) )
      {
	daq_accessor		daqa( server );
	daq_channels_type	daq_channels;
	int			nchannels;
	/* Error reading from cache database or cache database out of date, so
	 * retrieve channel list from server. */
	rc = daq_recv_channels( daqa(), NULL, 0, &nchannels );

	if (rc)
	{
	  errno = NDS::ERROR_LAST + rc;
	  throw NDS::connection::daq_error( rc );
	}

	daq_channels.resize( nchannels );

	rc = daq_recv_channels( daqa( ),
				&(daq_channels[0]),
				nchannels,
				&nchannels );
	if ( rc )
	{
	  errno = NDS::ERROR_LAST + rc;
	  throw NDS::connection::daq_error( rc );
	}

	try
	{
	  channels( daq_channels );
	}
	catch( ... )
	{
	  errno = EIO;
	  throw;
	}
      }
    }

    bool sqlite3db::
    update_hash( )
    {
      bool		retval = true;
      sqlite3_stmt*	stmt = NULL;
      time_t		now;

      time( &now );
      timestamp( now );

      if ( ( sqlite3_prepare_v2( cache,
				 "SELECT timestamp, hash FROM servers",
				 -1,
				 &stmt,
				 NULL) == SQLITE_OK )
	   && ( sqlite3_step( stmt ) == SQLITE_ROW ) )
      {
	time_t timestamp = sqlite3_column_int64( stmt, 0 );
	const void* hash = sqlite3_column_blob( stmt, 1 );
	int hash_size = sqlite3_column_bytes( stmt, 1 );

	if ( ( timestamp + HASH_CHECK_INTERVAL ) >= now )
	{
	   /* Cache has not yet expired. */
	  retval = false;
	}
	else
	{
	  const channel::hash_type& shash = server.hash( );
	  if ( shash.size( )
	       && ( shash.size( ) == hash_size )
	       && ( shash.compare( hash ) ) )
	  {
	    /* Hashes match. No update needed. */
	    retval = false;
	    sqlite3_finalize(stmt);
	    /* Record new timestamp; we can wait another INTERVAL. */
	    /* FIXME: this needs to be moved out */
	    sqlite3_prepare_v2( cache,
				"UPDATE servers SET timestamp = ?",
				-1,
				&stmt,
				NULL )
	      || sqlite3_bind_int64( stmt, 1, now )
	      || sqlite3_step( stmt );
	  }
	}
	sqlite3_finalize( stmt );
      }

      return retval;
    }

    std::string sqlite3db::
    create_sql_query(query_type type,
                                 std::string channel_glob,
                                 channel::channel_type channel_type_mask,
                                 channel::data_type data_type_mask,
                                 channel::sample_rate_type min_sample_rate,
                                 channel::sample_rate_type max_sample_rate,
                                 bool use_glob)
    {
        static const char channel_cond[] =
      "_channel_type & ?2 AND _data_type & ?3 AND _sample_rate BETWEEN ?4 AND ?5";

        const char*		name_cond;
        int 			pattern_is_flat = true;
        std::ostringstream	sql;

        if ( channel_glob.empty( ) )
        {
      channel_glob = "*";
        }

        /* Determine if the channel name pattern is flat (has only literals and
         * wildcards, but no alternatives. */
        {
	  bash_pattern *pattern = bash_pattern_compile( channel_glob.c_str( ) );
	  if ( !pattern )
	  {
	    throw std::runtime_error("could not compile bash pattern");
	  }
	  pattern_is_flat = bash_pattern_is_flat( pattern );
	  bash_pattern_free( pattern );
        }

        /* If the channel name pattern is flat, then compare channel names using
         * GLOB instead of bash_pattern_matches. GLOB is faster than
         * bash_pattern_matches because it can use SQL column indices. */


        if ( !use_glob )
        {
      name_cond = "_name = ?1";
        }
        else if (pattern_is_flat)
        {
      name_cond = "_name GLOB ?1";
        }
        else
        {
      name_cond = "bash_pattern_matches(?1, _name)";
        }

        if ( server.get_protocol( ) == connection::PROTOCOL_ONE )
        {
      switch (type) {
      case FIND_QUERY: sql << "SELECT * FROM"; break;
      case COUNT_QUERY:sql << "SELECT COUNT(*) FROM"; break;
      default: throw std::runtime_error("Unknown query type");
      }
      sql <<
        " ("
        " SELECT"
        "   name AS _name,"
        "   channel_type AS _channel_type,"
        "   data_type AS _data_type,"
        "   sample_rate AS _sample_rate,"
        "   signal_gain,"
        "   signal_slope,"
        "   signal_offset,"
        "   signal_units"
        " FROM channels"
        " WHERE " << name_cond << " AND " << channel_cond
        ;
      if (channel_type_mask & channel::CHANNEL_TYPE_STREND && 1 >= min_sample_rate && 1 <= max_sample_rate)
      {
        if ( data_type_mask & channel::DATA_TYPE_FLOAT64 )
        {
          sql <<
            " UNION ALL"
            " SELECT"
            "   name || '.mean,s-trend' AS _name,"
            "   8 AS _channel_type,"
            "   16 AS _data_type,"
            "   1 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset,"
            "   signal_units"
            " FROM channels"
            " WHERE " << name_cond <<
            " UNION ALL"
            " SELECT"
            "   name || '.rms,s-trend' AS _name,"
            "   8 AS _channel_type,"
            "   16 AS _data_type,"
            "   1 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset,"
            "   signal_units"
            " FROM channels"
            " WHERE " << name_cond
            ;
        }
        if (data_type_mask & (channel::DATA_TYPE_INT32 | channel::DATA_TYPE_FLOAT32))
        {
          sql <<
            " UNION ALL "
            " SELECT"
            "   name || '.min,s-trend' AS _name,"
            "   8 AS _channel_type,"
            "   (CASE WHEN data_type == 32 THEN 8 ELSE data_type END) AS _data_type,"
            "   1 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset,"
            "   signal_units"
            " FROM channels"
            " WHERE " << name_cond << " AND data_type & ?3"
            " UNION ALL"
            " SELECT"
            "   name || '.max,s-trend' AS _name,"
            "   8 AS _channel_type,"
            "   (CASE WHEN data_type == 32 THEN 8 ELSE data_type END) AS _data_type,"
            "   1 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset,"
            "   signal_units"
            " FROM channels"
            " WHERE " << name_cond << " AND data_type & ?3"
            ;
        }
        if (data_type_mask & channel::DATA_TYPE_INT32)
        {
          sql <<
            " UNION ALL"
            " SELECT"
            "   name || '.n,s-trend' AS _name,"
            "   8 AS _channel_type,"
            "   2 AS _data_type, "
            "   1 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset,"
            "   signal_units"
            " FROM channels"
            " WHERE " << name_cond << " AND data_type & ?3"
            ;
        }
      }
      if (channel_type_mask & channel::CHANNEL_TYPE_MTREND && 1.0/60 >= min_sample_rate && 1.0/60 <= max_sample_rate)
      {
        if (data_type_mask & channel::DATA_TYPE_FLOAT64)
        {
          sql <<
            " UNION ALL "
            " SELECT"
            "   name || '.mean,m-trend' AS _name,"
            "   16 AS _channel_type,"
            "   16 AS _data_type,"
            "   1.0/60 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset, signal_units"
            " FROM channels"
            " WHERE " << name_cond <<
            " UNION ALL "
            " SELECT"
            "   name || '.rms,m-trend' AS _name,"
            "   16 AS _channel_type,"
            "   16 AS _data_type,"
            "   1.0/60 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset,"
            "   signal_units"
            " FROM channels"
            " WHERE " << name_cond
            ;
        }
        if (data_type_mask & (channel::DATA_TYPE_INT32 | channel::DATA_TYPE_FLOAT32))
        {
          sql <<
            " UNION ALL "
            " SELECT"
            "   name || '.min,m-trend' AS _name,"
            "   16 AS _channel_type,"
            "   (CASE WHEN data_type == 32 THEN 8 ELSE data_type END) AS _data_type,"
            "   1.0/60 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset,"
            "   signal_units"
            " FROM channels"
            " WHERE " << name_cond << " AND data_type & ?3"
            " UNION ALL "
            " SELECT"
            "   name || '.max,m-trend' AS _name,"
            "   16 AS _channel_type,"
            "   (CASE WHEN data_type == 32 THEN 8 ELSE data_type END) AS _data_type,"
            "   1.0/60 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset,"
            "   signal_units"
            " FROM channels WHERE " << name_cond << " AND data_type & ?3"
            ;
        }
        if (data_type_mask & channel::DATA_TYPE_INT32)
        {
          sql <<
            " UNION ALL "
            " SELECT"
            "   name || '.n,m-trend' AS _name,"
            "   16 AS _channel_type,"
            "   2 AS _data_type,"
            "   1.0/60 AS _sample_rate,"
            "   signal_gain,"
            "   signal_slope,"
            "   signal_offset,"
            "   signal_units"
            " FROM channels"
            " WHERE " << name_cond << " AND data_type & ?3"
            ;
        }
      }
      sql << ") ORDER BY _name";
        }
        else
        {
      /* (nds2_get_protocol(connection) == NDS2_PROTOCOL_TWO) */
      sql << "SELECT"
          << " name AS _name,"
          << " channel_type AS _channel_type,"
          << " data_type AS _data_type,"
          << " sample_rate AS _sample_rate,"
          << " signal_gain, signal_slope, signal_offset, signal_units FROM channels"
          << " WHERE " << name_cond
          << " AND " << channel_cond
          << " ORDER BY _name"
        ;
        }
        return sql.str();
    }

    size_t sqlite3db::
    count_channels( std::string channel_glob,
		   channel::channel_type channel_type_mask,
		   channel::data_type data_type_mask,
		   channel::sample_rate_type min_sample_rate,
		   channel::sample_rate_type max_sample_rate,
		   bool use_glob )
    {
      size_t	retval;

      int 			rc = 0;
      sqlite3_stmt*		stmt = NULL;

      if ( cache == NULL_HANDLE )
      {
	_db_find( true );
      }

      //-----------------------------------------------------------------
      // Ensure the current cache is up to date
      //-----------------------------------------------------------------
      update( );

      std::string sql;

      try {
        sql = create_sql_query(COUNT_QUERY,
                                         channel_glob,
                                         channel_type_mask,
                                         data_type_mask,
                                         min_sample_rate,
                                         max_sample_rate,
                                         use_glob);
      } catch(...) {
          goto fail;
      }

      //NDS::dout() << "count channels query is:\n" << sql << "\n" << std::endl;

      rc =
	sqlite3_exec( cache, create_tables, NULL, NULL, NULL );

      if ( rc != SQLITE_OK )
      {
	goto fail;
      }

      rc =
	sqlite3_prepare_v2( cache, sql.c_str( ), -1, &stmt, NULL );

      if ( rc != SQLITE_OK )
      {
	goto fail;
      }


      rc =
	sqlite3_bind_text( stmt, 1, channel_glob.c_str( ), -1, SQLITE_STATIC )
	|| sqlite3_bind_int( stmt, 2, (int) channel_type_mask )
	|| sqlite3_bind_int( stmt, 3, (int) data_type_mask )
	|| sqlite3_bind_double( stmt, 4, min_sample_rate )
	|| sqlite3_bind_double( stmt, 5, max_sample_rate );

      if ( rc != SQLITE_OK )
      {
	goto fail;
      }

      if ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
      {
          retval = static_cast<size_t>(sqlite3_column_int(stmt, 0));
      }

    fail:
      sqlite3_finalize(stmt);
      return retval;
    }

      channels_type_intl sqlite3db::
    find_channels( std::string channel_glob,
		   channel::channel_type channel_type_mask,
		   channel::data_type data_type_mask,
		   channel::sample_rate_type min_sample_rate,
		   channel::sample_rate_type max_sample_rate,
		   bool use_glob )
    {
        channels_type_intl	retval;

      int 			rc = 0;
      sqlite3_stmt*		stmt = NULL;

      if ( cache == NULL_HANDLE )
      {
	_db_find( true );
      }

      //-----------------------------------------------------------------
      // Ensure the current cache is up to date
      //-----------------------------------------------------------------
      update( );

      std::string sql;

      try {
        sql = create_sql_query(FIND_QUERY,
                                         channel_glob,
                                         channel_type_mask,
                                         data_type_mask,
                                         min_sample_rate,
                                         max_sample_rate,
                                         use_glob);
      } catch(...) {
          goto fail;
      }

      //NDS::dout() << "find channels query is:\n" << sql << "\n" << std::endl;

      rc = 
	sqlite3_exec( cache, create_tables, NULL, NULL, NULL );

      if ( rc != SQLITE_OK )
      {
	goto fail;
      }

      rc =
	sqlite3_prepare_v2( cache, sql.c_str( ), -1, &stmt, NULL );

      if ( rc != SQLITE_OK )
      {
	goto fail;
      }


      rc =
	sqlite3_bind_text( stmt, 1, channel_glob.c_str( ), -1, SQLITE_STATIC )
	|| sqlite3_bind_int( stmt, 2, (int) channel_type_mask )
	|| sqlite3_bind_int( stmt, 3, (int) data_type_mask )
	|| sqlite3_bind_double( stmt, 4, min_sample_rate )
	|| sqlite3_bind_double( stmt, 5, max_sample_rate );

      if ( rc != SQLITE_OK )
      {
	goto fail;
      }

      while ((rc = sqlite3_step(stmt)) == SQLITE_ROW)
      {
	retval.push_back( channel( (const char*)sqlite3_column_text(stmt, 0),
				   (channel::channel_type)sqlite3_column_int(stmt, 1),
				   (channel::data_type)sqlite3_column_int(stmt, 2),
				   (channel::sample_rate_type)sqlite3_column_double(stmt, 3),
				   (channel::signal_gain_type)sqlite3_column_double(stmt, 4),
				   (channel::signal_slope_type)sqlite3_column_double(stmt, 5),
				   (channel::signal_offset_type)sqlite3_column_double(stmt, 6),
				   (const char*)sqlite3_column_text(stmt, 7) ) );
      }

    fail:
      sqlite3_finalize(stmt);
      return retval;
    }
   
    void sqlite3db::
    _db_find( bool Create )
    {
#if _WIN32
      char base_path[MAX_PATH];

      /* Locate, and create if necessary, the directory
       * "C:\Documents and Settings\username\Local Settings\Application Data". */
      if (!SUCCEEDED(SHGetFolderPathA(NULL, CSIDL_LOCAL_APPDATA | CSIDL_FLAG_CREATE, NULL, 0, base_path)))
	goto done;

      /* Form path to database: e.g.
       * "C:\Documents and Settings\username\Local Settings\Application Data\nds2-client_0.10_nds.ligo-la.caltech.edu_31200.sqlite". */
	  db_dir = base_path;
	  db_dir.append("\\");

	  _db_open( Create );
#else /* _WIN32 */
      /* if CHANNEL_DB_DIR is explicitly specified, look for
       * database there, and fail if we don't find it */
      int			rc;

      char *env_dir = getenv("NDS2_CHANNEL_DB_DIR");

      if (env_dir)
      {
	db_dir = env_dir;
	_db_open( Create );

	/* otherwise look in system paths */
      }
      else
      {
	db_dir = "/var/cache/" PACKAGE_NAME ;

	/* try to open db at system path.  if opened, return.
	 * otherwise, try the user-specific /var/tmp dir.
	 */
	try
	{
	  if (_db_open( Create ))
	  {
	    rc = 0;
	    goto done;
	  }
	}
	catch( ... )
	{
	  // If unable to open the database, then try the next method
	}

	/* look up ID of current user. */
	uid_t uid = getuid();

	/* user-specific db dir areas */
    std::vector<std::string> dirs;
    dirs.push_back("/var/cache/" PACKAGE_NAME);
    dirs.push_back("/var/tmp/" PACKAGE_NAME);
    //dirs.push_back(":memory:");	// failsafe?

    for (std::vector<std::string>::const_iterator cur_dir = dirs.begin();
    	cur_dir != dirs.end();
    	++cur_dir)
    {
    	db_dir = *cur_dir;

		db_dir.append( " " );
		{
		  std::ostringstream num;
		  num << (long unsigned)( uid );
		  db_dir.append( num.str( ) );
		}

		/* make user db dir */
		if ( ( mkdir( db_dir.c_str(), 00755 ) == 0 ) ||
			( errno == EEXIST ))
		{
			if (_db_open( Create ))
			{
				goto done;
			}
		}
    }
    throw std::string("Could not open database");

      }
#endif /* _WIN32 */
    done:
      return;
    }

    bool sqlite3db::
    _db_open( bool Create )
    {
      bool			retval = false;
      std::ostringstream	db_filename;
      int			sqlite_mode;

      /* db path, e.g. */
      /* "DB_DIR/0.10_nds.ligo.caltech.edu_31200.sqlite". */
      db_filename << db_dir << "/" PACKAGE_VERSION
	      << "_" << server.get_host( )
	      << "_" << server.get_port( )
	      << ".sqlite"
	;
      if ( access( db_filename.str( ).c_str( ), F_OK) == 0 )
      {
	if ( access( db_filename.str( ).c_str( ), W_OK) == 0 )
	{
	  sqlite_mode = SQLITE_OPEN_READWRITE | SQLITE_OPEN_SHAREDCACHE;
	  readonly = false;
	} else {
	  sqlite_mode = SQLITE_OPEN_READONLY;
	  readonly = true;
	}
      }
      else
      {
	if ( Create == false )
	{
	  goto done;
	}
	sqlite_mode =
	  SQLITE_OPEN_CREATE
	  | SQLITE_OPEN_READWRITE
	  | SQLITE_OPEN_SHAREDCACHE;
      }

      if ( sqlite3_open_v2( db_filename.str( ).c_str( ),
			    &cache,
			    sqlite_mode,
			    NULL) == SQLITE_OK )
      {
	/* Register bash_pattern_matches as a custom function. */
	if ( _bash_pattern_matches_register_func( cache ) != SQLITE_OK )
	{
	  errno = EIO;
	}
	//---------------------------------------------------------------
	// Make sure the tables will be available
	//---------------------------------------------------------------
	{
	  transaction	t( cache );

	  if ( t.exec( create_tables ) != SQLITE_OK )
	  {
	  	NDS::dout() << "create tables failed" << std::endl;
	  }
	  t.commit( );
	}
		retval = true;
      } 

    done:
      return retval;
    }
  }
}

namespace
{
  void
  _bash_pattern_free_auxdata(void *arg)
  {
    _bash_pattern_auxdata *auxdata = (_bash_pattern_auxdata *) arg;
    if (auxdata)
    {
      free(auxdata->expr);
      auxdata->expr = NULL;
      bash_pattern_free(auxdata->pattern);
      auxdata->pattern = NULL;
      free(auxdata);
    }
  }

  void
  _bash_pattern_matches_func(sqlite3_context *ctx, int nargs, sqlite3_value **args)
  {
    _bash_pattern_auxdata *auxdata;
    const char *expr;
    const char *text;
    int ret = 0;

    /* If there are not exactly two arguments, then fail. */
    if (nargs != 2)
    {
      sqlite3_result_error(ctx, "exptected 2 arguments", -1);
      goto fail;
    }

    /* If the pattern expression is NULL, return false. */
    expr = (const char *)sqlite3_value_text(args[0]);
    if (!expr)
      goto done;

    /* If the text is NULL, return false. */
    text = (const char *)sqlite3_value_text(args[1]);
    if (!text)
      goto done;

    auxdata = (_bash_pattern_auxdata *) sqlite3_get_auxdata(ctx, 0);
    if (!auxdata || strcmp(expr, auxdata->expr))
    {
      auxdata = (_bash_pattern_auxdata *)malloc(sizeof(_bash_pattern_auxdata));
      if (!auxdata)
      {
	sqlite3_result_error_nomem(ctx);
	goto fail;
      }
      auxdata->expr = strdup(expr);
      if (!auxdata->expr)
      {
	free(auxdata);
	sqlite3_result_error_nomem(ctx);
	goto fail;
      }
      auxdata->pattern = bash_pattern_compile(expr);
      if (!auxdata->pattern)
      {
	free(auxdata->expr);
	free(auxdata);
	sqlite3_result_error(ctx, "cannot compile pattern", -1);
	goto fail;
      }
      sqlite3_set_auxdata(ctx, 0, auxdata, _bash_pattern_free_auxdata);
    }

    ret = bash_pattern_matches(auxdata->pattern, text);
  done:
    sqlite3_result_int(ctx, ret);
  fail:
    return;
  }

  int
  _bash_pattern_matches_register_func(sqlite3 *db)
  {
    return sqlite3_create_function( db,
				    "bash_pattern_matches",
				    2,
				    SQLITE_UTF8,
				    NULL,
				    _bash_pattern_matches_func,
				    NULL,
				    NULL );
  }
}
