#include <cerrno>
#include <map>
#include <memory>
#include <sstream>
#include <vector>

#include "debug_stream.hh"

#include "daq_config.h"
#include "daqc.h"
#include "daqc_internal.h"
#include "daqc_response.h"

#include "nds_buffer_internal.hh"

#include "nds_db.hh"
#include "nds_gap_handler.hh"
#include "nds_request_fragment.hh"
#include "nds_iterate_handler.hh"
#include "nds_parameter_block.hh"

#include "nds_channel_internal.hh"

#define SharedPtr std::auto_ptr

namespace NDS
{
  class daq_accessor
  {
  public:
    daq_accessor( const connection& Server );

    daq_t* operator( )( );

  private:
    const connection&   server;
  };

  class buffer_initializer
  {
  public:
    buffer_initializer(buffer::gps_second_type gps_start, 
      buffer::gps_second_type gps_stop): gps_start(gps_start), gps_stop(gps_stop) {};

    void reset_buffer(buffer* cur_buffer, const channel &channel_info) const;
    
  private:
    buffer::gps_second_type gps_start;
    buffer::gps_second_type gps_stop;
  };

namespace helper {

  /**
   * \brief A basic channel filter.
   * \remarks Used by the find_channels and count_channels code paths
   */
  class basic_channel_filter {
  public:
      basic_channel_filter(channel::data_type data_type_mask,
                           channel::sample_rate_type min_sample_rate,
                           channel::sample_rate_type max_sample_rate):
                           data_type_mask_(data_type_mask),
                           min_sample_rate_(min_sample_rate),
                           max_sample_rate_(max_sample_rate)
      {}
      bool operator()(const channel& ch)
      {
        return ( (
                    (( ch.DataType() & data_type_mask_ ) != 0) ||
                       data_type_mask_ == NDS::channel::DEFAULT_DATA_MASK
                 ) &&
                 (ch.SampleRate() <= max_sample_rate_) &&
                 (ch.SampleRate() >= min_sample_rate_) );
      }

  private:
      channel::data_type data_type_mask_;
      channel::sample_rate_type min_sample_rate_;
      channel::sample_rate_type max_sample_rate_;
  };

  /**
   * \brief Provide a functor to push channels onto a vector
   */
  class push_back_channel {
  public:
      push_back_channel(std::vector<channel>& buffer): buffer_(buffer) {}
      void operator()(const channel& ch) { buffer_.push_back(ch); }
  private:
      std::vector<channel>& buffer_;
  };

  /**
   * \brief A functor that counts the number of calls
   */
  class count_channels {
  public:
    count_channels(): count_(0) {}
    void operator()(const channel& ch) { ++count_; }

    size_t count() const { return count_; }
  private:
    size_t count_;
  };

}
  //---------------------------------------------------------------------
  // Connection::p_type
  //---------------------------------------------------------------------
  struct connection::p_type
  {
#if 0
    struct _nds2_connection
    {
      sqlite3 *db;

      /* State variables for making iterative data requests */
      long requested_end_time;
      int request_in_progress;

      /* FIXME: hack for sqlite versions < 3.7.11.  should be
       * removed when we can depend on newer sqlite versions */
      int _db_readonly;
    };
#endif /* 0 */

    typedef SharedPtr< NDS::db >	channel_cache_interal_type;
    typedef std::map<std::string, daq_channel_t> channel_mem_cache_type;
    typedef NDS::db&			channel_cache_type;
    typedef long			time_type;


	typedef nds_socket_type socket_t;

    connection::host_type       host;
    connection*			parent;
    connection::port_type       port;
    connection::protocol_type   protocol;
    daq_t                       handle;
    bool                        connected;
    channel_cache_interal_type	channel_cache_;
    channel_mem_cache_type channel_mem_cache_;

    time_type			request_start_time_;
    time_type			request_end_time_;
    bool			request_in_progress_;

    parameter_block parameters;

    p_type( connection*			Parent,
	    const connection::host_type& Host,
            connection::port_type       Port,
            connection::protocol_type   Protocol );

    ~p_type( );

    void connect( );

    availability_list_type    
    get_availability( buffer::gps_second_type gps_start,
           buffer::gps_second_type gps_stop,
           const channel_names_type& channel_names );

    bool
    is_any_data_available(buffer::gps_second_type gps_start,
    buffer::gps_second_type gps_stop,
    const channel_names_type& channel_names);

    void
    require_any_data_available(buffer::gps_second_type gps_start,
                               buffer::gps_second_type gps_stop,
                               const channel_names_type& channel_names);

    bool check( buffer::gps_second_type gps_start,
      buffer::gps_second_type gps_stop,
      const channel_names_type& channel_names );


    void fetch( buffer::gps_second_type gps_start,
			buffer::gps_second_type gps_stop,
			const channel_names_type& channel_names,
            buffers_type_intl& retval,
            channels_type_intl* reference_channels=NULL );

    size_t
    count_channels_sqlite( std::string channel_glob,
                   channel::channel_type channel_type_mask = channel::DEFAULT_CHANNEL_MASK,
                   channel::data_type data_type_mask = channel::DEFAULT_DATA_MASK,
                   channel::sample_rate_type min_sample_rate = channel::MIN_SAMPLE_RATE,
                   channel::sample_rate_type max_sample_rate = channel::MAX_SAMPLE_RATE );

    size_t
    count_channels_nds2( std::string channel_glob,
                   channel::channel_type channel_type_mask = channel::DEFAULT_CHANNEL_MASK,
                   channel::data_type data_type_mask = channel::DEFAULT_DATA_MASK,
                   channel::sample_rate_type min_sample_rate = channel::MIN_SAMPLE_RATE,
                   channel::sample_rate_type max_sample_rate = channel::MAX_SAMPLE_RATE );

    size_t
    count_channels( std::string channel_glob,
                   channel::channel_type channel_type_mask = channel::DEFAULT_CHANNEL_MASK,
                   channel::data_type data_type_mask = channel::DEFAULT_DATA_MASK,
                   channel::sample_rate_type min_sample_rate = channel::MIN_SAMPLE_RATE,
                   channel::sample_rate_type max_sample_rate = channel::MAX_SAMPLE_RATE )
    {
        if (protocol == PROTOCOL_ONE)
        {
            return count_channels_sqlite( channel_glob,
                                        channel_type_mask,
                                        data_type_mask,
                                        min_sample_rate,
                                        max_sample_rate
                                      );
        }
        return count_channels_nds2( channel_glob,
                                    channel_type_mask,
                                    data_type_mask,
                                    min_sample_rate,
                                    max_sample_rate
                                  );
    }

    void
    find_channels(channels_type_intl& output,
                   std::string channel_glob,
                   channel::channel_type channel_type_mask = channel::DEFAULT_CHANNEL_MASK,
                   channel::data_type data_type_mask = channel::DEFAULT_DATA_MASK,
                   channel::sample_rate_type min_sample_rate = channel::MIN_SAMPLE_RATE,
                   channel::sample_rate_type max_sample_rate = channel::MAX_SAMPLE_RATE
                  )
    {
      if (protocol == PROTOCOL_ONE) 
      {
        return find_channels_sqlite( output,
            channel_glob,
            channel_type_mask,
            data_type_mask,
            min_sample_rate,
            max_sample_rate
          );
      }
      return find_channels_nds2( output,
          channel_glob,
          channel_type_mask,
          data_type_mask,
          min_sample_rate,
          max_sample_rate
        );
    }

    void
    find_channels_sqlite(channels_type_intl &output, std::string channel_glob,
                         channel::channel_type channel_type_mask = channel::DEFAULT_CHANNEL_MASK,
                         channel::data_type data_type_mask = channel::DEFAULT_DATA_MASK,
                         channel::sample_rate_type min_sample_rate = channel::MIN_SAMPLE_RATE,
                         channel::sample_rate_type max_sample_rate = channel::MAX_SAMPLE_RATE);

    void
    find_channels_nds2( channels_type_intl& output, std::string channel_glob,
                   channel::channel_type channel_type_mask = channel::DEFAULT_CHANNEL_MASK,
                   channel::data_type data_type_mask = channel::DEFAULT_DATA_MASK,
                   channel::sample_rate_type min_sample_rate = channel::MIN_SAMPLE_RATE,
                   channel::sample_rate_type max_sample_rate = channel::MAX_SAMPLE_RATE );

    epochs_type
    get_epochs();

    bool
    set_epoch(std::string epoch);

    bool
    set_epoch(buffer::gps_second_type gps_start,
            buffer::gps_second_type gps_stop);

    epoch
    current_epoch() const;

    channel_cache_type channel_cache( );

    const channel::hash_type& hash( ) const;

    void iterate( buffer::gps_second_type gps_start,
		  buffer::gps_second_type gps_stop,
		  buffer::gps_second_type stride,
		  const channel_names_type& channel_names,
          std::vector<NDS::channel>* final_channel_list = 0);

    void iterate_simple_gaps( buffer::gps_second_type gps_start,
                       buffer::gps_second_type gps_stop,
                       buffer::gps_second_type stride,
                       const channel_names_type& channel_names );

    void iterate_fast( buffer::gps_second_type gps_start,
        buffer::gps_second_type gps_stop,
        buffer::gps_second_type stride,
        const channel_names_type& channel_names );

    void iterate_available( buffer::gps_second_type gps_start,
       buffer::gps_second_type gps_stop,
       buffer::gps_second_type stride,
       const channel_names_type& channel_names );

    void iterate_full( buffer::gps_second_type gps_start,
       buffer::gps_second_type gps_stop,
       buffer::gps_second_type stride,
       const channel_names_type& channel_names );

    bool has_next();
    void next( buffers_type_intl& output);

    void next_raw_buffer( buffers_type_intl& output );

    void shutdown( );

    void fill_gap( channel::data_type DataType, channel::size_type DataSizeType, unsigned char *start, unsigned char *end);

    inline time_type
    request_start_time( ) const
    {
        return request_start_time_;
    }

    inline void
    request_start_time( time_type Value )
    {
        request_start_time_ = Value;
    }

    inline time_type
    request_end_time( ) const
    {
      return request_end_time_;
    }

    inline void
    request_end_time( time_type Value )
    {
      request_end_time_ = Value;
    }

    inline bool
    request_in_progress( ) const
    {
      return request_in_progress_;
    }

    inline void
    request_in_progress( bool Value )
    {
      request_in_progress_ = Value;
    }

    /// Given a [start,stop) range and a channel list, determine a good stride value to use
    /// \param gps_start Start time inclusive
    /// \param gps_stop Stop time exclusive
    /// \param selected_channels List of channels
    /// \return A stride that takes into account the presence of trends channels and likely frame file lengths
    NDS::buffer::gps_second_type
    calculate_stride(NDS::buffer::gps_second_type gps_start, NDS::buffer::gps_second_type gps_stop,
                       NDS::channels_type_intl &selected_channels) const;

      inline void
    termination_block( )
    {
      //-----------------------------------------------------------------
      // NDS1 transfers end with a 'termination block', an empty block
      // that is indistinguisable from a 'data not found' condition.
      // If this is an NDS1 connection, we must digest the termination
      // block.
      //-----------------------------------------------------------------
      if ( ( request_in_progress( ) )
	   && ( protocol == PROTOCOL_ONE ) )
      {
	int rc = daq_recv_next( &handle );
	
	if ( rc != DAQD_NOT_FOUND )
	{
	  //-----------------------------------------------------
	  // Unexpected error
	  //-----------------------------------------------------
	  throw daq_error( rc );
	}
      }
    }

    /**
     *
     * @return The current time of the connected nds1 server
     * @note raises an exception if not connected to an nds1 server
     */
    NDS::buffer::gps_second_type
    cur_nds1_gpstime();

    void validate( ) const;

    void validate_daq( int RetCode ) const;

    void infer_daq_channel_info( const std::string &channel_name, daq_channel_t &channel, time_type gps );

    void channel_mask_to_query_type_strings(channel::channel_type channel_type_mask,
                                            std::vector<std::string>& queryTypes);

    // Helper functions for error messages
    std::string get_last_message() const throw();
    std::string err_msg_lookup_failure(const std::string &name);
    std::string err_msg_lookup_ambigous(const std::string &name);
    std::string err_msg_unexpected_no_data_found(buffer::gps_second_type gps_start, buffer::gps_second_type gps_stop, const channel::channel_names_type &names);

    void plan_fetches (buffer::gps_second_type gps_start,
                       buffer::gps_second_type gps_stop,
                       const channel_names_type& channel_names,
                       buffers_type_intl& dest_buffers,
                       request_fragments_type &retval );
    void sync_parameters()
    {
        handle.conceal->max_command_count = static_cast<size_t>(parameters.max_nds1_command_count());
    }
  protected:
    void setup_daq_chanlist(buffer::gps_second_type gps_start,
      const channel_names_type& channel_names,
      bool &have_minute_trends,
      double &bytes_per_sample,
      std::vector<NDS::channel>* final_channel_list = 0);
    void process_check_data_result(int result, bool gaps_ok);

    // Given a channel name return a channel object that it maps to, or raise a daq_error
    // if no good match could be found.
    // Do not use for online channels
    channel select_channel( const std::string & Name );

    void fetch_fragment( request_fragment& fragment, const buffer_initializer &initializer, bool buffers_initialized );
    std::vector<buffers_type_intl> fetch_available( buffer::gps_second_type gps_start,
      buffer::gps_second_type gps_stop,
      const channel_names_type& channel_names );

    SharedPtr < iterate_handler > iterate_handler_;

    epochs_type epoch_cache_;
    epoch current_epoch_;
    channel::hash_type hash_;

    void load_epochs_to_cache();

    channel _parse_nds2_get_channel_line(char *buffer);
    bool _read_uint4(socket_t fd, uint4_type *dest);
    bool _read_buffer(socket_t fd, void *dest, size_t size);

    template<typename Filter, typename Function>
    void retreive_channels_from_nds2(const std::vector<std::string>& types,
                                     const std::string& channel_glob,
                                     Filter& filt,
                                     Function& fn);

  private:
	static bool initialized;
  };

  template<typename Filter, typename Function>
  void connection::p_type::
  retreive_channels_from_nds2(const std::vector<std::string>& types,
                              const std::string& channel_glob,
                              Filter& filter,
                              Function& fn)
 {
      std::vector<char> lineBuffer;
      lineBuffer.resize(512);

      for (std::vector<std::string>::const_iterator cur = types.begin();
           cur != types.end();
           ++cur)
      {
          NDS::dout() << "Retreiving channels for type " << *cur << std::endl;
          {
            std::ostringstream cmdBuf;
            //cmdBuf << "` " << current_epoch_.gps_start << " " << cur_type;
            if (current_epoch_.gps_stop == current_epoch_.gps_start + 1)
                cmdBuf << "get-channels " << current_epoch_.gps_start << " ";
            else
                cmdBuf << "get-channels 0 ";
            cmdBuf << *cur;
            if (channel_glob.compare("*") != 0) cmdBuf << " {" << channel_glob << "}";
            cmdBuf << ";\n";
            validate_daq(daq_send(&(handle), cmdBuf.str().c_str()));
            NDS::dout() << "Sent command '" << cmdBuf.str() << "'" << std::endl;
          }

          socket_t fd = handle.conceal->sockfd;
          uint4_type count=0;
          NDS::dout() << "Reading channel count" << std::endl;
          if (!_read_uint4(fd, &count)) throw unexpected_channels_received_error();
          //result.reserve(count);

          NDS::dout() << "Expecting " << count << "channels" << std::endl;
          for (uint4_type i = 0; i < count; ++i) {
            uint4_type line_size = 0;
            if (!_read_uint4(handle.conceal->sockfd, &line_size)) throw unexpected_channels_received_error();
            if (line_size < lineBuffer.size()) lineBuffer.resize(line_size);
            if (!_read_buffer(handle.conceal->sockfd, &(lineBuffer[0]), line_size)) throw unexpected_channels_received_error();
            channel curCh = _parse_nds2_get_channel_line(&(lineBuffer[0]));
            if (filter(curCh))
            {
                fn(curCh);
            }
          }
      }
  }

} // namespace NDS
