/* -*- mode: C++ ; c-basic-offset: 2; indent-tabs-mode: nil -*- */

/*
 * High-level NDS client C++ library
 *
 * Copyright (C) 2015  Edward Maros <ed.maros@ligo.org>
 *
 * This work is derived from the C NDS clientlibrary created by Leo Singer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef SWIG__COMMON__NDS_CONNECTION_HH
#define SWIG__COMMON__NDS_CONNECTION_HH


/**
 * \if (DEVELOPER_SECTION)
 * \page NDSClientBindings  NDS Client Bindings
 * \else
 * \mainpage NDS Client Bindings
 * \endif
 * \section Introduction
 * The %NDS Client bindings are a set of C/C++ libraries that provide access to the LIGO project %NDS servers.
 * The libraries provide a uniform interface to both NDS1 and NDS2 servers.  In addition to the C/C++ libraries a series
 * of modules are provided to allow Java, MATLAB, Octave, and Python programs to interface with the %NDS systems.
 *
 * \section Main Components
 * The main components of the library are the \link NDS::connection connection \endlink, \link NDS::buffer buffer \endlink objects, and \link NDS::channel channel \endlink objects.
 *
 * \subsecection Connections Connections
 * The libraries are connection oriented.  Each \link NDS::connection connection \endlink object connects to a single %NDS server and allows the user
 * to query for available channels and retreive data.  This is the basic class you use to interact with the %NDS server.  All requests for data are done through
 * a connection object.
 *
 * \subscection Channels Channels
 * The \link NDS::channel channel \endlink class represents a stored channel.  It provides access to the metadata, such as name, rate, frame type, and data type.
 * The channel does not give access to the data.  For that a buffer class is used.
 *
 * \subsection Buffers Buffers
 * The \link NDS::buffer buffer \endlink class is the combination of channels and data.  When you make requests for data to a connection you recieve a series of buffers.
 * 
 * \subsection Using Using the NDS client
 * Each of the C++ classes are broken out into their own header files.  However there is a combined header file that should be included by the application.
 * \code
 * #include "nds.hh"
 * \endcode
 *
 * \section NDS1vs2 Differences between NDS1 and NDS2
 * NDS1 has no concept of channel history.  It maintains one channel list, the current channel list.  It cannot read historical data if channels have changed name or rate.
 * NDS1 is only available in the LIGO control rooms.  It is not exported to the outside world and is part of the LIGO control systems.
 *
 * NDS2 Contains the full history of LIGO data.  It tracks each name and rate change.  Channels may have several co-existing versions, and may have long stretches where 
 * they do not exist.  When accessing data, users should keep in mind that recent data is often cached on disk and as such can be read with relatively low latency.  However
 * data that is not cached on disk must be pulled from tape archives and will have a high latency for the request.
 *
 * \subsection ChannelNames Channel names in NDS2
 * With the full history of channels in NDS2 a name may map to multiple channels.  The NDS2 server will attempt to extrapolate the channel you ask for based
 * on the name and timespans given.  However a user may specify more information in a channel name in order to force the NDS2 server to select the desired channel.
 * The user must specify a base name, and may also specify channel types, and rates.
 *
 * For example the H1 strain data is in H1:GDS-CALIB_STRAIN.  However there are multiple sources of the data.  The real time online data and the archived reduced type.
 * You can fully qualify the name by including frame type and rate information.  So you could qualify the name as H1:GDS-CALIB_STRAIN,reduced,16384 or H1:GDS-CALIB_STRAIN,reduce to
 * denote archived strain data.  To specify the online, real time stream of strain you could use H1:GDS-CALIB_STRAIN,online,16384 or H1:GDS-CALIB_STRAIN,online.
 */

#include <stdexcept>
#include <string>
#include <ostream>

#include "nds_memory.hh"

#include "nds_export.hh"
#include "nds_buffer.hh"
#include "nds_channel.hh"
#include "nds_availability.hh"
#include "nds_epoch.hh"

namespace NDS
{
  /**
   * \brief A connection to the NDS/NDS2 server
   * \details The connection object is used to interact with the NDS (v1 or v2) servers to retrieve
   *          data and channel lists.
   *          Connnection objects have a few main categories of methods:
   *          1. Data retreival (fetch and iterate)
   *          2. Channel listing (find_channels)
   *
   *          In addition there are methods designed to deal with the volume of data and channels in a NDS2 server:
   *          1. Setting limits/epochs on requests (get_epoch, set_epoch, current_epoch)
   *          2. Channel availability checking (get_availability)
   *
   * \headerfile nds.hh nds.hh
   */
  class connection
  {
  public:
    /**
     * \brief Base class for NDS errors
     */
    class error
      : public std::runtime_error
    {
    public:
      error( const std::string& What,
             int ErrNo = 0 );
    };

    /**
     * \brief Signal that the connection has already been closed.
     */
    class already_closed_error
      : public error
    {
    public:
      already_closed_error( );
    };

    /**
     * \brief Signals that a minute trend has been requested but the start/stop times are not divisible by 60.
     */
    class minute_trend_error
      : public error
    {
    public:
      minute_trend_error( );
    };

    /**
     * \brief A transfer is already in progress
     */
    class transfer_busy_error
      : public error
    {
    public:
      transfer_busy_error( );
    };

    /**
     * \brief An unexpected channel or data was received
     */
    class unexpected_channels_received_error
      : public error
    {
    public:
      unexpected_channels_received_error( );
    };

    /**
     * \brief an error state was returned by the NDS server
     */
    class daq_error
      : public error
    {
    public:
	  daq_error( int daq_code );
	  daq_error( int daq_code, const std::string &additional_information);
	  daq_error( int daq_code, const char* additional_information);

      inline int DAQCode( ) const
      {
        return error_code;
      }

    private:
	  int		index;
      int       error_code;

	  static int counter_;

	  static std::string format( int DAQCode, const char *extra = 0);
    };

    /**
     * \brief Connection constants
     */
    typedef enum {
      PROTOCOL_INVALID = -1,  ///< Unknown or invalid connection
      PROTOCOL_ONE = 1,       ///< Connect with NDS1 protocol
      PROTOCOL_TWO = 2,       ///< Connect with NDS2 protocol
      PROTOCOL_TRY = 3        ///< Autodetect server protocol
    } protocol_type;

    typedef std::string host_type;  ///< Host name type
    typedef int port_type;          ///< Host port number type
    typedef size_t count_type;      ///< Generic count time

    /**
     * \brief A list of channel names
     */
    typedef channel::channel_names_type channel_names_type;
    typedef std::vector<std::string> parameters_type;

    static const port_type DEFAULT_PORT = 31200;  ///< Default NDS2 port number

    /**
     *  \brief Default constructor
     */
    DLL_EXPORT
    connection( );

    /**
     *  \brief Create a connection
     *  \param[in]  host Server to connect to
     *  \param[in]  port Port number to connect to (defaults to DEFAULT_PORT)
     *  \param[in]  protocol Protocol version to use (defaults to PROTOCOL_TRY)
     */
    DLL_EXPORT
    connection( const host_type& host,
                port_type port = DEFAULT_PORT,
                protocol_type protocol = PROTOCOL_TRY );
    /**
     * \brief Destroy the connection
     */
    DLL_EXPORT
    ~connection( );

    /**
     * \brief Close the connection
     */
    DLL_EXPORT
    void close( );

    /**
     * \brief Return the count of channels matching the given parameters.
     * \detail Return a count of channels that are available form this connection within the given contstraints.
     * \param[in] channel_glob A string to match channels to.  Bash style globs are allowed (* to mach anything, ? matches 1 character).  "*" matches all channels.
     * \param[in] channel_type_mask Set of channel types to search.  Defaults to all.
     * \param[in] data_type_mask A set of data types to search.  Defaults to all.
     * \param[in] min_sample_rate The lowest sample rate to return.
     * \param[in] max_sample_rate The highest sample rate to return.
     *
     * \return A count of channels matching the input parameters.
     *
     * \remarks For NDS1 a local cache of the channel list is searched.  For NDS2 this call may result in one or
     *          requests to the server, no local cache is used.
     */
    DLL_EXPORT count_type
    count_channels( std::string channel_glob = "*",
                   channel::channel_type channel_type_mask = channel::DEFAULT_CHANNEL_MASK,
                   channel::data_type data_type_mask = channel::DEFAULT_DATA_MASK,
                   channel::sample_rate_type min_sample_rate = channel::MIN_SAMPLE_RATE,
                   channel::sample_rate_type max_sample_rate = channel::MAX_SAMPLE_RATE );

    /**
     * \brief Retrieve a list of channels
     * \detail Return a list of channels that are available form this connection within the given contstraints.
     * \param[in] channel_glob A string to match channels to.  Bash style globs are allowed (* to mach anything, ? matches 1 character).  "*" matches all channels.
     * \param[in] channel_type_mask Set of channel types to search.  Defaults to all.
     * \param[in] data_type_mask A set of data types to search.  Defaults to all.
     * \param[in] min_sample_rate The lowest sample rate to return.
     * \param[in] max_sample_rate The highest sample rate to return.
     *
     * \return A list of channels (as a chanenls_type) matching the input parameters.
     *
     * \remarks For NDS1 a local cache of the channel list is searched.  For NDS2 this call may result in one or
     *          requests to the server, no local cache is used.
     */
    DLL_EXPORT channels_type
    find_channels( std::string channel_glob = "*",
                   channel::channel_type channel_type_mask = channel::DEFAULT_CHANNEL_MASK,
                   channel::data_type data_type_mask = channel::DEFAULT_DATA_MASK,
                   channel::sample_rate_type min_sample_rate = channel::MIN_SAMPLE_RATE,
                   channel::sample_rate_type max_sample_rate = channel::MAX_SAMPLE_RATE );

    /**
     * \brief Return a list of epochs
     * \detail NDS2 has the concept of epochs.  A epoch may be set on a connection to constrain all data and channel requests
     * to a given time frame.  This call returns the currently available epochs.
     *
     * \return A list of epocsh
     *
     * \remarks Only available on NDS2 connections.
     */
    DLL_EXPORT epochs_type
    get_epochs();

    /**
     * \brief Set the epoch for the connection
     * \detail Given a epoch value described as a string set the current epoch.
     * All requests for data and channel lists will be constrained to this epoch.
     *
     * \param[in] epoch The epoch value as a string.
     *
     * \returns True if the epoch can be set
     *
     * \remark The epoch may be a named epoch, ie "O1" or a string with start/stop gps times, ie "1126621184-1137258496".
     * \remark This command only works for NDS2, NDS1 servers will silently ignore it.
     */
    DLL_EXPORT bool
    set_epoch(std::string epoch);

    /**
     * \brief Set the epoch for the connection
     * \detail Given a epoch value described as a start, stop pair of gps set the current epoch.
     * All requests for data and channel lists will be constrained to this epoch.
     *
     * \param[in] gps_start The start time in seconds from the gps epoch.
     * \param[in] gps_stop The one second after the last second of the requeste epoch, in seconds fromt the gps epoch.
     *
     * \returns True if the epoch can be set
     *
     * \remark Sets the epoch to \verbatim[gps_start,gps_stop)\endverbatim
     * \remark This command only works for NDS2, NDS1 servers will silently ignore it.
     */
    DLL_EXPORT bool
    set_epoch(buffer::gps_second_type gps_start,
            buffer::gps_second_type gps_stop);

    /**
     * \brief Return the current epoch
     * \detail Return the currently set epoch.
     *
     * \returns A epoch value representing the current epoch.
     * \remark This command only works for NDS2
     */
    DLL_EXPORT epoch
    current_epoch() const;

    /**
     * \brief Given a list of channels return their availability over the current epoch.
     * 
     * \param[in] channel_names A list of channels to return availability for.
     *
     * \returns A list a channel availabilities.  There is one set of aviailabilities returned per channel requested.
     */
    DLL_EXPORT availability_list_type    
    get_availability( const channel_names_type& channel_names );

    /**
     * \brief Check to see if data is avaiable
     * \detail Given a start/stop time and a channel list check to see if a fetch request would suceed.  This will return failure if there is a gap in the data, or
     * if the data is on tape.
     *
     * \param[in] gps_start Start time of request
     * \param[in] gps_stop Stop time of request
     * \param[in] channel_names The channel list
     *
     * \returns True if the data is availble and not on tape
     */
    DLL_EXPORT bool
    check( buffer::gps_second_type gps_start,
           buffer::gps_second_type gps_stop,
           const channel_names_type& channel_names );

    /**
     * \brief Retreive data from the server
     * \detail Given a start/stop time and a channel list retreive the associated data.
     *
     * \param[in] gps_start Start time of request
     * \param[in] gps_stop Stop time of request
     * \param[in] channel_names The channel list
     *
     * \returns A list of buffers containing the request data.
     *
     * \remarks This command can respond to missing data and high latency situations in various ways.  Use the (get/set)_parameter function to set how data gaps
     * or data on tape situations are handled.
     *
     * \remarks On systems where this is not built as C++11 this may incure a large overhead, do the need to the compiler doing extra copies of the return buffers.
     */
    DLL_EXPORT buffers_type
    fetch( buffer::gps_second_type gps_start,
           buffer::gps_second_type gps_stop,
           const channel_names_type& channel_names );

    /**
     * \brief Retreive data from the server
     * \detail Given a start/stop time and a channel list retreive the associated data.
     *
     * \param[in] gps_start Start time of request
     * \param[in] gps_stop Stop time of request
     * \param[in] channel_names The channel list
     * \param[out] output List of buffers to hold the output.
     *
     * \returns A list of buffers containing the request data.
     *
     * \remarks This command can respond to missing data and high latency situations in various ways.  Use the (get/set)_parameter function to set how data gaps
     * or data on tape situations are handled.
     *
     * \remarks If an exception is thrown, then output is not modified.  Output will be resized to fit the data.
     */
    DLL_EXPORT void
    fetch( buffer::gps_second_type gps_start,
           buffer::gps_second_type gps_stop,
           const channel_names_type& channel_names,
           buffers_type& output );

    /**
     * \brief Return the host that is currently connected.
     * \returns The hostname
     */
    DLL_EXPORT host_type get_host( ) const;

    /**
     * \brief Return the port number of the current connection.
     * \returns The connection port number.
     */
    DLL_EXPORT port_type get_port( ) const;

    /**
     * \brief Return the protocol version in use.
     * \returns The protocol version (PROTOCOL_ONE|PROTOCOL_TWO)
     */
    DLL_EXPORT protocol_type get_protocol( ) const;

    /**
     * \brief Change the default behavior of the connection.
     * \detail The connection object has a series of parameters that can be set.  Currently the parameters
	 * that can be set are "ALLOW_DATA_ON_TAPE", "GAP_HANDLER", and "ITERATE_USE_GAP_HANDLERS"
     *
     * \section parameters Parameters
     * \subsection tape ALLOW_DATA_ON_TAPE
     * NDS2 only.  The NDS2 server may serve data that resides on a high latency storage layer, such as a tape system.  This may lead
     * to data requests taking minutes or hours to complete, depending on the load on the storage system.  As of version 0.12 of the client
     * the default is to raise an error when accessing data that is on a high latency storage layer.  This allows the
     * application to provide feedback (if needed) to a users regarding amount of time that a request may take.  If this parameter is set to a 
     * true value ("True", "1", "yes") then an error will not be raised when requesting data on a high latency storage.
     *
     * \subsection gaps GAP_HANDLER
     * For a given request there may not be be data available to fill the request completely.  This happens due to issues upstream of the
     * NDS server.  How this is handled is application specific.  Setting the "GAP_HANDLER" parameter allows the application to
     * specify what to do.  This includes options such as abort, zero fill the data, ...
     *
     * Available GAP_HANDLERS
     *  * "ABORT_HANDLER" This aborts the request when a gap is found in the data.
     *  * "STATIC_HANDLER_ZERO" This zero fills any missing data.
     *  * "STATIC_HANDLER_ONE" This fills any missing data with ones.
     *  * "STATIC_HANDLER_NAN" This fills any missing data with NaN values (or zero for integer channels).
     *  * "STATIC_HANDLER_POS_INF" This fills any missing data with +infinity (or the maximum integer value for integer channels).
     *  * "STATIC_HANDLER_NEG_INF" This fills any missing data with -infinity (or the minimum integer value for integer channels).
     *
	 * \subsection iterate ITERATE_USE_GAP_HANDLERS
	 * The iterate methods have a special case.  Unlike fetch operations which work on a single block, the iterate methods retrieve chunks of data that
	 * may not need to be contigous.  Setting ITERATE_USE_GAP_HANDLERS to "false" configures the connection to simply skip any gaps in the data and only return
	 * the data that is available.
	 *
	 * Please note that if you are asking for multiple channels that do not have identical gaps the NDS servers will return a data not found error if ITERATE_USE_GAP_HANDLERS is set to false.
	 *
     * \param[in] parameter A parameter name as a string.
     * \param[in] value The value to set parameter to, as a string.
     *
     * \returns True if the value could be set, else false.
     *
     */
    DLL_EXPORT bool set_parameter(const std::string &parameter, const std::string &value);

    /**
     * \brief Retreive the current parameter setting on a connection.
     * \detail Retreive the current value set for parameter.  See #set_parameter for documentation
     * on the available parameters.
     *
     * \param[in] parameter Parameter name, as a string.
     *
     * \returns The parameter value, or "" if an invalid parameter is found.
     */
    DLL_EXPORT std::string get_parameter( const std::string &parameter ) const;

    /**
     * \brief Return a list of supported parameters.
     * \returns a list of parameter names that may be used with #set_parameter or #get_parameter
     */
    DLL_EXPORT parameters_type get_parameters( ) const;

    /**
     * \brief Return the NDS2 channel hash.
     * \detail NDSv2 server can return a hash of its internal channel list.  This hash value can be used
     * by a client to determine if the channel list has updated.
     *
     * \returns A hash value.
     *
     * \remarks NDSv2 only.
     */
    DLL_EXPORT const channel::hash_type& hash( ) const;

    /**
     * \brief Retreive data in segments.
     * \detail Setup an iterative data retrieval process.  This function initiates the request, but does
     * not actually retreive the data.  Once iterate is called, then the data is retreived via the #next method.
     *
     * \param[in] channel_names The list of channels to retrieve data for.
     *
     * \returns A reference to the connection.
     *
     * \remarks This version is for reading on-line data, streaming online type channels from the current time. 
     * This autoconfigures the chunck size for returned data.
     */
    DLL_EXPORT connection&
    iterate( const channel_names_type& channel_names ) { return iterate( 0, 0, 0, channel_names); };

    /**
     * \brief Retreive data in segments.
     * \detail Setup an iterative data retrieval process.  This function initiates the request, but does
     * not actually retreive the data.  Once iterate is called, then the data is retreived via the #next method.
     *
     * \param[in] stride The number of seconds of data to return in each chunk.
     * \param[in] channel_names The list of channels to retrieve data for.
     *
     * \returns A reference to the connection.
     *
     * \remarks This version is for reading on-line data, streaming online type channels from the current time.
     */
    DLL_EXPORT connection&
    iterate( buffer::gps_second_type stride,
             const channel_names_type& channel_names ) { return iterate( 0, 0, stride, channel_names); }

    /**
     * \brief Retreive data in segments.
     * \detail Setup an iterative data retrieval process.  This function initiates the request for data in [gps_start, gps_stop), but does
     * not actually retreive the data.  Once iterate is called, then the data is retreived via the #next method.
     *
     * \param[in] gps_start The start time of the request.
     * \param[in] gps_stop The end time of the request [gps_start, gps_stop).
     * \param[in] channel_names The list of channels to retrieve data for.
     *
     * \returns A reference to the connection.
     *
     * \remarks This version autoconfigures a stride value.
     */
    DLL_EXPORT connection&
    iterate( buffer::gps_second_type gps_start,
             buffer::gps_second_type gps_stop,
             const channel_names_type& channel_names ) { return iterate( gps_start, gps_stop, 0, channel_names ); }

    /**
     * \brief Retreive data in segments.
     * \detail Setup an iterative data retrieval process.  This function initiates the request for data in [gps_start, gps_stop), but does
     * not actually retreive the data.  Once iterate is called, then the data is retreived via the #next method.
     *
     * \param[in] gps_start The start time of the request.
     * \param[in] gps_stop The end time of the request [gps_start, gps_stop).
     * \param[in] stride The number of seconds of data to return in each chunk.
     * \param[in] channel_names The list of channels to retrieve data for.
     *
     * \returns A reference to the connection.
     */
    DLL_EXPORT connection&
    iterate( buffer::gps_second_type gps_start,
             buffer::gps_second_type gps_stop,
             buffer::gps_second_type stride,
             const channel_names_type& channel_names );

    /**
     * \brief Retreive the next data block from an iterate request.
     *
     * \returns A list of buffers.
     */
    DLL_EXPORT buffers_type
    next( );

    /**
     * \brief Retreive the next data block from an iterate request.
     * \detail Retreive the next data block from an iterate reqeust and store the data in the passed output buffer.
     *
     * \param[out] output The output buffer.
     *
     * \remark This version of next is provided to avoid the overhead of copying a buffers_type object.
     */
    DLL_EXPORT void
    next( buffers_type& output );

    /**
     * \brief Query the connection to see if a request is in progress.
     *
     * \returns True if a request is in progress, else false.
     *
     * \remarks A connection can only do on request at a time.
     */
    DLL_EXPORT bool request_in_progress( ) const;

    /**
     * \brief Manually close the connection
     */
    DLL_EXPORT void shutdown( );

  private:
    friend class daq_accessor;

    struct p_type;

    p_type* p_;

    //-------------------------------------------------------------------
    /// \brief Copy Constructor
    ///
    /// The copy constructor is not available in this interface.
    //-------------------------------------------------------------------
    connection( connection& Source );

    /**
     * \brief Check for more data in the current iterate request
     * \detail has_next returns true if the current iterate request has more data to be read.  Data should be read using the #next() method.
     *
     * \returns True if there is more data, else false
     */
    bool
    has_next( );

  };

  DLL_EXPORT extern std::ostream &operator<<(std::ostream &os,const connection &conn);

} // namespace - NDS

#endif /* SWIG__COMMON__NDS_CONNECTION_HH */
