#include "nds.hh"
#include "nds_request_fragment.hh"
#include "nds_connection.hh"
#include "nds_connection_ptype.hh"
#include "nds_gap_handler.hh"
#include "nds_composer.hh"

#include "debug_stream.hh"

namespace NDS
{
  fixed_point_gap_handler composer::zero_gap_handler( fixed_point_gap_handler::static_val::ZERO_VAL );

  composer::
  composer(request_fragment::working_buffers& buffers, const buffer_initializer &initializer, const gap_handler &_gap_handler, bool empty_buffers_are_safe): \
   buffers(buffers), initializer(initializer), \
   _gap_handler(_gap_handler), \
   cur_progress(buffers.size(), 0), \
   delayed_handlers(), empty_is_safe(empty_buffers_are_safe), \
   initialized(false), finished(false) { }

  composer::~composer()
  {
    while (!delayed_handlers.empty())
    {
      delayed_gap_handler *cur = delayed_handlers.back();
      delayed_handlers.pop_back();
      if (cur)
      {
        delete cur;
      }
    }
  }

  void composer::
  add_segment(const buffers_type_intl &cur_bufs)
  {
  	if (finished) return;
  	if (cur_bufs.size() != buffers.size())
  	{
  		return;
  	}
  	// first time through we need to size everything right
    // we put it off until here so we know the sample rates and such
  	if (!initialized) initialize_buffers(cur_bufs);
  	
    size_t i = 0;
  	for (buffers_type_intl::const_iterator cur = cur_bufs.begin();
  		cur != cur_bufs.end();
  		++cur, ++i)
    {
      const buffer &cur_buffer = *cur;
      buffer::gps_second_type offset_secs = cur_buffer.Start() - buffers[i]->Start();
      buffer::size_type offset_samples = cur_buffer.seconds_to_samples( offset_secs );

      bounds_check( *(buffers[i]), cur_progress[i], offset_samples, offset_samples + cur_buffer.Samples());

      size_t ssize = cur_buffer.DataTypeSize();
      unsigned char *start = 0, *end = 0;
      if (cur_progress[i] != offset_samples)
      {
      	// only print the debug output once
        if (i == 0)
          NDS::dout() << "   GAP: " << cur_buffer.samples_to_seconds(cur_progress[i]) + buffers[i]->Start() << " - " << (size_t)(offset_secs+buffers[i]->Start()) << std::endl;
        fill_gap( *(buffers[i]), cur_progress[i], offset_samples);
      }
      if (i == 0) 
      {
        NDS::dout() << std::endl << "  " << cur_bufs.size() << " channels" << std::endl;
        NDS::dout() << "  " << cur_buffer.Start() << ":" << cur_buffer.StartNano() << " - " << cur_buffer.Stop() << std::endl << std::endl;
      }
      unsigned char *dest = const_cast<unsigned char *>(&( (*buffers[i])[cur_buffer.samples_to_bytes(offset_samples)] ));
      start = const_cast<unsigned char*>(&(cur_buffer[0]));
      end = start + (cur_buffer.samples_to_bytes(cur_buffer.Samples()));
      std::copy(start, end, dest);
      NDS::dout() << "i: " << i << " start = " << (void *)start << " stop = " << (void *)end << " dest = " << (void *)dest;
      NDS::dout() << " Samples: " << cur_buffer.Samples() << " sample size = " << cur_buffer.DataTypeSize() << std::endl;
      cur_progress[i] = offset_samples + cur_buffer.Samples();
    }
  }

  void composer::
  finish()
  {
  	if (finished) return;
	finished = true;
	if (!initialized && !empty_is_safe) throw connection::daq_error( DAQD_NOT_FOUND, "None of the selected channels returned data." );
  	/* handle trailing gaps */
  	size_t i=0;
    for (request_fragment::working_buffers::iterator cur = buffers.begin();
    	cur != buffers.end();
    	++cur, ++i)
    {
      bool gap_printed = false;
      const buffer& cur_buffer = *(*cur);
      if (cur_progress[i] < cur_buffer.Samples())
      {
        buffer::size_type gap_size = cur_buffer.Samples() - cur_progress[i];
        if (!gap_printed)
        {
          gap_printed = true;
          NDS::dout() << "   Trailing GAP: " << cur_buffer.samples_to_seconds(cur_progress[i]) + cur_buffer.Start() << " - " << cur_buffer.Stop() << std::endl;
        }
        // trailing gap
        fill_gap( *(buffers[i]), cur_progress[i], cur_buffer.Samples() );
        cur_progress[i] = cur_buffer.Samples( );
      }
    }
    while (!delayed_handlers.empty())
    {
    	delayed_gap_handler *cur = delayed_handlers.back();
    	delayed_handlers.pop_back();
    	if (cur)
    	{
    		(*cur)();
    		delete cur;
    	}
    }
  	finished = true;
  }

  void composer::
  initialize_buffers(const buffers_type_intl &ref_set)
  {
    if (initialized) return;
    if (ref_set.size() != buffers.size()) return;
    int j = 0;
    for (buffers_type_intl::const_iterator cur_buf = ref_set.begin();
          cur_buf != ref_set.end();
          ++cur_buf, ++j)
    {
      initializer.reset_buffer( buffers[j], *cur_buf );
      NDS::dout() << "Initializing buffer for " << buffers[j]->Name() << " with " << buffers[j]->Samples() << " entries" << std::endl;
    }
    initialized = true;
  }

  void composer::
  bounds_check(const buffer& cur_buffer, buffer::size_type cur_fill, buffer::size_type offset_start, buffer::size_type offset_end)
  {
    buffer::size_type dest_samples = cur_buffer.Samples();

    if (offset_start >= dest_samples || offset_start > offset_end ||
      offset_end > dest_samples || offset_start < cur_fill)
    {
      throw connection::unexpected_channels_received_error();
    }
  }
}
