/*
 * High-level NDS client library
 *
 * Copyright (C) 2012  Leo Singer <leo.singer@ligo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#if HAVE_CONFIG_H
#include "daq_config.h"
#endif /* HAVE_CONFIG_H */

#include "nds.h"

#include <daqc.h>
#include <daqc_response.h>
#include <channel.h>

#if HAVE_IO_H
#include <io.h>
#endif /* HAVE_IO_H */
#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <math.h>
#include <sqlite3.h>

#include "nds_os.h"

#include "bash_pattern.h"
#include "asprintf.c"

/* includes for locating database file */
#ifdef _WIN32
#include <Shlobj.h>
#define snprintf _snprintf
#ifndef R_OK
#define R_OK 4
#define W_OK 2
#define F_OK 0
#endif /* R_OK */
#else
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#endif

struct _nds2_connection
{
	daq_t daq;
	sqlite3 *db;
	char *host;
	int port;
	int connected;

	/* State variables for making iterative data requests */
	long requested_end_time;
	int request_in_progress;

	/* FIXME: hack for sqlite versions < 3.7.11.  should be
	 * removed when we can depend on newer sqlite versions */
	int _db_readonly;
};


const char *_nds2_query_create_tables =
"CREATE TABLE IF NOT EXISTS servers ("
"	protocol INTEGER NOT NULL DEFAULT 3,"
"	timestamp INTEGER NOT NULL DEFAULT 0,"
"	hash BLOB NOT NULL DEFAULT \'????????\'"
");"
"CREATE TABLE IF NOT EXISTS channels ("
"	name TEXT NOT NULL,"
"	channel_type INTEGER NOT NULL,"
"	data_type INTEGER NOT NULL,"
"	sample_rate DOUBLE NOT NULL,"
"	signal_gain FLOAT NOT NULL,"
"	signal_slope FLOAT NOT NULL,"
"	signal_offset FLOAT NOT NULL,"
"	signal_units TEXT NOT NULL"
");"
"CREATE INDEX IF NOT EXISTS channel_name_index ON channels (name);"
;


static const int _NDS2_TYPICAL_BYTES_PER_FRAME = (1 << 26);


typedef struct {
	char *expr;
	bash_pattern *pattern;
} _bash_pattern_auxdata;


static void _bash_pattern_free_auxdata(void *arg)
{
	_bash_pattern_auxdata *auxdata = (_bash_pattern_auxdata *) arg;
	if (auxdata)
	{
		free(auxdata->expr);
		auxdata->expr = NULL;
		bash_pattern_free(auxdata->pattern);
		auxdata->pattern = NULL;
		free(auxdata);
	}
}


static void _bash_pattern_matches_func(sqlite3_context *ctx, int nargs, sqlite3_value **args)
{
	_bash_pattern_auxdata *auxdata;
	const char *expr;
	const char *text;
	int ret = 0;

	/* If there are not exactly two arguments, then fail. */
	if (nargs != 2)
	{
		sqlite3_result_error(ctx, "exptected 2 arguments", -1);
		goto fail;
	}

	/* If the pattern expression is NULL, return false. */
	expr = (const char *)sqlite3_value_text(args[0]);
	if (!expr)
		goto done;

	/* If the text is NULL, return false. */
	text = (const char *)sqlite3_value_text(args[1]);
	if (!text)
		goto done;

	auxdata = (_bash_pattern_auxdata *) sqlite3_get_auxdata(ctx, 0);
	if (!auxdata || strcmp(expr, auxdata->expr))
	{
		auxdata = malloc(sizeof(_bash_pattern_auxdata));
		if (!auxdata)
		{
			sqlite3_result_error_nomem(ctx);
			goto fail;
		}
		auxdata->expr = strdup(expr);
		if (!auxdata->expr)
		{
			free(auxdata);
			sqlite3_result_error_nomem(ctx);
			goto fail;
		}
		auxdata->pattern = bash_pattern_compile(expr);
		if (!auxdata->pattern)
		{
			free(auxdata->expr);
			free(auxdata);
			sqlite3_result_error(ctx, "cannot compile pattern", -1);
			goto fail;
		}
		sqlite3_set_auxdata(ctx, 0, auxdata, _bash_pattern_free_auxdata);
	}

	ret = bash_pattern_matches(auxdata->pattern, text);
done:
	sqlite3_result_int(ctx, ret);
fail:
	return;
}


static int _bash_pattern_matches_register_func(sqlite3 *db)
{
	return sqlite3_create_function(db, "bash_pattern_matches", 2, SQLITE_UTF8, NULL, _bash_pattern_matches_func, NULL, NULL);
}


static int _nds2_db_create_tables(sqlite3 *db)
{
	return sqlite3_exec(db, _nds2_query_create_tables, NULL, NULL, NULL);
}


static int _nds2_db_start_transaction(sqlite3 *db)
{
	return sqlite3_exec(db, "BEGIN", NULL, NULL, NULL)
		|| _nds2_db_create_tables(db);
}


static int _nds2_db_end_transaction(sqlite3 *db)
{
	return sqlite3_exec(db, "COMMIT", NULL, NULL, NULL);
}


static nds2_channel_type _nds2_chantype_to_channel_type(chantype_t c)
{
	nds2_channel_type channel_type;

	switch (c)
	{
		case cOnline:
			channel_type = NDS2_CHANNEL_TYPE_ONLINE;
			break;
		case cRaw:
			channel_type = NDS2_CHANNEL_TYPE_RAW;
			break;
		case cRDS:
			channel_type = NDS2_CHANNEL_TYPE_RDS;
			break;
		case cSTrend:
			channel_type = NDS2_CHANNEL_TYPE_STREND;
			break;
		case cMTrend:
			channel_type = NDS2_CHANNEL_TYPE_MTREND;
			break;
		case cTestPoint:
			channel_type = NDS2_CHANNEL_TYPE_TEST_POINT;
			break;
		case cStatic:
			channel_type = NDS2_CHANNEL_TYPE_STATIC;
			break;
		default:
			channel_type = 0;
	}

	return channel_type;
}


static chantype_t _nds2_chantype_to_channel_type_reverse(nds2_channel_type c)
{
	chantype_t channel_type;

	switch (c)
	{
		case NDS2_CHANNEL_TYPE_ONLINE:
			channel_type = cOnline;
			break;
		case NDS2_CHANNEL_TYPE_RAW:
			channel_type = cRaw;
			break;
		case NDS2_CHANNEL_TYPE_RDS:
			channel_type = cRDS;
			break;
		case NDS2_CHANNEL_TYPE_STREND:
			channel_type = cSTrend;
			break;
		case NDS2_CHANNEL_TYPE_MTREND:
			channel_type = cMTrend;
			break;
		case NDS2_CHANNEL_TYPE_TEST_POINT:
			channel_type = cTestPoint;
			break;
		case NDS2_CHANNEL_TYPE_STATIC:
			channel_type = cStatic;
			break;
		default:
			channel_type = cUnknown;
	}

	return channel_type;
}


static nds2_data_type _nds2_data_to_data_type(daq_data_t c)
{
	nds2_data_type data_type;

	switch (c)
	{
		case _16bit_integer:
			data_type = NDS2_DATA_TYPE_INT16;
			break;
		case _32bit_integer:
			data_type = NDS2_DATA_TYPE_INT32;
			break;
		case _64bit_integer:
			data_type = NDS2_DATA_TYPE_INT64;
			break;
		case _32bit_float:
			data_type = NDS2_DATA_TYPE_FLOAT32;
			break;
		case _64bit_double:
			data_type = NDS2_DATA_TYPE_FLOAT64;
			break;
		case _32bit_complex:
			data_type = NDS2_DATA_TYPE_COMPLEX32;
			break;
		case _32bit_uint:
			data_type = NDS2_DATA_TYPE_UINT32;
			break;
		default:
			data_type = 0;
	}

	return data_type;
}


static daq_data_t _nds2_data_to_data_type_reverse(nds2_data_type c)
{
	daq_data_t data_type;

	switch (c)
	{
		case NDS2_DATA_TYPE_INT16:
			data_type = _16bit_integer;
			break;
		case NDS2_DATA_TYPE_INT32:
			data_type = _32bit_integer;
			break;
		case NDS2_DATA_TYPE_INT64:
			data_type = _64bit_integer;
			break;
		case NDS2_DATA_TYPE_FLOAT32:
			data_type = _32bit_float;
			break;
		case NDS2_DATA_TYPE_FLOAT64:
			data_type = _64bit_double;
			break;
		case NDS2_DATA_TYPE_COMPLEX32:
			data_type = _32bit_complex;
			break;
		case NDS2_DATA_TYPE_UINT32:
			data_type = _32bit_uint;
			break;
		default:
			data_type = _undefined;
	}

	return data_type;
}


/**
 * return: 0=success, -1=failure
 */
static int _nds2_chan_req_to_channel(nds2_channel *channel, chan_req_t *chanreq)
{
	/* Check to make sure name field will fit. */
	if (strlen(chanreq->name) >= sizeof(channel->name))
	{
		errno = NDS2_CHANNEL_NAME_TOO_LONG;
		goto fail;
	}

	/* Copy fields. */
	strncpy(channel->name, chanreq->name, sizeof(channel->name));
	channel->channel_type = _nds2_chantype_to_channel_type(chanreq->type);
	channel->data_type = _nds2_data_to_data_type(chanreq->data_type);
	channel->sample_rate = chanreq->rate;
	channel->signal_gain = chanreq->s.signal_gain;
	channel->signal_slope = chanreq->s.signal_slope;
	channel->signal_offset = chanreq->s.signal_offset;
	strncpy(channel->signal_units, chanreq->s.signal_units, sizeof(channel->signal_units));

	return 0;
fail:
	return -1;
}

/**
 * Open database in specified db directory.  The access mode of the db
 * object will be determined by the access mode of the db file itself.
 * return: 0=success, db opened
 *         1=skipped, db not opened because of no create
 *        -1=failure, db not opened because of db failure
 */
static int _nds2_open_db(nds2_connection *connection, const char *db_dir, const int create)
{
	int ret = -1;
	char *db_path = NULL;
	int sqlite_mode;
	int result;
	/* FIXME: hack for sqlite versions < 3.7.11.  should be
	 * removed when we can depend on newer sqlite versions */
	int readonly = 0;

	/* db path, e.g. */
	/* "DB_DIR/0.10_nds.ligo.caltech.edu_31200.sqlite". */
	if (asprintf(&db_path,
		     "%s/" PACKAGE_VERSION "_%s_%d.sqlite",
		     db_dir, connection->host, connection->port) < 0)
		goto done;


	if (access(db_path, F_OK) == 0) {
		if (access(db_path, W_OK) == 0) {
			sqlite_mode = SQLITE_OPEN_READWRITE | SQLITE_OPEN_SHAREDCACHE;
		} else {
			sqlite_mode = SQLITE_OPEN_READONLY;
			readonly = 1;
		}
	} else {
		if (!create) {
			ret = 1;
			goto done;
		}
		sqlite_mode = SQLITE_OPEN_CREATE | SQLITE_OPEN_READWRITE | SQLITE_OPEN_SHAREDCACHE;
	}

	if (sqlite3_open_v2(db_path, &connection->db, sqlite_mode, NULL) != SQLITE_OK) {
		goto done;
	}

	connection->_db_readonly = readonly;

	/* Register bash_pattern_matches as a custom function. */
	result = _bash_pattern_matches_register_func(connection->db);
	if (result != SQLITE_OK)
	{
		errno = EIO;
		goto done;
	}

	ret = 0;
 done:
	free(db_path);
	return ret;
}

/* FIXME: use the following once we can depend on sqlite > 3.7.11 */
/**
 * Return readonly status of db: 0=read/write, 1=readonly
 *
static int _nds2_db_is_readonly(sqlite3 *db)
{
	return sqlite3_db_readonly(db, NULL);
}
*/


/**
 * Find and open channel database object.  Various user-defined and
 * system paths are searched.
 * return: see _nds2_open_db() above
 */
static int _nds2_find_db(nds2_connection *connection, const int create)
{
	int ret = -1;
	char *db_dir = NULL;

#if _WIN32
	char base_path[MAX_PATH];

	/* Locate, and create if necessary, the directory
	 * "C:\Documents and Settings\username\Local Settings\Application Data". */
	if (!SUCCEEDED(SHGetFolderPathA(NULL, CSIDL_LOCAL_APPDATA | CSIDL_FLAG_CREATE, NULL, 0, base_path)))
		goto done;

	/* Form path to database: e.g.
	 * "C:\Documents and Settings\username\Local Settings\Application Data\nds2-client_0.10_nds.ligo-la.caltech.edu_31200.sqlite". */
	if (asprintf(&db_dir, "%s\\", base_path) < 0)
		goto done;

	ret = _nds2_open_db(connection, db_dir, create);
#else
	/* if CHANNEL_DB_DIR is explicitly specified, look for
	 * database there, and fail if we don't find it */
	char *env_dir = getenv("NDS2_CHANNEL_DB_DIR");

	if (env_dir) {

		if (asprintf(&db_dir, "%s", env_dir) < 0)
			goto done;

		ret = _nds2_open_db(connection, db_dir, create);

	/* otherwise look in system paths */
	} else {

		if (asprintf(&db_dir,
			     "/var/cache/" PACKAGE_NAME) < 0)
			goto done;

		/* try to open db at system path.  if opened, return.
		 * otherwise, try the user-specific /var/tmp dir.
		 */
		if (_nds2_open_db(connection, db_dir, create) == 0) {
			ret = 0;
			goto done;
		}

		/* look up ID of current user. */
		uid_t uid = getuid();

		/* user-specific db dir */
		if (asprintf(&db_dir,
			     "/var/tmp/" PACKAGE_NAME "_%lu",
			     (long unsigned)uid) < 0)
			goto done;

		/* make user db dir */
		if (mkdir(db_dir, 00755) == -1 && errno != EEXIST)
			goto done;

		/* try opening the db */
		ret = _nds2_open_db(connection, db_dir, create);
	}
#endif
 done:
	free(db_dir);
	return ret;
}


/**
 * Attempt to determine protocol version by database lookup.
 * return: 0=success, -1=failure
 */
static nds2_protocol _nds2_db_get_protocol(sqlite3 *db)
{
	nds2_protocol ret = -1;
	int result;
	sqlite3_stmt *stmt = NULL;

	result = sqlite3_prepare_v2(db,
		"SELECT protocol FROM servers", -1, &stmt, NULL);
	if (result != SQLITE_OK)
		goto fail;

	result = sqlite3_step(stmt);
	if (result != SQLITE_ROW)
		goto fail;

	ret = sqlite3_column_int(stmt, 0);

fail:
	sqlite3_finalize(stmt);
	return ret;
}


/**
 * Record protocol version in database.
 * return: 0=success, -1=failure
 */
static int _nds2_db_put_protocol(sqlite3 *db, nds2_protocol protocol)
{
	int result;
	int old_protocol;
	sqlite3_stmt *stmt;

	result = sqlite3_exec(db, "BEGIN", NULL, NULL, NULL);
	if (result != SQLITE_OK)
		goto fail;

	result = sqlite3_exec(db, _nds2_query_create_tables, NULL, NULL, NULL);
	if (result != SQLITE_OK)
		goto rollback;

	result = sqlite3_prepare_v2(db,
		"SELECT protocol FROM servers",
		-1, &stmt, NULL);
	if (result == SQLITE_OK)
		result = sqlite3_step(stmt);
	if (result == SQLITE_ROW)
		old_protocol = sqlite3_column_int(stmt, 0);
	sqlite3_finalize(stmt);

	/* If we were able to find the old protocol in the database, and it matches
	 * the new protocol, then stop here. */
	if (result == SQLITE_ROW && protocol == old_protocol)
		goto done;

	result = sqlite3_exec(db, "DELETE FROM servers", NULL, NULL, NULL);
	if (result != SQLITE_OK)
		goto rollback;

	result = sqlite3_prepare_v2(db,
		"INSERT INTO servers (protocol) VALUES (?)",
		-1, &stmt, NULL);
	if (result != SQLITE_OK)
		goto rollback;
	result = sqlite3_bind_int(stmt, 1, protocol);
	if (result == SQLITE_OK)
		result = sqlite3_step(stmt);

	sqlite3_finalize(stmt);

	if (result != SQLITE_DONE)
		goto rollback;

done:
	sqlite3_exec(db, "COMMIT", NULL, NULL, NULL);
	return 0;
rollback:
	sqlite3_exec(db, "ROLLBACK", NULL, NULL, NULL);
fail:
	errno = EIO;
	return -1;
}


static int strjoin(char **dest, const char *next, const char *sep)
{
	char *new_dest;
	size_t new_len = strlen(next);

	if (*dest)
	{
		size_t dest_len = strlen(*dest);
		size_t sep_len = strlen(sep);
		size_t new_dest_len = dest_len + new_len + sep_len;
		new_dest = realloc(*dest, new_dest_len + 1);
		if (!new_dest) {
			free(*dest);
			*dest = NULL;
			return -1;
		}
		memcpy(new_dest + dest_len, sep, sep_len);
		memcpy(new_dest + dest_len + sep_len, next, new_len);
		new_dest[new_dest_len] = '\0';
	} else {
		new_dest = malloc(new_len + 1);
		if (!new_dest) {
			return -1;
		}
		memcpy(new_dest, next, new_len + 1);
	}

	*dest = new_dest;
	return 0;
}


static int endswith(const char *str, const char *end)
{
	const size_t str_len = strlen(str);
	const size_t end_len = strlen(end);

	return end_len <= str_len
		&& memcmp(str + (str_len - end_len), end, end_len) == 0;
}


char *nds2_data_type_to_string(nds2_data_type data_type)
{
	static const char *sep = " | ";
	char *ret = NULL;

	if (data_type & NDS2_DATA_TYPE_INT16)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_INT16)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_INT32)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_INT32)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_INT64)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_INT64)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_FLOAT32)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_FLOAT32)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_FLOAT64)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_FLOAT64)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_COMPLEX32)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_COMPLEX32)), sep))
			return ret;

	if (!ret)
		ret = strdup(data_type_name(0));

	return ret;
}


char *nds2_channel_type_to_string(nds2_channel_type channel_type)
{
	static const char *sep = " | ";
	char *ret = NULL;

	if (channel_type & NDS2_CHANNEL_TYPE_ONLINE)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_ONLINE)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_RAW)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_RAW)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_RDS)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_RDS)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_STREND)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_STREND)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_MTREND)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_MTREND)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_TEST_POINT)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_TEST_POINT)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_STATIC)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_STATIC)), sep))
			return ret;

	if (!ret)
		ret = strdup(cvt_chantype_str(0));

	return ret;
}


/**
 * Initialize NDS client library.
 * FIXME: this static initialization is not entirely thread safe.
 * return: 0=success, -1=failure
*/
static int _nds2_startup()
{
	static int started = 0;
	int ret;
	if (started)
	{
		ret = 0;
	} else {
		int result = daq_startup();
		if (result == DAQD_OK)
		{
			ret = 0;
			started = 1;
		} else {
			ret = -1;
			errno = NDS2_ELAST + result;
		}
	}
	return ret;
}


nds2_channel *nds2_channel_alloc(
	const char *name, int channel_type, int data_type, double sample_rate,
	float signal_gain, float signal_slope, float signal_offset,
	const char *signal_units)
{
	nds2_channel *ret;

	if (strlen(name) + 1 > sizeof(ret->name) ||
		strlen(signal_units) + 1 > sizeof(ret->signal_units))
	{
		errno = NDS2_CHANNEL_NAME_TOO_LONG;
		return NULL;
	}

	ret = (nds2_channel *)calloc((size_t)1, sizeof(nds2_channel));
	if (ret)
	{
		strcpy(ret->name, name);
		ret->channel_type = (nds2_channel_type)channel_type;
		ret->data_type = (nds2_data_type)data_type;
		ret->sample_rate = sample_rate;
		ret->signal_gain = signal_gain;
		ret->signal_slope = signal_slope;
		ret->signal_offset = signal_offset;
		strcpy(ret->signal_units, signal_units);
	}

	return ret;
}


nds2_connection *nds2_open(const char *host, int port, nds2_protocol protocol)
{
	int result;
	nds2_connection *connection = NULL;

	if (_nds2_startup())
		goto fail;

	if (protocol <= NDS2_PROTOCOL_INVALID || protocol > NDS2_PROTOCOL_TRY)
	{
		errno = EINVAL;
		goto fail;
	}

	connection = calloc((size_t)1, sizeof(struct _nds2_connection));
	if (!connection)
		goto fail;

	connection->host = strdup(host);
	if (!connection->host)
		goto fail;

	/* Populate fields. */
	connection->port = port;
	connection->request_in_progress = 0;
	connection->requested_end_time = 0;
	connection->connected = 1;

	/* Open the connection. */
	if (protocol == NDS2_PROTOCOL_TRY)
		result = daq_connect(&connection->daq, host, port, nds_try);
	else
		result = daq_connect(&connection->daq, host, port, protocol);

	if (result != DAQD_OK)
	{
		errno = NDS2_ELAST + result;
		goto fail;
	}

	goto done;
fail:
	/* nds2_close calls daq_disconnect, and daq_disconnect may set errno.
	 * Save errno before calling nds2_close. */
	result = errno;
	nds2_close(connection);
	errno = result;
	connection = NULL;
done:
	return connection;
}


void nds2_shutdown(nds2_connection *connection)
{
	if (connection)
	{
		if (connection->db)
		{
			sqlite3_close(connection->db);
			connection->db = NULL;
		}
		daq_disconnect(&connection->daq);
		free(connection->host);
		connection->host = NULL;
		connection->connected = 0;
	} else
		errno = EFAULT;
}


void nds2_close(nds2_connection *connection)
{
	if (connection)
	{
		nds2_shutdown(connection);
		free(connection);
	} else
		errno = EFAULT;
}


char *nds2_get_host(nds2_connection *connection)
{
	char *ret = NULL;

	if (!connection)
		errno = EFAULT;
	else if (!connection->connected)
		errno = NDS2_ALREADY_CLOSED;
	else
		ret = strdup(connection->host);

	return ret;
}


int nds2_get_port(nds2_connection *connection)
{
	int ret;

	if (!connection)
		ret = -(errno = EFAULT);
	else if (!connection->connected)
		ret = -(errno = NDS2_ALREADY_CLOSED);
	else
		ret = connection->port;

	return ret;
}


nds2_protocol nds2_get_protocol(nds2_connection *connection)
{
	int ret;

	if (!connection)
		ret = -(errno = EFAULT);
	else if (!connection->connected)
		ret = -(errno = NDS2_ALREADY_CLOSED);
	else {
		ret = connection->daq.nds_versn;
		if (ret == nds_try)
			ret = NDS2_PROTOCOL_TRY;
	}

	return ret;
}


/**
 * Check if the cache needs to be updated. If it does, return
 * the new channel list hash from the server (or a dummy value
 * for NDS1 connections, which don't have a channel list hash
 * feature). If the cache does not need to be updated, return NULL.
 */
static void *_nds2_db_get_hash(nds2_connection *connection, int *hashsize, time_t *t)
{
	const void *hash = NULL;
	void *received_hash = NULL;
	int hash_size = 0, received_hash_size = 0;
	sqlite3_stmt *stmt = NULL;
	time(t);

	if (sqlite3_prepare_v2(connection->db, "SELECT timestamp, hash FROM servers", -1, &stmt, NULL) == SQLITE_OK
		&& sqlite3_step(stmt) == SQLITE_ROW)
	{
		time_t timestamp = sqlite3_column_int64(stmt, 0);
		hash = sqlite3_column_blob(stmt, 1);
		hash_size = sqlite3_column_bytes(stmt, 1);

		if (timestamp + 86400 >= *t)
			goto done; /* Cache has not yet expired. Just return. */
	}

	received_hash_size = 255;
	received_hash = malloc((size_t)received_hash_size);
	/* Returning NULL if this malloc failed would mean that the
	 * cache does NOT need to be updated. If we can't malloc()
	 * a measly 255 bytes, we might as well die. */
	if (!received_hash) abort();
	memset(received_hash, '?', (size_t)received_hash_size);

	if (nds2_get_protocol(connection) == NDS2_PROTOCOL_TWO
		&& !daq_recv_channel_hash(&connection->daq, received_hash, &received_hash_size, (time_t)0, cUnknown)
		&& received_hash_size == hash_size
		&& !memcmp(hash, received_hash, (size_t)hash_size))
	{
		/* Hashes match. Prepare to return NULL. */
		sqlite3_finalize(stmt);
		free(received_hash);
		received_hash = NULL;

		/* Record new timestamp; we can wait another 24 hours. */
		/* FIXME: this needs to be moved out */
		sqlite3_prepare_v2(connection->db,
			"UPDATE servers SET timestamp = ?", -1, &stmt, NULL)
			|| sqlite3_bind_int64(stmt, 1, *t)
			|| sqlite3_step(stmt);
	}

done:
	sqlite3_finalize(stmt);
	*hashsize = received_hash_size;
	return received_hash;
}


static nds2_channel **_nds2_db_find_channels(
	nds2_connection *connection,
	size_t *count_channels,
	const char *channel_glob,
	nds2_channel_type channel_type_mask,
	nds2_data_type data_type_mask,
	double min_sample_rate,
	double max_sample_rate,
	int use_glob
)
{
	nds2_channel **ret = NULL;
	nds2_channel **channels = malloc(sizeof(nds2_channel *));
	size_t count_channels_found = 0, count_channels_allocated = 1;
	int retval;
	sqlite3_stmt *stmt = NULL;
	int pattern_is_flat;
	char *sql = NULL;
	const char *name_cond;
	static const char channel_cond[] = "_channel_type & ?2 AND _data_type & ?3 AND _sample_rate BETWEEN ?4 AND ?5";

	if (!connection->db)
		goto fail;

	if (!channels)
		goto fail;

	if (!channel_glob)
		channel_glob = "*";

	/* Determine if the channel name pattern is flat (has only literals and
	 * wildcards, but no alternatives. */
	{
		bash_pattern *pattern = bash_pattern_compile(channel_glob);
		if (!pattern)
			goto fail;
		pattern_is_flat = bash_pattern_is_flat(pattern);
		bash_pattern_free(pattern);
	}

	/* If the channel name pattern is flat, then compare channel names using
	 * GLOB instead of bash_pattern_matches. GLOB is faster than
	 * bash_pattern_matches because it can use SQL column indices. */


	if (!use_glob)
		name_cond = "_name = ?1";
	else if (pattern_is_flat)
		name_cond = "_name GLOB ?1";
	else
		name_cond = "bash_pattern_matches(?1, _name)";

	if (nds2_get_protocol(connection) == NDS2_PROTOCOL_ONE)
	{
		if (strjoin(&sql, "SELECT * FROM", " ")) goto fail;
		if (strjoin(&sql, "SELECT name AS _name, channel_type AS _channel_type, data_type AS _data_type, sample_rate AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " (")) goto fail;
		if (strjoin(&sql, name_cond, " ")) goto fail;
		if (strjoin(&sql, channel_cond, " AND ")) goto fail;
		if (channel_type_mask & NDS2_CHANNEL_TYPE_STREND && 1 >= min_sample_rate && 1 <= max_sample_rate)
		{
			if (data_type_mask & NDS2_DATA_TYPE_FLOAT64)
			{
				if (strjoin(&sql, "SELECT name || '.mean,s-trend' AS _name, 8 AS _channel_type, 16 AS _data_type, 1 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
				if (strjoin(&sql, "SELECT name || '.rms,s-trend' AS _name, 8 AS _channel_type, 16 AS _data_type, 1 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
			}
			if (data_type_mask & (NDS2_DATA_TYPE_INT32 | NDS2_DATA_TYPE_FLOAT32))
			{
				if (strjoin(&sql, "SELECT name || '.min,s-trend' AS _name, 8 AS _channel_type, (CASE WHEN data_type == 32 THEN 8 ELSE data_type END) AS _data_type, 1 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
				if (strjoin(&sql, "data_type & ?3", " and ")) goto fail;
				if (strjoin(&sql, "SELECT name || '.max,s-trend' AS _name, 8 AS _channel_type, (CASE WHEN data_type == 32 THEN 8 ELSE data_type END) AS _data_type, 1 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
				if (strjoin(&sql, "data_type & ?3", " and ")) goto fail;
			}
			if (data_type_mask & NDS2_DATA_TYPE_INT32)
			{
				if (strjoin(&sql, "SELECT name || '.n,s-trend' AS _name, 8 AS _channel_type, 2 AS _data_type, 1 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
				if (strjoin(&sql, "data_type & ?3", " and ")) goto fail;
			}
		}
		if (channel_type_mask & NDS2_CHANNEL_TYPE_MTREND && 1.0/60 >= min_sample_rate && 1.0/60 <= max_sample_rate)
		{
			if (data_type_mask & NDS2_DATA_TYPE_FLOAT64)
			{
				if (strjoin(&sql, "SELECT name || '.mean,m-trend' AS _name, 16 AS _channel_type, 16 AS _data_type, 1.0/60 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
				if (strjoin(&sql, "SELECT name || '.rms,m-trend' AS _name, 16 AS _channel_type, 16 AS _data_type, 1.0/60 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
			}
			if (data_type_mask & (NDS2_DATA_TYPE_INT32 | NDS2_DATA_TYPE_FLOAT32))
			{
				if (strjoin(&sql, "SELECT name || '.min,m-trend' AS _name, 16 AS _channel_type, (CASE WHEN data_type == 32 THEN 8 ELSE data_type END) AS _data_type, 1.0/60 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
				if (strjoin(&sql, "data_type & ?3", " and ")) goto fail;
				if (strjoin(&sql, "SELECT name || '.max,m-trend' AS _name, 16 AS _channel_type, (CASE WHEN data_type == 32 THEN 8 ELSE data_type END) AS _data_type, 1.0/60 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
				if (strjoin(&sql, "data_type & ?3", " and ")) goto fail;
			}
			if (data_type_mask & NDS2_DATA_TYPE_INT32)
			{
				if (strjoin(&sql, "SELECT name || '.n,m-trend' AS _name, 16 AS _channel_type, 2 AS _data_type, 1.0/60 AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " UNION ALL ")) goto fail;
				if (strjoin(&sql, name_cond, " ")) goto fail;
				if (strjoin(&sql, "data_type & ?3", " and ")) goto fail;
			}
		}
		if (strjoin(&sql, "ORDER BY _name", ") ")) goto fail;
	} else /* (nds2_get_protocol(connection) == NDS2_PROTOCOL_TWO) */ {
		if (strjoin(&sql, "SELECT name AS _name, channel_type AS _channel_type, data_type AS _data_type, sample_rate AS _sample_rate, signal_gain, signal_slope, signal_offset, signal_units FROM channels WHERE", " ")) goto fail;
		if (strjoin(&sql, name_cond, " ")) goto fail;
		if (strjoin(&sql, channel_cond, " AND ")) goto fail;
		if (strjoin(&sql, "ORDER BY _name", " ")) goto fail;
	}

	retval = _nds2_db_create_tables(connection->db)
	|| sqlite3_prepare_v2(connection->db, sql, -1, &stmt, NULL)
	|| sqlite3_bind_text(stmt, 1, channel_glob, -1, SQLITE_STATIC)
	|| sqlite3_bind_int(stmt, 2, (int) channel_type_mask)
	|| sqlite3_bind_int(stmt, 3, (int) data_type_mask)
	|| sqlite3_bind_double(stmt, 4, min_sample_rate)
	|| sqlite3_bind_double(stmt, 5, max_sample_rate);
	free(sql);

	if (retval != SQLITE_OK)
		goto fail;

	while ((retval = sqlite3_step(stmt)) == SQLITE_ROW)
	{
		nds2_channel *channel;

		if (count_channels_found >= count_channels_allocated)
		{
			nds2_channel **new_channels;
			count_channels_allocated <<= 1;
			new_channels = realloc(channels, sizeof(nds2_channel *) * count_channels_allocated);
			if (!new_channels)
				goto fail;
			channels = new_channels;
		}

		channel = nds2_channel_alloc(
			(const char*)sqlite3_column_text(stmt, 0),
			sqlite3_column_int(stmt, 1),
			sqlite3_column_int(stmt, 2),
			sqlite3_column_double(stmt, 3),
			(float)sqlite3_column_double(stmt, 4),
			(float)sqlite3_column_double(stmt, 5),
			(float)sqlite3_column_double(stmt, 6),
			(const char*)sqlite3_column_text(stmt, 7));

		if (!channel)
			goto fail;

		channels[count_channels_found] = channel;
		count_channels_found ++;
	}

	if (retval != SQLITE_DONE)
		goto fail;

	if (count_channels_found > 0)
	{
		ret = realloc(channels, sizeof(nds2_channel *) * count_channels_found);
		if (!ret)
			goto fail;
	}
	channels = NULL;
	*count_channels = count_channels_found;
fail:
	if (channels)
	{
		size_t i;
		for (i = 0; i < count_channels_found; i ++)
			free(channels[i]);
		free(channels);
		channels = NULL;
	}
	sqlite3_finalize(stmt);
	return ret;
}


/**
 * return: 0=success, -1=failure
 */
static int _nds2_db_put_channels(
	nds2_connection *connection,
	daq_channel_t *channels,
	size_t count_channels,
	const time_t *t,
	const void *hash,
	int hash_size
)
{
	int result;
	size_t i;
	sqlite3_stmt *stmt;
	char channel_name_buf[sizeof(((daq_channel_t *)NULL)->name) + 16];

	if (!connection->db)
		goto done;

	result = _nds2_db_start_transaction(connection->db);
	if (result != SQLITE_OK)
		goto commit;

	/* Wipe cache */
	result = sqlite3_exec(connection->db, "DELETE FROM servers; DELETE FROM channels;", NULL, NULL, NULL);
	if (result != SQLITE_OK)
		goto rollback;

	/* Write server metadata (protocol and timestamp) */
	result = sqlite3_prepare_v2(connection->db,
		"INSERT INTO servers (protocol, timestamp, hash) VALUES (?, ?, ?)", -1, &stmt, NULL)
	|| sqlite3_bind_int(stmt, 1, nds2_get_protocol(connection))
	|| sqlite3_bind_int64(stmt, 2, *t)
	|| sqlite3_bind_blob(stmt, 3, hash, hash_size, SQLITE_STATIC);
	if (result == SQLITE_OK)
		result = sqlite3_step(stmt);
	sqlite3_finalize(stmt);
	if (result != SQLITE_DONE)
		goto rollback;

	/* Write channel list */
	result = sqlite3_prepare_v2(connection->db,
		"INSERT INTO channels VALUES (?, ?, ?, ?, ?, ?, ?, ?)", -1, &stmt, NULL);
	if (result != SQLITE_OK)
	{
		sqlite3_finalize(stmt);
		goto rollback;
	}
	for (i = 0; i < count_channels; i ++)
	{
		nds2_channel_type channel_type = _nds2_chantype_to_channel_type(channels[i].type);
		nds2_data_type data_type = _nds2_data_to_data_type(channels[i].data_type);
		char *channel_name;

		switch (channel_type)
		{
			case NDS2_CHANNEL_TYPE_RDS:
				if (snprintf(channel_name_buf, sizeof(channel_name_buf), "%s,rds", channels[i].name) < 0)
				{
					sqlite3_finalize(stmt);
					goto rollback;
				}
				channel_name = channel_name_buf;
				break;
			case NDS2_CHANNEL_TYPE_STREND:
				if (snprintf(channel_name_buf, sizeof(channel_name_buf), "%s,s-trend", channels[i].name) < 0)
				{
					sqlite3_finalize(stmt);
					goto rollback;
				}
				channel_name = channel_name_buf;
				break;
			case NDS2_CHANNEL_TYPE_MTREND:
				if (snprintf(channel_name_buf, sizeof(channel_name_buf), "%s,m-trend", channels[i].name) < 0)
				{
					sqlite3_finalize(stmt);
					goto rollback;
				}
				channel_name = channel_name_buf;
				break;
			default:
				channel_name = channels[i].name;
		}
		result = sqlite3_bind_text(stmt, 1, channel_name, -1, SQLITE_STATIC)
		|| sqlite3_bind_int(stmt, 2, (int) channel_type)
		|| sqlite3_bind_int(stmt, 3, (int) data_type)
		|| sqlite3_bind_double(stmt, 4, channels[i].rate)
		|| sqlite3_bind_double(stmt, 5, channels[i].s.signal_gain)
		|| sqlite3_bind_double(stmt, 6, channels[i].s.signal_slope)
		|| sqlite3_bind_double(stmt, 7, channels[i].s.signal_offset)
		|| sqlite3_bind_text(stmt, 8, channels[i].s.signal_units, -1, SQLITE_STATIC);
		if (result == SQLITE_OK)
			result = sqlite3_step(stmt);
		if (result != SQLITE_DONE)
		{
			sqlite3_finalize(stmt);
			goto rollback;
		}
		result = sqlite3_reset(stmt);
		if (result != SQLITE_OK)
		{
			sqlite3_finalize(stmt);
			goto rollback;
		}
	}
	sqlite3_finalize(stmt);

commit:
	_nds2_db_end_transaction(connection->db);
done:
	return 0;
rollback:
	sqlite3_exec(connection->db, "ROLLBACK", NULL, NULL, NULL);
	return -1;
}

/**
 * update channel cache database
 * return: 0=success, -1=failure
 */
static int _nds2_update_db(nds2_connection *connection)
{
	void *hash;
	int hash_size;
	time_t t;
	int retval;

	if (!connection) {
		errno = EFAULT;
		goto fail;
	} else if (!connection->connected) {
		errno = NDS2_ALREADY_CLOSED;
		goto fail;
	}

	/* Fail if there is another transfer already in progress. */
	if (connection->request_in_progress) {
		errno = NDS2_TRANSFER_BUSY;
		goto fail;
	}

	/* If database doesn't have a record of the protocol, record it */
	if (_nds2_db_get_protocol(connection->db) <= NDS2_PROTOCOL_INVALID) {
		if (_nds2_db_put_protocol(connection->db, nds2_get_protocol(connection)))
			goto fail;
	}

	/* Check if database is up to date. */
	hash = _nds2_db_get_hash(connection, &hash_size, &t);

	/* If hash database is not up to date, fetch channel list. */
	if (hash) {
		daq_channel_t *daq_channels;
		int nchannels;

		/* Error reading from cache database or cache database out of date, so
		 * retrieve channel list from server. */
		retval = daq_recv_channels(&connection->daq, NULL, 0, &nchannels);
		if (retval) {
			free(hash);
			errno = NDS2_ELAST + retval;
			goto fail;
		}

		daq_channels = malloc(sizeof(daq_channel_t) * nchannels);
		if (!daq_channels) {
			free(hash);
			goto fail;
		}

		retval = daq_recv_channels(&connection->daq, daq_channels,
			nchannels, &nchannels);
		if (retval) {
			free(hash);
			free(daq_channels);
			errno = NDS2_ELAST + retval;
			goto fail;
		}

		if (_nds2_db_put_channels(connection, daq_channels, (size_t) nchannels, &t, hash, hash_size)) {
			free(hash);
			free(daq_channels);
			errno = EIO;
			goto fail;
		}
		free(hash);
		free(daq_channels);
	}
	return 0;
 fail:
	return -1;
}


static nds2_channel **_nds2_find_channels(
	nds2_connection *connection,
	size_t *count_channels,
	const char *channel_glob,
	nds2_channel_type channel_type_mask,
	nds2_data_type data_type_mask,
	double min_sample_rate,
	double max_sample_rate,
	int use_glob
)
{
	nds2_channel **ret;
	size_t count_channels_found = 1;

	if (!connection) {
		errno = EFAULT;
		return NULL;
	} else if (!connection->connected) {
		errno = NDS2_ALREADY_CLOSED;
		return NULL;
	} else if (!count_channels) {
		errno = EFAULT;
		return NULL;
	}

	/* Open the database if it hasn't been opened yet (and create
	 * if it hasn't been created */
	if (!connection->db) {
		if (_nds2_find_db(connection, 1))
			return NULL;
	}

	/* Update the database if it hasn't been opened readonly */
        /* FIXME: use the following once we can depend on sqlite > 3.7.11
	if (! _nds2_db_is_readonly(connection->db)) {
	*/
	if (connection->db && !connection->_db_readonly) {
		if (_nds2_update_db(connection) < 0)
			return NULL;
	}

	/* Query database. */
	ret = _nds2_db_find_channels(connection, &count_channels_found, channel_glob, channel_type_mask, data_type_mask, min_sample_rate, max_sample_rate, use_glob);
	if (ret || !count_channels_found) {
		*count_channels = count_channels_found;
		return ret;
	} else {
		/* Database query failed. */
		errno = EIO;
		return NULL;
	}
}


nds2_channel **nds2_find_channels(
	nds2_connection *connection,
	size_t *count_channels,
	const char *channel_glob,
	nds2_channel_type channel_type_mask,
	nds2_data_type data_type_mask,
	double min_sample_rate,
	double max_sample_rate
)
{
	return _nds2_find_channels(connection, count_channels, channel_glob, channel_type_mask, data_type_mask, min_sample_rate, max_sample_rate, 1);
}


int nds2_sizeof_data_type(nds2_data_type data_type)
{
	int ret;

	switch (data_type) {
		case NDS2_DATA_TYPE_INT16:
			ret = 2;
			break;
		case NDS2_DATA_TYPE_INT32:
		case NDS2_DATA_TYPE_FLOAT32:
		case NDS2_DATA_TYPE_UINT32:
			ret = 4;
			break;
		case NDS2_DATA_TYPE_INT64:
		case NDS2_DATA_TYPE_FLOAT64:
		case NDS2_DATA_TYPE_COMPLEX32:
			ret = 8;
			break;
		default:
			ret = -EINVAL;
	}

	if (ret < 0)
		errno = -ret;
	return ret;
}


nds2_buffer **nds2_fetch(
	nds2_connection *connection,
	long gps_start,
	long gps_stop,
	const char **channel_names,
	size_t count_channels
) {
	nds2_buffer **ret = NULL;
	unsigned char **ptrs = malloc(sizeof(char *) * count_channels);
	unsigned char **ends = malloc(sizeof(char *) * count_channels);
	int result;
	size_t count_received_channels;
	size_t i;

	if (!connection)
	{
		errno = EFAULT;
		goto fail;
	} else if (!connection->connected) {
		errno = NDS2_ALREADY_CLOSED;
		goto fail;
	}

	/* Check that all temporary arrays were allocated. */
	if (!ptrs || !ends)
		goto fail;

	/* Try to request data for indicated times and channels. */
	result = nds2_iterate(connection, gps_start, gps_stop, 0, channel_names,
		count_channels);
	if (result)
		goto fail;

	/* Try to retrieve the first set of buffers. */
	ret = nds2_next(connection, &count_received_channels);
	if (!ret)
		goto fail;

	/* Check that the correct number of buffers arrived. */
	if (count_channels != count_received_channels)
	{
		errno = NDS2_UNEXPECTED_CHANNELS_RECEIVED;
		goto fail;
	}

	/* Reallocate the received buffers to fit the entire request. */
	for (i = 0; i < count_channels; i ++)
	{
		nds2_buffer *new_buf;
		size_t unit_size = nds2_sizeof_data_type(ret[i]->channel.data_type);
		size_t needed_samples = (size_t)( (gps_stop - gps_start) * ret[i]->channel.sample_rate );
		size_t needed_bytes = needed_samples * unit_size;
		new_buf = realloc(ret[i], sizeof(nds2_buffer) + needed_bytes);
		if (!new_buf)
			goto fail;
		ret[i] = new_buf;
		ptrs[i] = ret[i]->data + ret[i]->length * unit_size;
		ends[i] = ret[i]->data + needed_bytes;
		ret[i]->length = needed_samples;
	}

	/* Retrieve more buffers as they arrive. */
	while (1) {
		chan_req_t *chan_reqs;
		char *block_data;

		/* Check to see if we need to retrieve any more data. */
		for (i = 0; i < count_channels; i ++)
			if (ptrs[i] < ends[i])
				break;
		if (i == count_channels)
			break;

		result = daq_recv_next(&connection->daq);
		if (result != DAQD_OK)
		{
			errno = NDS2_ELAST + result;
			goto fail;
		}

		/* Check that the correct number of buffers arrived. */
		if (count_channels != connection->daq.num_chan_request)
		{
			errno = NDS2_UNEXPECTED_CHANNELS_RECEIVED;
			goto fail;
		}

		/* Copy data buffer by buffer. */
		chan_reqs = connection->daq.chan_req_list;
		block_data = daq_get_block_data(&connection->daq);
		for (i = 0; i < count_channels; i ++)
		{
			/* Determine number of bytes received for this channel. */
			int nbytes = chan_reqs[i].status;
			size_t nbytes_to_copy;

			/* Check that no error condition occurred on this channel. */
			if (nbytes < 0)
			{
				errno = NDS2_ELAST - nbytes;
				goto fail;
			}

			/* Check that there is room to copy the new data. */
			if (ptrs[i] + nbytes > ends[i])
			{
				nbytes_to_copy = ends[i] - ptrs[i];
			} else {
				nbytes_to_copy = nbytes;
			}

			/* Copy the new data. */
			memcpy(ptrs[i], block_data + chan_reqs[i].offset, nbytes_to_copy);
			ptrs[i] += nbytes;
		}
	}

	/* NDS1 transfers end with a 'termination block', an empty block that
	 * is indistinguisable from a 'data not found' condition. If this is
	 * an NDS1 connection, we must digest the termination block. */
	if (connection->request_in_progress && connection->daq.nds_versn == nds_v1)
	{
		result = daq_recv_next(&connection->daq);
		if (result != DAQD_NOT_FOUND)
		{
			errno = NDS2_ELAST + result;
			goto fail;
		}
	}

	connection->request_in_progress = 0;

	/* No errors; jump to end! */
	goto done;
fail:
	/* Uh-oh, something wrong happened. */
	if (ret)
	{
		for (i = 0; i < count_channels; i ++)
			free(ret[i]);
		free(ret);
		ret = NULL;
	}
done:
	/* Done! */
	free(ptrs);
	free(ends);
	return ret;
}


int nds2_iterate(
	nds2_connection *connection,
	long gps_start,
	long gps_stop,
	long stride,
	const char **channel_names,
	size_t count_channels
)
{
	int ret = -1;
	int result;
	size_t i;
	int have_minute_trends = 0;
	double bps = 0;

	if (!connection)
	{
		errno = EFAULT;
		goto fail;
	} else if (!connection->connected) {
		errno = NDS2_ALREADY_CLOSED;
		goto fail;
	}

	/* Fail if there is another transfer already in progress. */
	if (connection->request_in_progress)
	{
		errno = NDS2_TRANSFER_BUSY;
		goto fail;
	}

	daq_clear_channel_list(&connection->daq);

	for (i = 0; i < count_channels; i ++)
	{
		daq_channel_t channel;
		chantype_t channel_type;
		double sample_rate;

		memset(&channel, 0, sizeof(channel));

		/* For NDS2, infer channel type and sample rate from
		 * name if possible (minute and second trends have
		 * special suffixes), otherwise fill in defaults.
		 */
		if (nds2_get_protocol(connection) == NDS2_PROTOCOL_TWO)
		{
			if (endswith(channel_names[i], ",m-trend"))
			{
				channel_type = cMTrend;
				sample_rate = 1.0 / 60;
				have_minute_trends = 1;
			}
			else if (endswith(channel_names[i], ",s-trend"))
			{
				channel_type = cSTrend;
				sample_rate = 1.0;
			}
			else
			{
				channel_type = cUnknown;
				sample_rate = 16384;
			}

			daq_init_channel(&channel,
					 channel_names[i],
					 channel_type,
					 sample_rate,
					 _undefined);

		/* For NDS1, query the server for the channel info. */
		} else
		{
			int num_received;

			result = daq_recv_channels_by_pattern(&connection->daq, &channel,
							      1, &num_received,
							      (time_t) gps_start, cUnknown,
							      channel_names[i]);
			if ((result != DAQD_OK) || (num_received != 1))
			{
				errno = NDS2_ELAST + result;
				goto fail;
			}
		}

		/* For the purpose of computing the byte rate, use the
		 * default channel bytes per sample (bps) and sample
		 * rate.
		 */
		bps += channel.bps * channel.rate;

		result = daq_request_channel_from_chanlist(&connection->daq, &channel);
		if (result != DAQD_OK)
		{
			errno = NDS2_ELAST + result;
			goto fail;
		}
	}

	/* NDS2 protocol does not support indefinite GPS stop time */
	if (nds2_get_protocol(connection) == NDS2_PROTOCOL_TWO && !gps_stop)
	{
		gps_stop = 1893024016; /* A safely huge value of January 1, 2040. */
	}

	if (stride == 0)
	{
		if (gps_start) /* offline request */
		{
			if (have_minute_trends) /* request includes minute trends */
			{
				/* start with smallest stride that is a multiple of 60 seconds and
				 * whose frame size in bytes is greater than or equal to
				 * _NDS2_TYPICAL_BYTES_PER_FRAME. */
				stride = (long)(60 * ceil(_NDS2_TYPICAL_BYTES_PER_FRAME / (60 * bps)));
			} else { /* request does not include minute trends */
				/* start with smallest stride that is a multiple of 1 second and
				 * whose frame size in bytes is greater than or equal to
				 * _NDS2_TYPICAL_BYTES_PER_FRAME. */
				stride = (long)(ceil(_NDS2_TYPICAL_BYTES_PER_FRAME / bps));
			}

			if (stride > gps_stop - gps_start) /* stride is longer than request duration */
			{
				stride = gps_stop - gps_start; /* set stride to request duration */
			}
		} else { /* online request */
			if (have_minute_trends) /* request includes minute trends */
				stride = 60; /* use shortest possible stride, 60 seconds */
			else
				stride = 1; /* use shortest possible stride, 1 second */
		}
	}

	/* Raise an error if any of the requested channels are minute-trends and
	 * the start or stop times are indivisible by 60. */
	if (have_minute_trends && (gps_start % 60 || gps_stop % 60 || stride % 60))
	{
		errno = NDS2_MINUTE_TRENDS;
		goto fail;
	}

	result = daq_request_data(&connection->daq, (time_t) gps_start, (time_t) gps_stop, stride);
	if (result != DAQD_OK)
	{
		errno = NDS2_ELAST + result;
		goto fail;
	}

	connection->requested_end_time = gps_stop;
	connection->request_in_progress = 1;

	ret = 0;
fail:
	return ret;
}


nds2_buffer **nds2_next(
	nds2_connection *connection,
	size_t *count_channels
)
{
	nds2_buffer **ret = NULL;
	size_t count_received_channels;
	chan_req_t *chan_reqs;
	size_t i;
	int result;

	if (!connection)
	{
		errno = EFAULT;
		goto done;
	} else if (!connection->connected) {
		errno = NDS2_ALREADY_CLOSED;
		goto done;
	}

	if (!connection->request_in_progress)
	{
		errno = NDS2_EOS;
		goto done;
	}

	result = daq_recv_next(&connection->daq);
	if (result != DAQD_OK)
	{
		errno = NDS2_ELAST + result;
		goto done;
	}

	count_received_channels = connection->daq.num_chan_request;

	ret = malloc(sizeof(nds2_buffer *) * count_received_channels);
	if (!ret)
		goto done;

	chan_reqs = connection->daq.chan_req_list;

	for (i = 0; i < count_received_channels; i ++)
	{
		int nbytes = chan_reqs[i].status;

		if (nbytes < 0)
		{
			errno = NDS2_ELAST - nbytes;
			break;
		}

		/* Allocate buffer, making room for data payload. */
		ret[i] = malloc(sizeof(nds2_buffer) + nbytes);
		if (!ret[i])
			break;

		/* Copy channel information. */
		result = _nds2_chan_req_to_channel(&ret[i]->channel, &chan_reqs[i]);
		if (result)
		{
			free(ret[i]);
			break;
		}

		/* Set buffer's metadata. */
		ret[i]->gps_seconds = daq_get_block_gps(&connection->daq);
		ret[i]->gps_nanoseconds = daq_get_block_gpsn(&connection->daq);
		ret[i]->length = nbytes / nds2_sizeof_data_type(ret[i]->channel.data_type);

		/* Copy buffer payload. */
		memcpy(ret[i]->data, daq_get_block_data(&connection->daq) + chan_reqs[i].offset, (size_t) nbytes);
	}

	if (i != count_received_channels)
	{
		size_t j;
		for (j = 0; j < i; j ++)
			free(ret[j]);
		free(ret);
		ret = NULL;
		goto done;
	}

	if (connection->requested_end_time != 0)
	{
		int has_next = 0;
		for (i = 0; i < count_received_channels; i ++)
		{
			if (ret[i]->gps_seconds < connection->requested_end_time
				&& (int)((connection->requested_end_time - ret[i]->gps_seconds)
				* ret[i]->channel.sample_rate) > ret[i]->length)
				has_next = 1;
		}
		if (!has_next)
		{
			connection->request_in_progress = 0;

			/* NDS1 transfers end with a 'termination block', an empty block that
			 * is indistinguisable from a 'data not found' condition. If this is
			 * an NDS1 connection, we must digest the termination block. */
			if (connection->daq.nds_versn == nds_v1)
			{
				result = daq_recv_next(&connection->daq);
				if (result != DAQD_NOT_FOUND)
				{
					errno = NDS2_ELAST + result;
					for (i = 0; i < count_received_channels; i ++)
						free(ret[i]);
					free(ret);
					ret = NULL;
					goto done;
				}
			}
		}
	}

	*count_channels = count_received_channels;
done:
	return ret;
}


int nds2_has_next(nds2_connection *connection)
{
	int ret = -1;

	if (!connection)
		errno = EFAULT;
	else if (!connection->connected)
		errno = NDS2_ALREADY_CLOSED;
	else
		ret = connection->request_in_progress;

	return ret;
}


int nds2_clear_cache(nds2_connection *connection)
{
	int ret = -1;
	int result;

	if (!connection)
	{
		errno = EFAULT;
		goto fail;
	} else if (!connection->connected) {
		errno = NDS2_ALREADY_CLOSED;
		goto fail;
	}

	result = sqlite3_exec(connection->db,
		"DROP INDEX IF EXISTS channel_name_index; DROP TABLE IF EXISTS servers; DROP TABLE IF EXISTS channels;", NULL, NULL, NULL);
	if (result != SQLITE_OK)
	{
		errno = EIO;
		goto fail;
	}

	ret = 0;
fail:
	return ret;
}


void nds2_perror(const char *s)
{
	fprintf(stderr, "%s: %s\n", s, nds2_strerror(errno));
}


const char *nds2_strerror(int errnum)
{
	const char *errstr;

	switch (errnum) {
		case NDS2_EOS:
			errstr = "Reached end of stream.";
			break;
		case NDS2_CHANNEL_NAME_TOO_LONG:
			errstr = "Channel name is too long.";
			break;
		case NDS2_UNEXPECTED_CHANNELS_RECEIVED:
			errstr = "Server sent more channels than were expected.";
			break;
		case NDS2_MINUTE_TRENDS:
			errstr = "One or more requested channels are minute-trends, but requested start and stop times are not multiples of 60 seconds.";
			break;
		case NDS2_TRANSFER_BUSY:
			errstr = "Another transfer is already in progress. Complete the transfer or retry on a new connection object.";
			break;
		case NDS2_NO_SUCH_CHANNEL:
			errstr = "No such channel.";
			break;
		case NDS2_ALREADY_CLOSED:
			errstr = "The connection was already closed.";
			break;
		default:
			if (errnum >= NDS2_ELAST)
				errstr = daq_strerror(errnum - NDS2_ELAST);
			else
				errstr = strerror(errnum);
	}

	return errstr;
}
