/* -*- mode: c; c-basic-offset: 2; -*- */
#if HAVE_CONFIG_H
#include "daq_config.h"
#endif /*HAVE_CONFIG_H */

#include <stdarg.h>
#include <stdio.h>

#include "daqc.h"
#include "nds_logging.h"
#include "nds_os.h"
#include "nds_mex_utils.h"

#define MAX_GROUP_LENGTH 1024

static void nds_logging_matlab_init_( );

void
mexFunction( int nlhs, mxArray *plhs[],
	     int nrhs, const mxArray *prhs[])
{
  char group_buf[ MAX_GROUP_LENGTH ];
  double value = -2;
  mxArray* retval = (mxArray*)NULL;

  /*** Input Validation ***/
  if(nlhs != 1)
  {
    mexErrMsgTxt("This function returns previous debugging level.");
  }
  
  if(nrhs != 2)
  {
    mexErrMsgTxt("This function requires two argument.");
  }

  nds_logging_matlab_init_( );

  if (mxGetString(prhs[0], group_buf, (mwSize)(sizeof(group_buf))))
  {
    mexErrMsgTxt("Group should be of the form \"group\" "
		 " and shorter than " STRINGIFY(MAX_GROUP_LENGTH) " characters.");
  }
  /* value */
  if((mxGetM(prhs[1]) * mxGetN(prhs[1])) != 1) {
    mexErrMsgTxt("Value must be a scalar.");
  }
  value = mxGetScalar(prhs[1]);

  {
    int group = nds_logging_group_from_string( group_buf );

    {
      char warning[1024];

      SNPRINTF( warning, sizeof( warning ),
		"group: %s(%d) - level: %g", group_buf, group, value );
      mexWarnMsgTxt( warning );
    }

    if ( group == NDS_LOG_BAD_GROUP )
    {
      char warning[ 1024 ];

      SNPRINTF( warning, sizeof( warning ),
		"Unknown debugging group: %s", group_buf );
      mexWarnMsgTxt( warning );
      value = -2;
    }
    else
    {
      value = nds_logging_debug_level( group, (int)(value) );
      nds_logging_enable( group );
    }
  }
  {
      static const char* fields[] = { "debug_level" };

      mwSize dims = 1;

      retval = mxCreateStructArray( 1, &dims,
				    1, fields );
      put_mxarray_double( retval, 0, (mwSize)(0), value );
      plhs[ 0 ] = retval;
  }
}

static void
nds_flush_matlab( )
{
  mexEvalString("drawnow");
}

static void
nds_log_matlab( const char* Message )
{
#if 1
  mexPrintf( "%s", Message );
#else
  mexWarnMsgTxt(Message);
#endif
}


void
NDS_LOGGING_HELPER_ENTRY( )
{
  nds_log_matlab( "NDS_LOGGING_HELPER_ENTRY( ): Entry\n" );
  nds_function_flush( nds_flush_matlab );
  nds_function_logging( nds_log_matlab );
  nds_log_matlab( "NDS_LOGGING_HELPER_ENTRY( ): Exit\n" );
}

static void
nds_logging_matlab_init_( )
{
  static int initialized = 0;

  mexWarnMsgTxt( "nds_logging_matlab_init_: Entry" );
  if ( ! initialized )
  {
    mexWarnMsgTxt( "Initializing" );
    NDS_LOGGING_HELPER_ENTRY( );
    nds_logging_flush( );
    nds_logging_init( );
    nds_logging_flush( );
    NDS_LOGGING_HELPER_ENTRY( );
    nds_logging_flush( );
    initialized = 1;
  }
  mexWarnMsgTxt( "nds_logging_matlab_init_: Exit" );
}
