/* -*- mode: c; c-basic-offset: 4; -*- */
/*******************************************************
  Matlab mex wrapper for NDS.
******************************************************/

#if HAVE_CONFIG_H
#include "daq_config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include "daqc.h"
#include "nds_os.h"
#include "nds_log_matlab.h"
#include "nds_logging.h"
#include "nds_mex_utils.h"

mxArray*
mlwrapper_chanlist_to_mxarray(int *err,daq_channel_t* channels,int num_channels);
int
mlwrapper_get_channel_list(mxArray **,char*);

/*   NDS_GetChannels function */
/*   ------------------------ */
/*   This function returns a list of available channels */
/*   The syntax is:                                     */
/*   NDS_GetChannels("server:port");                    */
void 
mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) {
    char hpbuf[MAX_HOST_LENGTH];
    int err;
	
    /*** Input Validation ***/
    if(nlhs != 1) {
	mexErrMsgTxt("This function only returns one paramter.");
    }
	
    if(nrhs != 1) {
	mexErrMsgTxt("This function requires only one argument.");
    }

    /* to make input string an actual C string */
    if (mxGetString(prhs[0], hpbuf, (mwSize)(sizeof(hpbuf))))
        mexErrMsgTxt("Hostname should be of the form \"host\" or \"host:port\""
                     " and shorter than " STRINGIFY(MAX_HOST_LENGTH) " characters.");

    err = mlwrapper_get_channel_list(plhs, hpbuf);

    if(err) {
	mexErrMsgTxt("Failed to get channel list.");
    } 

    return;
}

int
mlwrapper_get_channel_list(mxArray **plhs, char* host) {
    /* Local variable declaration */
    short port = 31200;
    daq_t daq;
    int err;
    int num_alloc    = 0;
    daq_channel_t* channels = NULL;
    int num_channels = 0;


    /* Initialization processing */
    daq_startup();

    /* Get the host ip and port number */
    parse_host_name(host, &port);

    /* Open up a client connection */
    err = daq_connect(&daq, host, port, nds_try);
    /* printf("daq_connect: return code %i\n", err); */
    if (err) {
	mexWarnMsgTxt("Failed to connect to nds.");
	return 13;
    }

    /*  Get the channel list */
    err = daq_recv_channels (&daq, channels, 0, &num_alloc);
    if (err || num_alloc < 1) {
	daq_recv_shutdown(&daq);
	mexWarnMsgTxt("get_channel_list() failed.");
	return 54;
    }

    channels = mxCalloc((mwSize)(num_alloc), sizeof(daq_channel_t));
    err = daq_recv_channels (&daq, channels, num_alloc, &num_channels);
    daq_disconnect(&daq);
    daq_recv_shutdown(&daq);
    if (err || num_channels < 1) {
	mexWarnMsgTxt("get_channel_list() failed.");
	return 54;
    }
    plhs[0] = mlwrapper_chanlist_to_mxarray(&err,channels,num_channels);

    if (err) {
	mexWarnMsgTxt("mlwrapper_chanlist_to_mxarray() failed.");
	if (channels != NULL) {
	    mxFree(channels);
	}
	return 83;
    }
    if (channels != NULL) {
	mxFree(channels);
    }

    return 0;
}

mxArray*
mlwrapper_chanlist_to_mxarray(int *err,daq_channel_t* channels,int num_channels)
{
    /* Retrieves the channel list from the NDS server,
       and converts it into a mxArray of Matlab structs */
    mxArray *ml_list;
    mwSize dims[1];
    char* fields[] = {"name", "group_num", "rate", "tpnum", "bps",
		      "data_type", "signal_gain", "signal_offset",
		      "signal_slope", "signal_units"};

    if(num_channels == 0) {
	*err = 1;
	return NULL;
    }

    dims[0] = (mwSize) num_channels;

    ml_list = mxCreateStructArray(1,dims,10,(const char**)fields);
    if(ml_list == NULL) {
	*err = 2;
	return NULL;
    }

    {
	mwSize i;
	int field_name = mxGetFieldNumber(ml_list,"name");
	int field_group_num = mxGetFieldNumber(ml_list,"group_num");
	int field_rate = mxGetFieldNumber(ml_list,"rate");
	int field_tpnum = mxGetFieldNumber(ml_list,"tpnum");
	int field_bps = mxGetFieldNumber(ml_list,"bps");
	int field_data_type = mxGetFieldNumber(ml_list,"data_type");
	int field_signal_gain = mxGetFieldNumber(ml_list,"signal_gain");
	int field_signal_offset = mxGetFieldNumber(ml_list,"signal_offset");
	int field_signal_slope = mxGetFieldNumber(ml_list,"signal_slope");
	int field_signal_units = mxGetFieldNumber(ml_list,"signal_units");

	*err = 0;
	for(i=0; i < num_channels; i++) {
	    /* Channel Name */
	    if (put_mxarray_str(ml_list, field_name, i, channels[i].name)) {
		*err = 3;
		break;
	    }
	
	    /* Channel type (formerly group number) */
	    if (put_mxarray_int(ml_list, field_group_num, i,
				(int)channels[i].type)){
		*err = 3;
		break;
	    }

	    /* Sample Rate */
	    if (put_mxarray_double(ml_list, field_rate, i, channels[i].rate)){
		*err = 3;
		break;
	    }

	    /* Test Point Number */
	    if (put_mxarray_int(ml_list, field_tpnum, i, channels[i].tpnum)){
		*err = 3;
		break;
	    }

	    /* bytes per sample */
	    if (put_mxarray_int(ml_list, field_bps, i, channels[i].bps)){
		*err = 3;
		break;
	    }

	    /* Data Type */
	    if (put_mxarray_int(ml_list, field_data_type, i, 
				(int)channels[i].data_type)){
		*err = 3;
		break;
	    }

	    /* Signal Gain */
	    if (put_mxarray_double(ml_list, field_signal_gain, i, 
				   channels[i].s.signal_gain)){
		*err = 3;
		break;
	    }

	    /* Signal Offset */
	    if (put_mxarray_double(ml_list, field_signal_offset, i, 
				   channels[i].s.signal_offset)){
		*err = 3;
		break;
	    }

	    /* Signal Slope */
	    if (put_mxarray_double(ml_list, field_signal_slope, i, 
				   channels[i].s.signal_slope)){
		*err = 3;
		break;
	    }

	    /* Signal Units */
	    if (put_mxarray_str(ml_list, field_signal_units, i, 
				channels[i].s.signal_units)) {
		*err = 3;
		break;
	    }
	}
    }
    if (*err) {
	mxDestroyArray(ml_list);
	ml_list = NULL;
    }
    return ml_list;
}
