/* -*- mode: c; c-basic-offset: 4; -*- */
#if HAVE_CONFIG_H
#include "daq_config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#else
#include <io.h>
#endif /* HAVE_UNISTD_H */
#if HAVE_INTTYPES_H
#include <inttypes.h>
#else
#include <stdint.h>
#endif /* HAVE_INTTYPES_H */

#include "daqc.h"
#include "daqc_internal.h"
#include "channel.h"
#include "trench.h"

#include "nds_logging.h"

#define MAX_GROUP_LIST   4096
#define MAX_CHANNEL_LIST 209715200

#define NEW_VECT(type,dim) ((type*)malloc(dim*sizeof(type)))

enum pgm_mode {
    list_group,
    chan_list,
    chan_count,
    chan_crc,
    chan_data,
    server_protocol,
    server_version,
    cmd_exec
};

#define DEFAULT_DELTA 2

static int optimize_channel_fetch = 1;

/************************************************************************
 *                                                                      *
 *       Dump out data                                                  *
 *                                                                      *
 ************************************************************************/
void
dump_data(daq_t* daq, size_t spl, const char* fmt) {
    uint4_type i;

    for (i=0; i<daq->num_chan_request; i++) {

	size_t j, k, N;
	daq_data_t dtype;

	chan_req_t* chan = daq->chan_req_list + i;
	if (chan->status < 0) {
	    printf("Channel: %s receive error (%i)\n", 
		   chan->name, -chan->status);	    
	    continue;
	}

	dtype = chan->data_type;

	N = (size_t)chan->status / data_type_size(dtype);
	printf("Channel: %s  type: %s  nWords: %zi\n", 
	       chan->name, data_type_name(dtype), N);

	switch (dtype) {
	case _16bit_integer: {
	    short* p = (short*)NULL;

            const char* sfmt = "%6i";
            if (fmt && *fmt) sfmt = fmt;
	    p = (short*)( daq_get_block_data( daq )
			  + chan->offset);
	    for (j=0; j<N; j+=spl) {
		printf("%5zi", j);
		for (k=j; k<j+spl && k<N; ++k) {
		    printf(" ");
		    printf(sfmt, (int)*p++);
		}
		printf("\n");
	    }
	    break;
	}
	case _32bit_integer: {
	    int32_t* p =(int32_t*)NULL;
            const char* ifmt = "%8i";
            if (fmt && *fmt) ifmt = fmt;
	    p = (int32_t*)(daq_get_block_data( daq )
			   + chan->offset);
	    for (j=0; j<N; j+=spl) {
		printf("%5zi", j);
		for (k=j; k<j+spl && k<N; ++k) {
		    printf(" ");
		    printf(ifmt, (int)*p++);
		}
		printf("\n");
	    }
	    break;
	}
	case _32bit_float: {
	    float* p = (float *)NULL;
            const char* ffmt = "%13.7g";
            if (fmt && *fmt) ffmt = fmt;
	    p = (float*)(daq_get_block_data( daq )
			 + chan->offset);
	    for (j=0; j<N; j+=spl) {
		printf("%5zi", j);
		for (k=j; k<j+spl && k<N; ++k) {
		    printf(" ");
		    printf(ffmt, *p++);
		}
		printf("\n");
	    }
	    break;
	}
	case _64bit_double: {
	    double* p = (double*)NULL;
            const char* dfmt = "%15.9g";
            if (fmt && *fmt) dfmt = fmt;
	    p = (double*)(daq_get_block_data( daq )
			  + chan->offset);
	    for (j=0; j<N; j+=spl) {
		printf("%5zi", j);
		for (k=j; k<j+spl && k<N; ++k) { 
		    printf(" ");
		    printf(dfmt, *p++);
		}
		printf("\n");
	    }
	    break;
	}
	default:
	    printf("Channel: %s data of unknown/unsupprted type (%i)\n", 
		   chan->name, chan->data_type);
	}
    }
}

/************************************************************************
 *                                                                      *
 *       Dump out channel status information                            *
 *                                                                      *
 ************************************************************************/
void
dump_status(daq_t* daq) {
    size_t i;
    printf("Data for %i seconds starting at GPS: %i\n", 
	   daq_get_block_secs( daq ),
	   daq_get_block_gps( daq ) );
    printf("%-40s %-7s %s\n", "Channel", "type", "nWords");
    for (i=0; i<daq->num_chan_request; i++) {
	chan_req_t* chan = daq->chan_req_list + i;
	daq_data_t dtype = chan->data_type;
	if (chan->status < 0) {
	    printf("%-40s %-7s: %s\n", chan->name, 
		   data_type_name(chan->data_type), 
		   daq_strerror(-chan->status));
	} else {
	    size_t N = (size_t) chan->status / data_type_size(dtype);
	    printf("%-40s %-7s %zi\n", chan->name, 
		   data_type_name(chan->data_type), N);
	}
    }
}

/************************************************************************
 *                                                                      *
 *       Main Routine                                                   *
 *                                                                      *
 ************************************************************************/
int
main(int argc, const char* argv[]) {
    enum nds_version vrsn = nds_try;
    enum pgm_mode mode   = chan_data;
    enum chantype c_type = cUnknown;

    short port_id   = DAQD_PORT;
    const char* cmd = 0;
    const char* node_id = "localhost";
    const char* epoch = 0;
    time_t gps_start=0, gps_end=0, delta=DEFAULT_DELTA;
    int verbose = 0;
    size_t spl = 8;
    const char* fmt = "";
    daq_t daq;
    int rc;

    int iarg;
    int syntax=0;

#if 0
    /*
     * This is an example of how to enable debugging.
     */
    nds_logging_enable( NDS_LOG_GROUP_CONNECTION );
    nds_logging_enable( NDS_LOG_GROUP_VERBOSE_ERRORS );
    nds_logging_enable( NDS_LOG_GROUP_TRACE_ENTRY_EXIT );
    nds_logging_enable( NDS_LOG_GROUP_STATUS_UPDATE );
    nds_logging_enable( NDS_LOG_GROUP_USER );
#endif /* 0 */


    nds_logging_init( );

    for (iarg=1; iarg<argc && *(argv[iarg]) == '-'; ++iarg) {
	if (!strcmp(argv[iarg], "-c")) {
	    mode = cmd_exec;
	    if (++iarg == argc) syntax = 1;
	    else                cmd  = argv[iarg];
	} else if (!strcmp(argv[iarg], "-d")) {
	    if (++iarg == argc) syntax = 1;
	    else                delta = strtol(argv[iarg], 0, 0);
	}

	/*------------------------------  End gps time               */
	else if (!strcmp(argv[iarg], "-e")) {
	    if (++iarg == argc) syntax = 1;
	    else                gps_end = strtol(argv[iarg], 0, 0);
	} 

	/*------------------------------  Epoch to be interrogated   */
	else if (!strcmp(argv[iarg], "--epoch")) {
	    if (++iarg == argc) syntax = 1;
	    else                epoch = argv[iarg];
	} 
	else if (!strcmp(argv[iarg], "--channel-list-pre-fetch")) {
	    optimize_channel_fetch = 0;
	} 
	else if (!strcmp(argv[iarg], "-g")) {
	    mode = list_group;
	} else if (!strcmp(argv[iarg], "--help")) {
	    syntax = 1;
	    break;
        } else if (!strcmp(argv[iarg], "-h") || 
		   !strcmp(argv[iarg], "--hash")) {
	    mode = chan_crc;
        } else if (!strcmp(argv[iarg], "-k") || 
		   !strcmp(argv[iarg], "--count")) {
	    mode = chan_count;
        } else if (!strcmp(argv[iarg], "-l") || 
		   !strcmp(argv[iarg], "--list")) {

	    if (mode != chan_crc && mode != chan_count) mode = chan_list;
	} else if (!strcmp(argv[iarg], "-n")) {
	    if (++iarg == argc) syntax = 1;
	    else                node_id = argv[iarg];
	} else if (!strcmp(argv[iarg], "-p")) {
	    if (++iarg == argc) syntax = 1;
	    else                port_id = (short) strtol(argv[iarg], 0, 0);
        } else if (!strcmp(argv[iarg], "-s")) {
            if (++iarg == argc) syntax = 1;
	    else                gps_start = strtol(argv[iarg], 0, 0);
	} else if (!strcmp(argv[iarg], "--sample-format")) {
	    if (++iarg == argc) syntax = 1;
	    else                fmt = argv[iarg];
	} else if (!strcmp(argv[iarg], "--samples-per-line")) {
	    if (++iarg == argc) syntax = 1;
	    else                spl = (size_t)strtol(argv[iarg], 0, 0);
	} else if (!strcmp(argv[iarg], "--server-protocol")) {
	    mode = server_protocol;
	} else if (!strcmp(argv[iarg], "--server-version")) {
	    mode = server_version;
	} else if (!strcmp(argv[iarg], "-t")) {
	    if (++iarg == argc) syntax = 1;
	    else                c_type = cvt_str_chantype(argv[iarg]);
	} else if (!strcmp(argv[iarg], "-v")) {
	    verbose++;
	} else if (!strcmp(argv[iarg], "--version")) {
	    char* pos = strrchr( argv[0], '/' );
	    if ( pos ) {
		++pos;
	    }
	    printf( "%s (%s) %s\n", (( pos ) ? pos :  argv[0]), PACKAGE_NAME, PACKAGE_VERSION );
	    exit( 0 );
	} else if (!strcmp(argv[iarg], "-1")) {
	    vrsn = nds_v1;
	} else if (!strcmp(argv[iarg], "-2")) {
	    vrsn = nds_v2;
	} else {
	    fprintf(stderr, "Unrecognized command: %s\n", argv[iarg]);
	    syntax = 1;
	    break;
	}
    }
    if ( mode == server_protocol ) {
	if (!port_id) port_id = DAQD_PORT;
	daq_startup();
	rc = daq_connect(&daq, node_id, port_id, vrsn);
	if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	    nds_logging_printf( "DEBUG: %06d - %s - rc %d\n",
				__LINE__,
				__FILE__,
				rc );
	}
	if (rc) {
	    printf("Error in daq_connect: %s\n", daq_strerror(rc));
	    return 1;
	} else {
	    printf( "server: %s port: %d protcol: %d\n", 
		    node_id, port_id, daq.nds_versn );
	    daq_disconnect( &daq );
	}
	exit ( 0 );
    } else if ( mode == server_version ) {
	if (!port_id) port_id = DAQD_PORT;
	daq_startup();
	rc = daq_connect(&daq, node_id, port_id, vrsn);
	if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	    nds_logging_printf( "DEBUG: %06d - %s - rc %d\n",
				__LINE__,
				__FILE__,
				rc );
	}
	if (rc) {
	    printf("Error in daq_connect: %s\n", daq_strerror(rc));
	    return 1;
	} else {
	    printf( "nds protocol: %d version: %d revision: %d\n", 
		    daq.nds_versn, daq.nds1_ver, daq.nds1_rev );
	    daq_disconnect( &daq );
	}
	exit ( 0 );
    } else if (mode == chan_data && iarg == argc) {
	fprintf(stderr, "No channels requested!\n");
	syntax = 1;
    }
    if (syntax) {
	fprintf(stderr, "Command syntax: \n");
	fprintf(stderr, "nds_query [--version] [--server-version] [--server-protocol]\n");
	fprintf(stderr, "          [--channel-list-pre-fetch] [--count | -k] [--hash | -h]\n");
	fprintf(stderr, "          [-g] [-l] [-n <server>] [-p <port>]\n");
	fprintf(stderr, "          [-s <start-gps>] [-e <end-gps>] [-d <delta-t>]\n");
	fprintf(stderr, "          [<channel-list>]\n");
	fprintf(stderr, "where: \n");
	fprintf(stderr, "    -d      Time stride [%i]\n", DEFAULT_DELTA);
	fprintf(stderr, "    -e      End gps time [start_gps + delta]\n");
	fprintf(stderr, "    -g      Request a list of channel groups (NDS1 only)\n");
	fprintf(stderr, "    -h      Get hash of selected channels\n");
	fprintf(stderr, "    -k      Get number of selected channels\n");
	fprintf(stderr, "    -l      List available channels\n");
	fprintf(stderr, "    -n      Specify server IP address\n");
	fprintf(stderr, "    -p      Specify port number [%i]\n", DAQD_PORT);
	fprintf(stderr, "    -s      Specify start time (as GPS)\n");
	fprintf(stderr, "    -t      Specify default channel type.\n");
	fprintf(stderr, "    -v      Verbose printout (dump channel data)\n");
	fprintf(stderr, "    -1      NDS1 protocol\n");
	fprintf(stderr, "    -2      NDS2 protocol\n");
        fprintf(stderr, "    --channel-list-pre-fetch");
	fprintf(stderr, "            Force prefetching of the channel list\n");
	fprintf(stderr, "            when communicating with version 2 servers\n");
        fprintf(stderr, "    --sample-format    printf format for data dump\n");
        fprintf(stderr, "    --samples-per-line samples dumped per line\n");
	fprintf(stderr, "    --version          print version of client and exit\n");
	fprintf(stderr, "    --server-version   print version of server and exit\n");
	fprintf(stderr, "    --server-protocol  print protocol used by the server and exit\n");
	fprintf(stderr, "    <channel-list> Space delimited channel list\n");
	fprintf(stderr, "If the protocol is not specified, nds_query tries\n");
	fprintf(stderr, "nds2 protocol, and if this fails, reverts to nds1\n");
	fprintf(stderr, "protocol\n");
	return 1;
    }
    if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	nds_logging_printf( "DEBUG: %06d - %s\n",
			    __LINE__,
			    __FILE__ );
    }
    daq_startup();

    if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	nds_logging_printf( "DEBUG: %06d - %s\n",
			    __LINE__,
			    __FILE__ );
    }
    /*--------------------------------------------------------------------*
     *                                                                    *
     *      Connect to the NDS server                                     *
     *                                                                    *
     *--------------------------------------------------------------------*/
    if (!port_id) port_id = DAQD_PORT;

    if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	nds_logging_printf( "DEBUG: %06d - %s - port_id %d\n",
			    __LINE__,
			    __FILE__,
			    port_id );
    }
    rc = daq_connect(&daq, node_id, port_id, vrsn);
    if ( nds_logging_check( NDS_LOG_GROUP_USER,	10 ) ) {
	nds_logging_printf( "DEBUG: %06d - %s - rc %d\n",
			    __LINE__,
			    __FILE__,
			    rc );
    }
    if (rc) {
	printf("Error in daq_connect: %s\n", daq_strerror(rc));
	return 1;
    }

    /*----------------------------------  request epoch                   */
    if (epoch != NULL) {
	rc = daq_set_epoch(&daq, epoch);
	if (rc) printf("error in set_epoch: %s\n", daq_strerror(rc));
    }


    /*--------------------------------------------------------------------*
     *                                                                    *
     *      Execute a specific command                                    *
     *                                                                    *
     *--------------------------------------------------------------------*/
    if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	nds_logging_printf( "DEBUG: %06d - %s\n",
			    __LINE__,
			    __FILE__ );
    }
    switch (mode) {
    case cmd_exec: {
	if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	    nds_logging_printf( "DEBUG: %06d - %s\n",
				__LINE__,
				__FILE__ );
	}
	rc = daq_send(&daq, (char*)cmd);
	printf("command sent, rc=%i\n", rc);
	break;
    }

    /*--------------------------------------------------------------------*
     *                                                                    *
     *      Get a group list in group list mode.                          *
     *                                                                    *
     *--------------------------------------------------------------------*/
    case list_group: {
	int i;
	daq_channel_group_t* group_list = 
	    NEW_VECT(daq_channel_group_t,MAX_GROUP_LIST);
	int nGroups = 0;

	if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	    nds_logging_printf( "DEBUG: %06d - %s\n",
				__LINE__,
				__FILE__ );
	}
	rc = daq_recv_channel_groups(&daq, group_list, 
				     MAX_GROUP_LIST, &nGroups);
	if (rc) {
	    printf("Error in daq_recv_channel_groups: %s\n", 
		   daq_strerror(rc));
	    return 2;
	}
	printf(" Group    ID\n");
	for (i=0; i<nGroups; i++) {
	    printf("%8s %5i\n", group_list[i].name, group_list[i].group_num);
	}
	break;
    }

    /*--------------------------------------------------------------------*
     *                                                                    *
     *      Get the channel list hash                                     *
     *                                                                    *
     *--------------------------------------------------------------------*/
    case chan_crc: {
	unsigned int hash;
	int len = sizeof(hash);

	if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	    nds_logging_printf( "DEBUG: %06d - %s\n",
				__LINE__,
				__FILE__ );
	}
	rc = daq_recv_channel_hash(&daq, &hash, &len, gps_start, c_type);
	if (rc)
	    printf("Error reading channel hash: %s\n", daq_strerror(rc));
	else
	    printf("Channel hash code: %x\n", hash);
	break;
    }

    /*--------------------------------------------------------------------*
     *                                                                    *
     *      Get the channel list                                          *
     *                                                                    *
     *--------------------------------------------------------------------*/
    case chan_list:
    case chan_data:
    case chan_count: 
    {
	/*------------------------------  Count the number of channels    */
	int i;
	int nAlloc = 0;
	int nChans = 0;
	daq_channel_t* channel_list = (daq_channel_t*)NULL;

	if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) )
	{
	    nds_logging_printf( "DEBUG: %06d - %s\n",
				__LINE__,
				__FILE__ );
	}
	if (  ( optimize_channel_fetch )
	      && ( mode == chan_data )
	      && ( daq.nds_versn == nds_v2 ) )
	{
	    /*
	     * When optimizing the retrieval of channel data for version 2
	     * protocol and beyond, do not retrieve the list of channels
	     */
	} else {
	    /*
	     * Retrieve a list of channels by first retrieving the number
	     * of channels
	     */
	    rc = daq_recv_channel_list(&daq, 0, 0, &nAlloc, gps_start, c_type);
	    if (rc) {
		printf("Error reading channel count: %s\n", daq_strerror(rc));
		break;
	    }
	    if (mode == chan_count) {
		printf("Number of channels: %i\n", nAlloc);
		break;
	    }

	    /*------------------------------  Allocate a channel list        */
	    if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
		nds_logging_printf( "DEBUG: %06d - %s\n",
				    __LINE__,
				    __FILE__ );
	    }
	    if (nAlloc > MAX_CHANNEL_LIST) {
		printf("The number of channels (%i) is larger than the max (%i)\n",
		       nAlloc, MAX_CHANNEL_LIST);
		printf("    Something must be wrong... Terminating\n");
		break;
	    }
	    channel_list = NEW_VECT(daq_channel_t, (size_t)(nAlloc));
	    
	    /*-----------------------------  Read in the channel list        */
	    rc = daq_recv_channel_list(&daq, channel_list, nAlloc, 
				       &nChans, gps_start, c_type);
	    if (rc) {
		printf("Error reading channel list: %s\n", daq_strerror(rc));
		free(channel_list);
		break;
	    }
	} // if optimize_channel_fetch ...
	/*-------------------------------------------------------------------*
	 *                                                                   *
	 *      Dump the channel list in list mode                           *
	 *                                                                   *
	 *-------------------------------------------------------------------*/ 
	if (mode == chan_list) {

	    if ( nChans > nAlloc )
	    {
		/*-----------------------------------------------------------*
		 * Make sure we do not go beyond our alloted memory.
		 *-----------------------------------------------------------*/
		nChans = nAlloc;
	    }
	    printf("Number of channels received = %i\n", nChans);
	    printf("Channel                  Rate  chan_type\n");
	    for (i=0; i<nChans; i++) {
		printf("%-40s %5g %8s %9s\n", channel_list[i].name, 
		       channel_list[i].rate, 
		       cvt_chantype_str(channel_list[i].type),
		       data_type_name(channel_list[i].data_type)
		       );
	    }
	    free(channel_list);
	    break;
	}

	/*-------------------------------------------------------------------*
	 *                                                                   *
	 *     Get and dump the channel data for the specified channels      *
	 *                                                                   *
	 *-------------------------------------------------------------------*/
	{
	    int nMTrend = 0, nChan = 0;
	    for (i=iarg; i<argc; ++i) {
		struct trench_struct tch;
		int j, chanok = 0;
		daq_channel_t chan;

		trench_init(&tch);
		trench_parse(&tch, argv[i]);
		if (daq.nds_versn == nds_v1) {
		    for (j=0; j<nChans; ++j) {
			if (!trench_cmp_base(&tch, channel_list[j].name)) {
			    trench_infer_chan_info(&tch, c_type,
						   channel_list[j].rate,
						   channel_list[j].data_type);
			    daq_init_channel(&chan, tch.str, tch.ctype,
					     tch.rate, tch.dtype);
			    chanok = 1;
			    break;
			}
		    }
		} else {
		    if ( channel_list )
		    {
			for (j=0; j<nChans; ++j) {
			    if (tch.ctype != cUnknown && 
				tch.ctype != channel_list[j].type) continue;

			    if (!strcmp(tch.str, channel_list[j].name)) {
				if (channel_list[j].type == cMTrend) nMTrend++;
				daq_init_channel(&chan, argv[i],
						 channel_list[j].type,
						 channel_list[j].rate,
						 channel_list[j].data_type);
				chanok = 1;
				break;
			    }
			}
		    }
		    else
		    {
			daq_init_channel( &chan, tch.str,
					  tch.ctype,
					  tch.rate,
					  tch.dtype );
			chanok = 1;
		    }
		}
		
		if (chanok) {
		    nChan++;
		    daq_request_channel_from_chanlist(&daq, &chan);
		} else {
		    printf("Channel: %s not in list... Ignored.\n", argv[i]);
		}
		trench_destroy(&tch);
	    }

	/*-------------------------------------------------------------------*
	 *                                                                   *
	 *     Get and dump the channel data for the specified channels      *
	 *                                                                   *
	 *-------------------------------------------------------------------*/
	    if (!gps_end) {
		gps_end = gps_start + (time_t) delta;
	    } else if (gps_end < gps_start) {
		gps_end += gps_start;
	    }
	    if (gps_end < gps_start + delta) {
		delta = gps_end - gps_start;
	    }
	    if (nMTrend != 0 && ((gps_start%60) !=0 || (delta%60) != 0)) {
		printf("Minute trend requests start & end on a GPS minute\n");
		return 2;
	    }

	    rc = daq_request_data(&daq, gps_start, gps_end, delta);
	    if (rc) {
		printf("Error in daq_request_data: %s\n", daq_strerror(rc));
	    } else {
		time_t gps, dt=delta;
		for (gps=gps_start; gps<gps_end; gps+=dt) {
		    int rc = daq_recv_next(&daq);
		    if (rc) {
			printf("Error in daq_recv_next: %s\n", 
			       daq_strerror(rc));
			break;
		    }
		    dump_status(&daq);
		    if (verbose) dump_data(&daq, spl, fmt);
		    dt = (time_t) daq_get_block_secs( &daq );
		}
	    }
	    free(channel_list);
	}
    }
    case server_protocol:
    case server_version:
	// Nothing needs to be done here
	break;
	    
    }

    /*--------------------------------------------------------------------*
     *                                                                    *
     *     All done, go away...                                           *
     *                                                                    *
     *--------------------------------------------------------------------*/
    if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	nds_logging_printf( "DEBUG: %06d - %s\n",
			    __LINE__,
			    __FILE__ );
    }

    daq_disconnect(&daq);
    if ( nds_logging_check( NDS_LOG_GROUP_USER, 10 ) ) {
	nds_logging_printf( "DEBUG: %06d - %s\n",
			    __LINE__,
			    __FILE__ );
    }
    daq_recv_shutdown(&daq);
    return 0;
}
