/* -*- mode: c; c-basic-offset: 4; -*- */
/*******************************************************
  Matlab mex wrapper for NDS.
  
  This allows access to full sample rate data (i.e., not
  trend data).
******************************************************/

#if HAVE_CONFIG_H
#include "daq_config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <stdio.h>
#include <math.h>
#include "daqc.h"
#include "nds_log_matlab.h"
#include "nds_logging.h"
#include "nds_mex_utils.h"

typedef double* mxDataReal_type;
typedef double* mxDataImag_type;

#define NDS_MSG_LEN 256
static const size_t ndsmsglen = NDS_MSG_LEN;

/*  Internal functions */
void mlwrapper_request_data(daq_t*, const mxArray*);

mxArray* mlwrapper_get_channel_data(daq_t* daq, double start, double duration);


void 
mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) {
    /* Matlab arguments */
    /* func(request_chans, start_time, duration, host) */
    /*                  0           1         2     3  */
    char hpbuf[MAX_HOST_LENGTH];
    char msg[NDS_MSG_LEN];
    double start, duration;
    mxArray* retval = NULL;
    short port=0;
    daq_t daq;
    int rc;

    /*** Input Validation ***/
    if(nlhs != 1) {
	mexErrMsgTxt("This function only returns one paramter.");
    }

    if(nrhs != 4) {
	mexErrMsgTxt("This function requires four arguments.\n \
                      channel list to query, \n \
                      start time (GPS seconds)\n \
                      duration (seconds)\n \
                      hostport_string\n");
    }

    /* channel list must be a Cell type */
    if( ! mxIsCell(prhs[0]) ) {
	mexErrMsgTxt("Channel list must be a cell type, e.g. {'Chan1';'Chan2';Chan3} etc.");
    }

    /* start and duration */
    /* start */
    if((mxGetM(prhs[1]) * mxGetN(prhs[1])) != 1) {
	mexErrMsgTxt("Start time must be a scalar.");
    }
    start = mxGetScalar(prhs[1]);

    /* duration */
    if((mxGetM(prhs[2]) * mxGetN(prhs[2])) != 1) {
	mexErrMsgTxt("Duration must be a scalar.");
    }
    duration = mxGetScalar(prhs[2]);

    /* to make input string an actual C string */
    if (mxGetString(prhs[3], hpbuf, (mwSize)(sizeof(hpbuf))))
        mexErrMsgTxt("Hostname should be of the form \"host\" or \"host:port\""
                     " and shorter than " STRINGIFY(MAX_HOST_LENGTH) " characters.");
    parse_host_name(hpbuf, &port);

    /* Connect to the nds server... */
    daq_startup();
    rc = daq_connect (&daq, hpbuf, port, nds_v2);
    if (rc) {
	SNPRINTF(msg, ndsmsglen,
		 "Error attempting to connect to %s: %s",
		 hpbuf, daq_strerror(rc));
	mexErrMsgTxt(msg);
    }

    /*---  Build up the channel request list */
    mlwrapper_request_data(&daq, prhs[0]);

    retval = mlwrapper_get_channel_data(&daq, start, duration);

    daq_disconnect(&daq);
    daq_recv_shutdown (&daq);

    if(retval == NULL) {
	mexErrMsgTxt("Fatal Error getting channel data.");
    }

    plhs[0] = retval;
}

/*----------------------------------  Add requested channels to the daq 
 *                                    request list.
 */
void
mlwrapper_request_data(daq_t* daq, const mxArray* chan_query_list) {
    mxArray* cur_list;
    char current_name[MAX_LONG_CHANNEL_NAME_LENGTH];
    char msg[NDS_MSG_LEN];	
    int nElem;
    int i;

    /* Check the dimensions of the channel list */
    if (mxGetNumberOfDimensions(chan_query_list) > 2 ||
	(mxGetM(chan_query_list) > 1 && mxGetN(chan_query_list) > 1)) {
        mexWarnMsgTxt("Channel list should be a row or column cell array.");
    }

    /* Must be strings */
    nElem = mxGetNumberOfElements(chan_query_list);
    for(i=0; i < nElem; i++) {
	cur_list = mxGetCell(chan_query_list,i);
	if( ! mxIsChar(cur_list)) {
	    SNPRINTF(msg, ndsmsglen, "Channel name index %d is not a string.", 
		     i+1);
	    msg[ndsmsglen-1] = '\0';
	    mexWarnMsgTxt(msg);
	} else {
	    chantype_t ctype = cUnknown;
	    double rate  = 0.0;
	    char* p;

	    mxGetString(cur_list, current_name, MAX_LONG_CHANNEL_NAME_LENGTH);
	    p = strstr(current_name, ",");
	    if (p != NULL) {
		*p++ = 0;
		ctype = cvt_str_chantype(p);
	    } 
	    daq_request_channel(daq, current_name, ctype, rate);
	}
    }
}

/*  Get channel data function */
mxArray* 
mlwrapper_get_channel_data(daq_t* daq, double start, double duration) {
    /* local variables */
    static const char* fields[] = {
	"name", "chan_type", "rate", "data_type", "signal_gain", 
	"signal_offset", "signal_slope", "signal_units", 
	"start_gps_sec", "duration_sec", "data", "exists"};

    double bduration;
    double bstart;

    mwSize nChanReq;
    mxArray* array_root;

    /* Go through each element of indices and
       attempts to get channel data for that channel */
    char msg[NDS_MSG_LEN];

    /* ---  Request the data from the server. */
    time_t tStart = floor(start);
    time_t tEnd   = ceil(start + duration);
    time_t tDt    = tEnd - tStart;
    int err = daq_request_data(daq, tStart, tEnd, tDt);
    if (err) {
	mexWarnMsgTxt("daq_request_data failed\n");
	return NULL;
    }

    err = daq_recv_next(daq);
    if (err) {
	mexWarnMsgTxt("daq_recv_next failed\n");
	return NULL;
    }

    bduration = daq_get_block_secs(daq);
    bstart    = daq_get_block_gps(daq);

    /*  Construct the array_root matlab array
     */
    nChanReq = daq->num_chan_request;
    array_root = mxCreateStructArray(1, &nChanReq, 12, fields);
    if(array_root == NULL) {
	mexWarnMsgTxt("mlwrapper_get_channel_data(): Could not allocate \
                       initial array.");
	return NULL;
    }

    {
	mwSize i;

	/* Get destination array field numbers */
	int field_name = mxGetFieldNumber(array_root,"name");
	int field_chan_type = mxGetFieldNumber(array_root,"chan_type");
	int field_rate = mxGetFieldNumber(array_root,"rate");
	int field_data_type = mxGetFieldNumber(array_root,"data_type");
	int field_signal_gain = mxGetFieldNumber(array_root,"signal_gain");
	int field_signal_offset = mxGetFieldNumber(array_root,"signal_offset");
	int field_signal_slope = mxGetFieldNumber(array_root,"signal_slope");
	int field_signal_units = mxGetFieldNumber(array_root,"signal_units");
	int field_exists = mxGetFieldNumber(array_root,"exists");
	int field_start_gps_sec = mxGetFieldNumber(array_root,"start_gps_sec");
	int field_duration_sec = mxGetFieldNumber(array_root,"duration_sec");
	/* fputs("OK 1.6\n",stderr); */

	/* Now, try to fill in data */
	for(i=0; i < nChanReq; i++) {
	    int err;
	    daq_data_t dtype;
	    chan_req_t* req;

	    /* fputs("OK 1.7\n",stderr); */
	    req = daq->chan_req_list + i;

	    /* Copy over the necessary info */
	    /* --- name                     */
	    err = put_mxarray_str(array_root, field_name, i, req->name);
	    if (err) mexErrMsgTxt("Could not create string array.");

	    /* --- channel type             */
	    err = put_mxarray_str(array_root, field_chan_type, i, 
				  cvt_chantype_str(req->type));
	    if (err) mexErrMsgTxt("Could not create string array.");

	    /* --- sample rate              */
	    err = put_mxarray_double(array_root, field_rate, i, req->rate);
	    if (err) mexErrMsgTxt("Could not create numeric array.");

	    /* --- data type                */
	    dtype = req->data_type;
	    err = put_mxarray_str(array_root, field_data_type, i, 
				  data_type_name(dtype));
	    if (err) mexErrMsgTxt("Could not create string array.");

	    /* --- signal gain              */
	    err = put_mxarray_double(array_root, field_signal_gain, i, 
				     (float)req->s.signal_gain);
	    if (err) mexErrMsgTxt("Could not create numeric array.");

	    /* --- signal offset            */
	    err = put_mxarray_double(array_root, field_signal_offset, i, 
				     (float)req->s.signal_offset);
	    if (err) mexErrMsgTxt("Could not create numeric array.");

	    /* --- signal slope             */
	    err = put_mxarray_double(array_root, field_signal_slope, i, 
				     req->s.signal_slope);
	    if (err) mexErrMsgTxt("Could not create numeric array.");

	    /* --- signal units             */
	    err = put_mxarray_str(array_root, field_signal_units, i, 
				  req->s.signal_units);
	    if (err) mexErrMsgTxt("Could not create string array.");
	
	    /* Set start time and duration fields, even if channel does not "exist"
	     */
	    err = put_mxarray_double(array_root, field_start_gps_sec, i, bstart);
	    if (err) mexErrMsgTxt("Could not create numeric array.");

	    err = put_mxarray_double(array_root, field_duration_sec, i, bduration);
	    if (err) mexErrMsgTxt("Could not create numeric array.");

	    err = put_mxarray_bool(array_root, field_exists, i, 0);
	    if (err) mexErrMsgTxt("Could not create logical scalar.");

	    /*------------------------  Get the number of bytes read in */
	    if (req->status < 0) {
		SNPRINTF(msg, ndsmsglen,
			 "Channel %s error status; %s.",
			 req->name, daq_strerror(- req->status));
		mexWarnMsgTxt(msg);
		continue;
	    }

	    /*------------------------  Get the data type and number of words */
	    {
		size_t nByt = req->status;
		mxComplexity complexity;
		mxClassID    dataclass;
		mwSize       nWord = 0;
		mxArray* dest_array;
		mxDataReal_type dest_real = (mxDataReal_type)NULL;
		mxDataImag_type dest_imag = (mxDataImag_type)NULL;

		switch(dtype) {
		case _16bit_integer:
		    dataclass = mxINT16_CLASS;
		    complexity = mxREAL;
		    nWord      = nByt/2;
		    break;
		case _32bit_integer:
		    dataclass = mxINT32_CLASS;
		    complexity = mxREAL;
		    nWord      = nByt/4;
		    break;
		case _32bit_float:
		    dataclass = mxSINGLE_CLASS;
		    complexity = mxREAL;
		    nWord      = nByt/4;
		    break;
		case _64bit_integer:
		    dataclass = mxINT64_CLASS;
		    complexity = mxREAL;
		    nWord      = nByt/8;
		    break;
		case _64bit_double:
		    dataclass = mxDOUBLE_CLASS;
		    complexity = mxREAL;
		    nWord      = nByt/8;
		    break;
		case _32bit_complex:
		    dataclass = mxSINGLE_CLASS;
		    complexity = mxCOMPLEX;
		    nWord      = nByt/8;
		    break;
		default:
		    SNPRINTF(msg, ndsmsglen, 
			     "%s has an unknown data class.",
			     req->name);
		    mexWarnMsgTxt(msg);
		    continue;
		}

		/*------------------------------  Allocate the data array */
		dest_array 
		    = mxCreateNumericArray(1, &nWord, dataclass, complexity);
		if(dest_array == NULL) {
		    SNPRINTF(msg, ndsmsglen,
			     "%s could not get data array pointer.",
			     req->name);
		    mexWarnMsgTxt(msg);
		    continue;
		}

		dest_real = mxGetData(dest_array);
		if(dest_real == NULL) {
		    SNPRINTF(msg, ndsmsglen,
			     "%s could not get real data array pointer.",
			     req->name);
		    mexWarnMsgTxt(msg);
		    continue;
		}

		if(complexity == mxCOMPLEX) {
		    /* also get imaginary part */
		    dest_imag = mxGetImagData(dest_array);
		    if (!dest_imag) {
			SNPRINTF(msg, ndsmsglen,
				 "%s could not get imaginary data array pointer.",
				 req->name);
			mexWarnMsgTxt(msg);
			continue;
		    }
		}

		{
		    /*------------------------------  Copy in the data             */
		    char* p = daq_get_block_data(daq) + req->offset;
		    if ( complexity == mxCOMPLEX )
		    {
			switch( data_type_word( req->data_type ) )
			{
			case 4:
			    {
				static const mwSize word_size = 4;

				stride_copy( dest_real, p, nWord, word_size );
				stride_copy( dest_imag, p + word_size, nWord, word_size );
			    }
			    break;
			default:
			    break;
			}
		    }
		    else
		    {
			memcpy(dest_real, p, nByt);
			dest_real = (mxDataReal_type)((char*)dest_real + nByt);
		    }
		    mxSetField(array_root, i, "data", dest_array);
		}
	    }
	    
	    /*------------------------------  Set the exists flag          */
	    err = put_mxarray_bool(array_root, field_exists, i, 1);
	    if (err) mexErrMsgTxt("Could not create logical scalar.");
	}
    }
    return array_root;
}
