% -*- mode: octave -*-
% Unit tests against mock-up NDS 1 server
%
% Copyright (C) 2014  Leo Singer <leo.singer@ligo.org>
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License along
% with this program; if not, write to the Free Software Foundation, Inc.,
% 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
%

% path
global unit_test;

unit_test = UnitTest;

bytepatterns = {
    cast([hex2dec('B0'); hex2dec('00')], 'uint8'),
    cast([hex2dec('DE'); hex2dec('AD'); hex2dec('BE'); hex2dec('EF')], 'uint8'),
    cast([hex2dec('00'); hex2dec('00'); hex2dec('00'); hex2dec('00'); hex2dec('FE'); hex2dec('ED'); hex2dec('FA'); hex2dec('CE')], 'uint8'),
    cast([hex2dec('FE'); hex2dec('ED'); hex2dec('FA'); hex2dec('CE');], 'uint8')
};

hostname = unit_test.hostname( );
port = unit_test.port( );

conn = ndsm.Connection(hostname, port, ndsm.Connection.PROTOCOL_ONE);
unit_test.check( conn.gethost(), ...
		 hostname, ...
		 'Hostname verification' );
unit_test.check( conn.getport(), ...
		 port, ...
		 'Port number verification' );
unit_test.check( conn.getprotocol(), ...
		 ndsm.Connection.PROTOCOL_ONE, ...
		 'Protocol verification' );

%------------------------------------------------------------------------
% This test does a simple count of channels
%
% There is channels A - I (9)
% Each channel has:
%     Raw Data (+1)
%     Second Trend Data (+5)
%     Minute Trend Data (+5)
%
% NOTE:
%     update this check when the channel count is
%     changed on the mock server
%------------------------------------------------------------------------

channels = conn.findchannels('*');
base_channel_count = 9;
unit_test.check( length(channels), ...
		 ( base_channel_count * (5*2 + 1) ), ...
		 'Verification of findchannels for query "*"' );

%------------------------------------------------------------------------
% Make sure names are correct
%------------------------------------------------------------------------

channels = conn.findchannels('X*:{A,B,C,G,H,I}');
unit_test.check( length(channels), ...
		 6, ...
		 'Verification of findchannels for query "X*:{A,B,C,G,H,I}"' );
validate_meta_data( channels(1), ...
		    '<X1:A (16Hz, ONLINE, INT16)>', ...
		    ndsm.Channel.CHANNEL_TYPE_ONLINE, ...
		    ndsm.Channel.DATA_TYPE_INT16, ...
		    2, ...
		    16.0, 1.0, 1.0, 0.0, 'Undef' );
validate_meta_data( channels(2), ...
		    '<X1:B (16Hz, ONLINE, INT32)>', ...
		    ndsm.Channel.CHANNEL_TYPE_ONLINE, ...
		    ndsm.Channel.DATA_TYPE_INT32, ...
		    4, ...
		    16.0, 1.0, 1.0, 0.0, 'Undef' );
validate_meta_data( channels(3), ...
		    '<X1:C (16Hz, ONLINE, INT64)>', ...
		    ndsm.Channel.CHANNEL_TYPE_ONLINE, ...
		    ndsm.Channel.DATA_TYPE_INT64, ...
		    8, ...
		    16.0, 1.0, 1.0, 0.0, 'Undef' );
validate_meta_data( channels(4), ...
		    '<X1:G (16Hz, ONLINE, UINT32)>', ...
		    ndsm.Channel.CHANNEL_TYPE_ONLINE, ...
		    ndsm.Channel.DATA_TYPE_UINT32, ...
		    4, ...
		    16.0, 1.0, 1.0, 0.0, 'Undef' );
validate_meta_data( channels(5), ...
		    '<X1:H (65536Hz, ONLINE, FLOAT32)>', ...
		    ndsm.Channel.CHANNEL_TYPE_ONLINE, ...
		    ndsm.Channel.DATA_TYPE_FLOAT32, ...
		    4, ...
		    65536.0, 1.0, 1.0, 0.0, 'Undef' );
validate_meta_data( channels(6), ...
		    '<X1:I (16384Hz, ONLINE, UINT32)>', ...
		    ndsm.Channel.CHANNEL_TYPE_ONLINE, ...
		    ndsm.Channel.DATA_TYPE_UINT32, ...
		    4, ...
		    16384.0, 1.0, 1.0, 0.0, 'Undef' );

%------------------------------------------------------------------------
% Make sure trend names are correct
%------------------------------------------------------------------------

channels = conn.findchannels( 'X1:G*' )

cn = [{'<X1:G (16Hz, ONLINE, UINT32)>'}
      {'<X1:G.max,m-trend (0.0166667Hz, MTREND, UINT32)>'} 
      {'<X1:G.max,s-trend (1Hz, STREND, UINT32)>'}
      {'<X1:G.mean,m-trend (0.0166667Hz, MTREND, FLOAT64)>'} 
      {'<X1:G.mean,s-trend (1Hz, STREND, FLOAT64)>'}
      {'<X1:G.min,m-trend (0.0166667Hz, MTREND, UINT32)>'} 
      {'<X1:G.min,s-trend (1Hz, STREND, UINT32)>'}
      {'<X1:G.n,m-trend (0.0166667Hz, MTREND, INT32)>'}
      {'<X1:G.n,s-trend (1Hz, STREND, INT32)>'}
      {'<X1:G.rms,m-trend (0.0166667Hz, MTREND, FLOAT64)>'} 
      {'<X1:G.rms,s-trend (1Hz, STREND, FLOAT64)>'}];

unit_test.check( length( cn ), ...
		 length( channels ), ...
		 'Verification of findchannels for query "X1:G*"' );
%for i = 1:length( channels )
%  unit_test.check( channels( i ).tostring( ), ...
%		   cn( i ), ...
%		   'Verification of channel name' );
%end

%------------------------------------------------------------------------
% Validate channel contents - method 1
%------------------------------------------------------------------------

cn = [{'<X1:A (16Hz, ONLINE, INT16)>'}
      {'<X1:B (16Hz, ONLINE, INT32)>'}
      {'<X1:C (16Hz, ONLINE, INT64)>'}
      {'<X1:G (16Hz, ONLINE, UINT32)>'}];

start = 1000000000
stop  = 1000000004
bufs = conn.fetch(start, stop, {'X1:A', 'X1:B', 'X1:C', 'X1:G' })
unit_test.check( length(bufs), ...
		 length(cn), ...
		 'Verification of fetch for query "{X1:A, X1:B, X1:C, X1:G }"')
for i = 1:length(bufs)
    lead = sprintf( 'Verifying buffer(%d)', ...
				      i ); % )
    buf = bufs(i);
    bytepattern = bytepatterns{i};

    unit_test.check( buf.start(), ...
		     start, ...
		     [ lead, '.start' ] );
    unit_test.check ( buf.startnano(), ...
		      0, ...
		      [ lead, '.startnano' ] );
    unit_test.check( buf.stop(), ...
		     stop, ...
		     [ lead, '.stop' ] );
    unit_test.check( ( buf.samples( ) / buf.samplerate( )  ), ...
		     4, ...
		     [ lead, ' samples/samplerate' ] );
    unit_test.check( length(buf.getdata()), ...
		     4 * buf.samplerate(), ...
		     [ lead, ' buffer size' ] );
    unit_test.check( all(typecast(buf.getdata(), 'uint8') == repmat(bytepattern, length(buf.getdata()), 1)), ...
    		     true, ...
    		     [ lead, ' data values' ] );
end

%%------------------------------------------------------------------------
%% Validate channel contents - method 2
%%------------------------------------------------------------------------

cn = [{'<X1:A (16Hz, ONLINE, INT16)>'}
      {'<X1:B (16Hz, ONLINE, INT32)>'}
      {'<X1:C (16Hz, ONLINE, INT64)>'}
      {'<X1:G (16Hz, ONLINE, UINT32)>'}];

start = 1000000000
stop  = 1000000004
bufs = conn.fetch(start, stop, {'X1:A', 'X1:B', 'X1:C', 'X1:G' })
unit_test.check( length( bufs ), ...
		 length( cn ), ...
		 'Verify number of buffers returned from fetch for query "{X1:A, X1:B, X1:C, X1:G }"' );
for i = 1:length(bufs)
    lead = sprintf( 'Verifying buffer(%d)', ...
				      i ); % )
    buf = bufs(i);
    bytepattern = bytepatterns{i};

    unit_test.check( buf.start(), ...
		     start, ...
		     [ lead, '.start' ] );
    unit_test.check ( buf.startnano(), ...
		      0, ...
		      [ lead, '.startnano' ] );
    unit_test.check( buf.stop(), ...
		     stop, ...
		     [ lead, '.stop' ] );
    unit_test.check( length(buf.getdata()), ...
		     4 * buf.samplerate(), ...
		     [ lead, ' buffer size' ] );
    unit_test.check( all(typecast(buf.getdata(), 'uint8') == repmat(bytepattern, length(buf.getdata()), 1)), ...
    		     true, ...
    		     [ lead, ' data values' ] );
end

%------------------------------------------------------------------------
% Trend channels - minute and second
%------------------------------------------------------------------------

%------------------------------------------------------------------------
% Iterator example
%------------------------------------------------------------------------

iter = conn.iterate({'X1:A', 'X1:B', 'X1:C'});
bufs = iter.next();
unit_test.check( length(bufs), ...
		 3, ...
		 'Verify number of buffers returned from iterate' );
for i = 1:length(bufs)
    lead = sprintf( 'Verifying buffer(%d)', ...
				      i ); % )
    buf = bufs(i);
    bytepattern = bytepatterns{i};
    unit_test.check( buf.start(), ...
		     1000000000, ...
		     [ lead, '.start' ]);
    unit_test.check( buf.startnano(), ...
		     0, ...
		     [ lead, '.startnano' ]);
    unit_test.check( length( buf.getdata() ), ...
		     buf.samplerate, ...
		     [ lead, ' number of samples' ]);
				      
    unit_test.check( all(typecast(buf.getdata(), 'uint8') == repmat(bytepattern, length(buf.getdata()), 1)), ...
    		     true, ...
    		     ' data values' );
end

% -----------------------------------------------------------------------
% connection automatically closed when garbage collected,
% but test explicit close anyway
% -----------------------------------------------------------------------
conn.close()
