#lang scheme

; Location = (make-location Nat Nat Nat) | Symbol
(define-struct location (line char offset))

; Source = (make-source Location Location)
(define-struct source (start stop))

; Document = (make-document Prolog Element (listof Misc))
(define-struct document (prolog element misc))

; Prolog = (make-prolog (listof Misc) Document-type (listof Misc))
(define-struct prolog (misc dtd misc2))

; Document-type = (make-document-type sym External-dtd #f)
;               | #f
(define-struct document-type (name external inlined))

; External-dtd = (make-external-dtd/public str str)
;              | (make-external-dtd/system str)
;              | #f
(define-struct external-dtd (system))
(define-struct (external-dtd/public external-dtd) (public))
(define-struct (external-dtd/system external-dtd) ())

; Element = (make-element Location Location Symbol (listof Attribute) (listof Content))
(define-struct (element source) (name attributes content))

; Attribute = (make-attribute Location Location Symbol String)
(define-struct (attribute source) (name value))

; Pcdata = (make-pcdata Location Location String)
(define-struct (pcdata source) (string))

; Cdata = (make-cdata Location Location String)
(define-struct (cdata source) (string))

; Content = Pcdata  
;         |  Element
;         |  Entity
;         |  Misc
;         |  Cdata

; Misc = Comment
;      |  Processing-instruction

; Entity = (make-entity Location Location (U Nat Symbol))
(define-struct (entity source) (text))

; Processing-instruction = (make-p-i Location Location String String)
; also represents XMLDecl
(define-struct (p-i source) (target-name instruction))

; Comment = (make-comment String)
(define-struct comment (text))

; permissive? : parameter bool
(define permissive? (make-parameter #f))

(define permissive/c
  (make-proj-contract 'permissive/c
                      (lambda (pos neg src-info name)
                        (lambda (v)
                          (if (permissive?)
                              v
                              (raise-contract-error
                               v src-info pos name "not in permissive mode"))))
                      (lambda (v)
                        (permissive?))))

; content? : TST -> Bool
(define content/c
  (or/c pcdata? element? entity? comment? cdata? p-i? permissive/c))

(define misc/c
  (or/c comment? p-i?))

(define location/c
  (or/c location? symbol? false/c))
(provide/contract
 (struct location ([line exact-nonnegative-integer?]
                   [char exact-nonnegative-integer?]
                   [offset exact-nonnegative-integer?]))
 [location/c contract?]
 (struct source ([start location/c]
                 [stop location/c]))
 (struct external-dtd ([system string?]))
 (struct (external-dtd/public external-dtd) ([system string?]
                                             [public string?]))
 (struct (external-dtd/system external-dtd) ([system string?]))
 (struct document-type ([name symbol?]
                        [external external-dtd?]
                        [inlined false/c]))
 (struct comment ([text string?]))
 (struct (p-i source) ([start location/c]
                       [stop location/c]
                       [target-name symbol?]
                       [instruction string?])) 
 [misc/c contract?]
 (struct prolog ([misc (listof misc/c)]
                 [dtd (or/c document-type? false/c)]
                 [misc2 (listof misc/c)]))
 (struct document ([prolog prolog?] 
                   [element element?]
                   [misc (listof misc/c)]))
 (struct (element source) ([start location/c]
                           [stop location/c]
                           [name symbol?]
                           [attributes (listof attribute?)]
                           [content (listof content/c)]))
 (struct (attribute source) ([start location/c]
                             [stop location/c]
                             [name symbol?]
                             [value (or/c string? permissive/c)]))
 [permissive? (parameter/c boolean?)]
 [permissive/c contract?]
 [content/c contract?] 
 (struct (pcdata source) ([start location/c]
                          [stop location/c]
                          [string string?]))
 (struct (cdata source) ([start location/c]
                         [stop location/c]
                         [string string?])) 
 (struct (entity source) ([start location/c]
                          [stop location/c]
                          [text (or/c symbol? exact-nonnegative-integer?)])))