

/* $Id: read_buf.c,v 1.3 2004/10/09 14:52:09 alien-science Exp $ */


#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>

#include "common.h"
#include "read_buf.h"

/*
 * Gets a pointer to the start of the buffer
 */
char *
rbuf_p(struct rbuf *rbuf)
{
   return &(rbuf->buf[rbuf->start]);
}

/*
 * Creates a new read buffer on the given file
 */
int
rbuf_new(struct rbuf *rbuf, size_t size, const char *fname)
{
   char *err_mess;

   /* Allocate memory for the buffer */
   rbuf->buf = XMALLOC(char, size);

   /* Open the file, use stdin on null */
   if ( 0 == fname ) {
      rbuf->fh = STDIN_FILENO;
      }
   else if (-1 == (rbuf->fh = open(fname, O_RDONLY, 0))) {

      err_mess = XMALLOC(char, strlen(fname) + 40);
      sprintf(err_mess, "Cannot open %s for reading\n", fname);
      perror(err_mess);
      return -1;
      }

   /* Set up the data structure */
   rbuf->bufsize  = size;
   rbuf->end      = 0;
   rbuf->start    = 0;

   return 0;
}


/*
 * Refills the buffer from disk setting the amount data 
 * available in the buffer that is available for reading
 */
int
refill(struct rbuf *rbuf, size_t *available)
{
   char *pstart;
   size_t   wanted,
            read_in,
            total_in;

   *available = rbuf->end - rbuf->start;

   /* See if the file is still open */
   if (-1 == rbuf->fh) return 0;

   /* Shuffle the buffer if there is data left */
   if (rbuf->start < rbuf->end &&
       rbuf->start > 0) {

      pstart = rbuf_p(rbuf);
      memcpy(rbuf->buf, pstart, rbuf->end - rbuf->start); 
      rbuf->end    = rbuf->end - rbuf->start;
      rbuf->start  = 0;
      }
   else if (rbuf->start >= rbuf->end) {
      /* Clear the used data */
      rbuf->start = 0;
      rbuf->end   = 0;
      }

   wanted   = rbuf->bufsize - rbuf->end;
   total_in = 0;

   /* Read in from the file */
   while (1) {
      read_in = read(rbuf->fh, &(rbuf->buf[rbuf->end]), wanted - total_in);
      total_in += read_in;

      /* Check for errors */
      if (-1 == read_in || 0 == read_in) {
         close(rbuf->fh);
         rbuf->fh = -1;
         break;
         }

      /* See how much needs to be read in now */
      if (wanted <= total_in) {
         break;
         }
      }

   /* Recalculate the buffer state */
   rbuf->end += total_in;
   *available = rbuf->end;

   return 0;

}

/*
 * Tries to guarantee amount bytes will be available in buffer
 * sets actual bytes available (which could be more or less)
 */
int 
rbuf_alloc(struct rbuf *rbuf, size_t amount, size_t *actual)
{
   if (amount > rbuf->end - rbuf->start) {
      ASSERT( refill(rbuf, actual));
      }
   else {
      *actual = rbuf->end - rbuf->start;
      }

   OK;
}


/*
 * Marks amount bytes at the start of the buffer as unused
 */
int 
rbuf_free(struct rbuf *rbuf, size_t amount)
{
   if (rbuf->end >= rbuf->start + amount) {
      /* Shrink the buffer */
      rbuf->start += amount;
      }
   else {
      ERROR("Cannot free more than available read buffer\n");
      }

   OK;
}

/*
 * Indicates if the buffer is empty
 */
int
rbuf_empty(struct rbuf *rbuf)
{
   if (-1 == rbuf->fh && 0 >= rbuf->end - rbuf->start) {
      return 1;
      }

   return 0;
}

/*
 * Indicates if the given pointer is valid for the buffer
 */
int 
rbuf_valid(struct rbuf *rbuf, char *p)
{
   return (p < &(rbuf->buf[rbuf->end]));
}


/*
 * Closes the read buffer
 */
void
rbuf_close(struct rbuf *rbuf)
{
   /* Close the file */
   if (-1 != rbuf->fh) {
      close(rbuf->fh);
      rbuf->fh = -1;
      }

   /* Free the memory */
   XFREE(rbuf->buf);
}

