# Copyright (c) 2011, Peter A. Bigot, licensed under New BSD (see COPYING)
# This file is part of msp430mcu (http://sourceforge.net/projects/mspgcc/)
#
# Read the chipflags.csv file and generate individual msp430_spec.spec
# files that will be referenced by the gcc spec file to determine
# device-specific default flags without requiring compiler
# modifications.

import sys
import csv
import os

msp430_root = os.environ.get('MSP430_ROOT', '/msp430')
msp430mcu_root = os.environ.get('MSP430MCU_ROOT', os.path.join(msp430_root, 'msp430mcu'))
analysis_dir = os.path.join(msp430mcu_root, 'analysis')
upstream_dir = os.path.join(msp430mcu_root, 'upstream')

def analysis_path (filename):
    global analysis_dir
    return os.path.join(analysis_dir, filename)

def upstream_path (filename):
    global upstream_dir
    return os.path.join(upstream_dir, filename)

reader = csv.reader(file(analysis_path('chipflags.csv')))

fields = reader.next()


cpu_spec = []
mpy_spec = []
ivcnt_spec = []

sections = { 'cpu': cpu_spec,
             'mpy': mpy_spec,
             'ivcnt': ivcnt_spec }

cpu_spec.append('mmcu=msp430generic:-mcpu=430')
mpy_spec.append('mmcu=msp430generic:-mmpy=none')

cpu_spec.append('mmcu=msp430xgeneric:-mcpu=430x')
mpy_spec.append('mmcu=msp430xgeneric:-mmpy=16')

for row in reader:
    args = dict(zip(('mcu', 'xmcu', 'cpu', 'mpy', 'intr'), row))

    mcu_selector = 'mmcu=%(mcu)s|mmcu=%(mcu)s_*' % args
    args['mcusel'] = mcu_selector
    # Define mcu-specific values for CPU, hardware multiplier, and
    # interrupt vector length.
    cpu_spec.append('%(mcusel)s:-mcpu=%(cpu)s' % args)
    mpy_spec.append('%(mcusel)s:-mmpy=%(mpy)s' % args)
    ivcnt_spec.append('%(mcusel)s:-mivcnt=%(intr)s' % args)

cpu_spec.append(':-mcpu=430')
cpu_spec[:] = [ '%{!mcpu=*:%{', '; '.join(cpu_spec), '}}', '%{mcpu=*}' ]

mpy_spec.append(':-mmpy=none')
mpy_spec[:] = [ '%{!mmpy=*:%{', '; '.join(mpy_spec), '}}', '%{mmpy=*}' ]

ivcnt_spec.append(':-mivcnt=16')
ivcnt_spec[:] = [ '%{!mivcnt=*:%{', '; '.join(ivcnt_spec), '}}', '%{mivcnt=*}' ]

outf = file(analysis_path('msp430mcu.spec'), 'w')

for (block, spec) in sections.items():
    outf.write("*msp430_%s: %s\n\n" % (block, " ".join(spec)))
