/*
 * jsimd_mips.c
 *
 * Copyright 2009 Pierre Ossman <ossman@cendio.se> for Cendio AB
 * Copyright 2009-2011 D. R. Commander
 * Copyright (C) 2013, MIPS Technologies, Inc., California
 *
 * Based on the x86 SIMD extension for IJG JPEG library,
 * Copyright (C) 1999-2006, MIYASAKA Masaru.
 * For conditions of distribution and use, see copyright notice in jsimdext.inc
 *
 * This file contains the interface between the "normal" portions
 * of the library and the SIMD implementations when running on
 * MIPS architecture.
 *
 * Based on the stubs from 'jsimd_none.c'
 */

#define JPEG_INTERNALS
#include "../jinclude.h"
#include "../jpeglib.h"
#include "../jsimd.h"
#include "../jdct.h"
#include "../jsimddct.h"
#include "jsimd.h"

#include <stdio.h>
#include <string.h>
#include <ctype.h>

static unsigned int simd_support = ~0;

#if defined(__linux__)

LOCAL(int)
parse_proc_cpuinfo(const char* search_string)
{
  const char* file_name = "/proc/cpuinfo";
  char cpuinfo_line[256];
  FILE* f = NULL;
  simd_support = 0;

  if ((f = fopen(file_name, "r")) != NULL) {
    while (fgets(cpuinfo_line, sizeof(cpuinfo_line), f) != NULL) {
      if (strstr(cpuinfo_line, search_string) != NULL) {
        fclose(f);
        simd_support |= JSIMD_MIPS_DSPR2;
        return 1;
      }
    }
    fclose(f);
  }
  /* Did not find string in the proc file, or not Linux ELF. */
  return 0;
}
#endif

/*
 * Check what SIMD accelerations are supported.
 *
 * FIXME: This code is racy under a multi-threaded environment.
 */
LOCAL(void)
init_simd (void)
{
  if (simd_support != ~0U)
    return;

  simd_support = 0;

#if defined(__mips__) && defined(__mips_dsp) && (__mips_dsp_rev >= 2)
  simd_support |= JSIMD_MIPS_DSPR2;
#elif defined(__linux__)
  /* We still have a chance to use MIPS DSPR2 regardless of globally used
   * -mdspr2 options passed to gcc by performing runtime detection via
   * /proc/cpuinfo parsing on linux */
  if (!parse_proc_cpuinfo("MIPS 74K"))
    return;
#endif
}
static const int mips_idct_ifast_coefs[4] = {
  0x45404540,           // FIX( 1.082392200 / 2) =  17734 = 0x4546
  0x5A805A80,           // FIX( 1.414213562 / 2) =  23170 = 0x5A82
  0x76407640,           // FIX( 1.847759065 / 2) =  30274 = 0x7642
  0xAC60AC60            // FIX(-2.613125930 / 4) = -21407 = 0xAC61
};

GLOBAL(int)
jsimd_can_rgb_ycc (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if ((RGB_PIXELSIZE != 3) && (RGB_PIXELSIZE != 4))
    return 0;
  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_rgb_gray (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if ((RGB_PIXELSIZE != 3) && (RGB_PIXELSIZE != 4))
    return 0;
  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_ycc_rgb (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if ((RGB_PIXELSIZE != 3) && (RGB_PIXELSIZE != 4))
    return 0;
  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(void)
jsimd_rgb_ycc_convert (j_compress_ptr cinfo,
                       JSAMPARRAY input_buf, JSAMPIMAGE output_buf,
                       JDIMENSION output_row, int num_rows)
{
  void (*mipsdspr2fct)(JDIMENSION, JSAMPARRAY, JSAMPIMAGE, JDIMENSION, int);
  switch(cinfo->in_color_space)
  {
    case JCS_EXT_RGB:
      mipsdspr2fct=jsimd_extrgb_ycc_convert_mips_dspr2;
      break;
    case JCS_EXT_RGBX:
    case JCS_EXT_RGBA:
      mipsdspr2fct=jsimd_extrgbx_ycc_convert_mips_dspr2;
      break;
    case JCS_EXT_BGR:
      mipsdspr2fct=jsimd_extbgr_ycc_convert_mips_dspr2;
      break;
    case JCS_EXT_BGRX:
    case JCS_EXT_BGRA:
      mipsdspr2fct=jsimd_extbgrx_ycc_convert_mips_dspr2;
      break;
    case JCS_EXT_XBGR:
    case JCS_EXT_ABGR:
      mipsdspr2fct=jsimd_extxbgr_ycc_convert_mips_dspr2;

      break;
    case JCS_EXT_XRGB:
    case JCS_EXT_ARGB:
      mipsdspr2fct=jsimd_extxrgb_ycc_convert_mips_dspr2;
      break;
    default:
      mipsdspr2fct=jsimd_extrgb_ycc_convert_mips_dspr2;
      break;
  }

  if (simd_support & JSIMD_MIPS_DSPR2)
    mipsdspr2fct(cinfo->image_width, input_buf,
        output_buf, output_row, num_rows);
}

GLOBAL(void)
jsimd_rgb_gray_convert (j_compress_ptr cinfo,
                        JSAMPARRAY input_buf, JSAMPIMAGE output_buf,
                        JDIMENSION output_row, int num_rows)
{
  void (*mipsdspr2fct)(JDIMENSION, JSAMPARRAY, JSAMPIMAGE, JDIMENSION, int);
  switch(cinfo->in_color_space)
  {
    case JCS_EXT_RGB:
      mipsdspr2fct=jsimd_extrgb_gray_convert_mips_dspr2;
      break;
    case JCS_EXT_RGBX:
    case JCS_EXT_RGBA:
      mipsdspr2fct=jsimd_extrgbx_gray_convert_mips_dspr2;
      break;
    case JCS_EXT_BGR:
      mipsdspr2fct=jsimd_extbgr_gray_convert_mips_dspr2;
      break;
    case JCS_EXT_BGRX:
    case JCS_EXT_BGRA:
      mipsdspr2fct=jsimd_extbgrx_gray_convert_mips_dspr2;
      break;
    case JCS_EXT_XBGR:
    case JCS_EXT_ABGR:
      mipsdspr2fct=jsimd_extxbgr_gray_convert_mips_dspr2;
      break;
    case JCS_EXT_XRGB:
    case JCS_EXT_ARGB:
      mipsdspr2fct=jsimd_extxrgb_gray_convert_mips_dspr2;
      break;
    default:
      mipsdspr2fct=jsimd_extrgb_gray_convert_mips_dspr2;
      break;
  }

  if (simd_support & JSIMD_MIPS_DSPR2)
    mipsdspr2fct(cinfo->image_width, input_buf,
        output_buf, output_row, num_rows);

}

GLOBAL(void)
jsimd_ycc_rgb_convert (j_decompress_ptr cinfo,
                       JSAMPIMAGE input_buf, JDIMENSION input_row,
                       JSAMPARRAY output_buf, int num_rows)
{
  void (*mipsdspr2fct)(JDIMENSION, JSAMPIMAGE, JDIMENSION, JSAMPARRAY, int);

  switch(cinfo->out_color_space)
  {
    case JCS_EXT_RGB:
      mipsdspr2fct=jsimd_ycc_extrgb_convert_mips_dspr2;
      break;
    case JCS_EXT_RGBX:
    case JCS_EXT_RGBA:
      mipsdspr2fct=jsimd_ycc_extrgbx_convert_mips_dspr2;
      break;
    case JCS_EXT_BGR:
      mipsdspr2fct=jsimd_ycc_extbgr_convert_mips_dspr2;
      break;
    case JCS_EXT_BGRX:
    case JCS_EXT_BGRA:
      mipsdspr2fct=jsimd_ycc_extbgrx_convert_mips_dspr2;
      break;
    case JCS_EXT_XBGR:
    case JCS_EXT_ABGR:
      mipsdspr2fct=jsimd_ycc_extxbgr_convert_mips_dspr2;
      break;
    case JCS_EXT_XRGB:
    case JCS_EXT_ARGB:
      mipsdspr2fct=jsimd_ycc_extxrgb_convert_mips_dspr2;
      break;
  default:
      mipsdspr2fct=jsimd_ycc_extrgb_convert_mips_dspr2;
      break;
  }

  if (simd_support & JSIMD_MIPS_DSPR2)
    mipsdspr2fct(cinfo->output_width, input_buf,
        input_row, output_buf, num_rows);
}

GLOBAL(int)
jsimd_can_h2v2_downsample (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_h2v1_downsample (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(void)
jsimd_h2v2_downsample (j_compress_ptr cinfo, jpeg_component_info * compptr,
                       JSAMPARRAY input_data, JSAMPARRAY output_data)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_h2v2_downsample_mips_dspr2(cinfo->image_width,
        cinfo->max_v_samp_factor, compptr->v_samp_factor,
        compptr->width_in_blocks, input_data, output_data);
}

GLOBAL(void)
jsimd_h2v1_downsample (j_compress_ptr cinfo, jpeg_component_info * compptr,
                       JSAMPARRAY input_data, JSAMPARRAY output_data)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_h2v1_downsample_mips_dspr2(cinfo->image_width,
        cinfo->max_v_samp_factor, compptr->v_samp_factor,
        compptr->width_in_blocks, input_data, output_data);
}

GLOBAL(int)
jsimd_can_h2v2_upsample (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_h2v1_upsample (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(void)
jsimd_h2v2_upsample (j_decompress_ptr cinfo,
                     jpeg_component_info * compptr,
                     JSAMPARRAY input_data,
                     JSAMPARRAY * output_data_ptr)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_h2v2_upsample_mips_dspr2(cinfo->max_v_samp_factor,
        cinfo->output_width, input_data, output_data_ptr);
}

GLOBAL(void)
jsimd_h2v1_upsample (j_decompress_ptr cinfo,
                     jpeg_component_info * compptr,
                     JSAMPARRAY input_data,
                     JSAMPARRAY * output_data_ptr)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_h2v1_upsample_mips_dspr2(cinfo->max_v_samp_factor,
        cinfo->output_width, input_data, output_data_ptr);
}

GLOBAL(int)
jsimd_can_h2v2_fancy_upsample (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_h2v1_fancy_upsample (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(void)
jsimd_h2v2_fancy_upsample (j_decompress_ptr cinfo,
                           jpeg_component_info * compptr,
                           JSAMPARRAY input_data,
                           JSAMPARRAY * output_data_ptr)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_h2v2_fancy_upsample_mips_dspr2(cinfo->max_v_samp_factor,
        compptr->downsampled_width, input_data, output_data_ptr);
}

GLOBAL(void)
jsimd_h2v1_fancy_upsample (j_decompress_ptr cinfo,
                           jpeg_component_info * compptr,
                           JSAMPARRAY input_data,
                           JSAMPARRAY * output_data_ptr)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_h2v1_fancy_upsample_mips_dspr2(cinfo->max_v_samp_factor,
        compptr->downsampled_width, input_data, output_data_ptr);
}

GLOBAL(int)
jsimd_can_h2v2_merged_upsample (void)
{
  return 0;
}

GLOBAL(int)
jsimd_can_h2v1_merged_upsample (void)
{
  return 0;
}

GLOBAL(void)
jsimd_h2v2_merged_upsample (j_decompress_ptr cinfo,
                            JSAMPIMAGE input_buf,
                            JDIMENSION in_row_group_ctr,
                            JSAMPARRAY output_buf)
{
}

GLOBAL(void)
jsimd_h2v1_merged_upsample (j_decompress_ptr cinfo,
                            JSAMPIMAGE input_buf,
                            JDIMENSION in_row_group_ctr,
                            JSAMPARRAY output_buf)
{
}

GLOBAL(int)
jsimd_can_convsamp (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (sizeof(DCTELEM) != 2)
    return 0;

  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_convsamp_float (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(JCOEF) != 2)
    return 0;
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (sizeof(ISLOW_MULT_TYPE) != 2)
    return 0;

  if ((simd_support & JSIMD_MIPS_DSPR2))
    return 1;

  return 0;
}

GLOBAL(void)
jsimd_convsamp (JSAMPARRAY sample_data, JDIMENSION start_col,
                DCTELEM * workspace)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_convsamp_mips_dspr2(sample_data, start_col, workspace);
}

GLOBAL(void)
jsimd_convsamp_float (JSAMPARRAY sample_data, JDIMENSION start_col,
                      FAST_FLOAT * workspace)
{
  if ((simd_support & JSIMD_MIPS_DSPR2))
    jsimd_convsamp_float_mips_dspr2(sample_data, start_col, workspace);
}

GLOBAL(int)
jsimd_can_fdct_islow (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(DCTELEM) != 2)
    return 0;

  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_fdct_ifast (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(DCTELEM) != 2)
    return 0;

  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_fdct_float (void)
{
  return 0;
}

GLOBAL(void)
jsimd_fdct_islow (DCTELEM * data)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_fdct_islow_mips_dspr2(data);
}

GLOBAL(void)
jsimd_fdct_ifast (DCTELEM * data)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_fdct_ifast_mips_dspr2(data);
}

GLOBAL(void)
jsimd_fdct_float (FAST_FLOAT * data)
{
}

GLOBAL(int)
jsimd_can_quantize (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(JCOEF) != 2)
    return 0;
  if (sizeof(DCTELEM) != 2)
    return 0;

  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_quantize_float (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(JCOEF) != 2)
    return 0;
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (sizeof(ISLOW_MULT_TYPE) != 2)
    return 0;

  if ((simd_support & JSIMD_MIPS_DSPR2))
    return 1;

  return 0;
}

GLOBAL(void)
jsimd_quantize (JCOEFPTR coef_block, DCTELEM * divisors,
                DCTELEM * workspace)
{
  if (simd_support & JSIMD_MIPS_DSPR2)
    jsimd_quantize_mips_dspr2(coef_block, divisors, workspace);
}

GLOBAL(void)
jsimd_quantize_float (JCOEFPTR coef_block, FAST_FLOAT * divisors,
                      FAST_FLOAT * workspace)
{
  if ((simd_support & JSIMD_MIPS_DSPR2))
    jsimd_quantize_float_mips_dspr2(coef_block, divisors, workspace);
}

GLOBAL(int)
jsimd_can_idct_2x2 (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(JCOEF) != 2)
    return 0;
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (sizeof(ISLOW_MULT_TYPE) != 2)
    return 0;

  if ((simd_support & JSIMD_MIPS_DSPR2))
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_idct_4x4 (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(JCOEF) != 2)
    return 0;
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (sizeof(ISLOW_MULT_TYPE) != 2)
    return 0;

  if ((simd_support & JSIMD_MIPS_DSPR2))
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_idct_6x6 (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(JCOEF) != 2)
    return 0;
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (sizeof(ISLOW_MULT_TYPE) != 2)
    return 0;

  if ((simd_support & JSIMD_MIPS_DSPR2))
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_idct_12x12 (void)
{
  init_simd();

  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(JCOEF) != 2)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (sizeof(ISLOW_MULT_TYPE) != 2)
    return 0;

  if (simd_support & JSIMD_MIPS_DSPR2)
    return 1;

  return 0;
}

GLOBAL(void)
jsimd_idct_2x2 (j_decompress_ptr cinfo, jpeg_component_info * compptr,
                JCOEFPTR coef_block, JSAMPARRAY output_buf,
                JDIMENSION output_col)
{
  if ((simd_support & JSIMD_MIPS_DSPR2))
    jsimd_idct_2x2_mips_dspr2(compptr->dct_table, coef_block,
                              output_buf, output_col);
}

GLOBAL(void)
jsimd_idct_4x4 (j_decompress_ptr cinfo, jpeg_component_info * compptr,
                JCOEFPTR coef_block, JSAMPARRAY output_buf,
                JDIMENSION output_col)
{
  if ((simd_support & JSIMD_MIPS_DSPR2))
  {
    int workspace[DCTSIZE*4];  /* buffers data between passes */
    jsimd_idct_4x4_mips_dspr2(compptr->dct_table, coef_block,
                              output_buf, output_col, workspace);
  }
}

GLOBAL(void)
jsimd_idct_6x6 (j_decompress_ptr cinfo, jpeg_component_info * compptr,
           JCOEFPTR coef_block, JSAMPARRAY output_buf,
           JDIMENSION output_col)
{
    if ((simd_support & JSIMD_MIPS_DSPR2))
      jsimd_idct_6x6_mips_dspr2(compptr->dct_table, coef_block,
                                output_buf, output_col);
}

GLOBAL(void)
jsimd_idct_12x12 (j_decompress_ptr cinfo, jpeg_component_info * compptr,
                  JCOEFPTR coef_block,
                  JSAMPARRAY output_buf, JDIMENSION output_col)
{
  if (simd_support & JSIMD_MIPS_DSPR2) {
    int workspace[96];
    int output[12] = {
      (int)(output_buf[0] + output_col),
      (int)(output_buf[1] + output_col),
      (int)(output_buf[2] + output_col),
      (int)(output_buf[3] + output_col),
      (int)(output_buf[4] + output_col),
      (int)(output_buf[5] + output_col),
      (int)(output_buf[6] + output_col),
      (int)(output_buf[7] + output_col),
      (int)(output_buf[8] + output_col),
      (int)(output_buf[9] + output_col),
      (int)(output_buf[10] + output_col),
      (int)(output_buf[11] + output_col),
    };
    jsimd_idct_12x12_pass1_mips_dspr2(coef_block,
                                      compptr->dct_table, workspace);
    jsimd_idct_12x12_pass2_mips_dspr2(workspace, output);
  }
}

GLOBAL(int)
jsimd_can_idct_islow (void)
{
  return 0;
}

GLOBAL(int)
jsimd_can_idct_ifast (void)
{
  init_simd();

  /* The code is optimised for these values only */
  if (DCTSIZE != 8)
    return 0;
  if (sizeof(JCOEF) != 2)
    return 0;
  if (BITS_IN_JSAMPLE != 8)
    return 0;
  if (sizeof(JDIMENSION) != 4)
    return 0;
  if (sizeof(IFAST_MULT_TYPE) != 2)
    return 0;
  if (IFAST_SCALE_BITS != 2)
    return 0;

  if ((simd_support & JSIMD_MIPS_DSPR2))
    return 1;

  return 0;
}

GLOBAL(int)
jsimd_can_idct_float (void)
{
  return 0;
}

GLOBAL(void)
jsimd_idct_islow (j_decompress_ptr cinfo, jpeg_component_info * compptr,
                JCOEFPTR coef_block, JSAMPARRAY output_buf,
                JDIMENSION output_col)
{
}

GLOBAL(void)
jsimd_idct_ifast (j_decompress_ptr cinfo, jpeg_component_info * compptr,
                  JCOEFPTR coef_block, JSAMPARRAY output_buf,
                  JDIMENSION output_col)
{
  if (simd_support & JSIMD_MIPS_DSPR2) {
    JCOEFPTR inptr;
    IFAST_MULT_TYPE * quantptr;
    DCTELEM workspace[DCTSIZE2];  /* buffers data between passes */

    /* Pass 1: process columns from input, store into work array. */

    inptr = coef_block;
    quantptr = (IFAST_MULT_TYPE *) compptr->dct_table;

    jsimd_idct_ifast_cols_mips_dspr2(inptr, quantptr,
                                     workspace, mips_idct_ifast_coefs);

    /* Pass 2: process rows from work array, store into output array. */
    /* Note that we must descale the results by a factor of 8 == 2**3, */
    /* and also undo the PASS1_BITS scaling. */

    jsimd_idct_ifast_rows_mips_dspr2(workspace, output_buf,
                                     output_col, mips_idct_ifast_coefs);
  }
}

GLOBAL(void)
jsimd_idct_float (j_decompress_ptr cinfo, jpeg_component_info * compptr,
                JCOEFPTR coef_block, JSAMPARRAY output_buf,
                JDIMENSION output_col)
{
}
