# sample time series data, converted from the original R project datasets package
# note: some series are not included since 19th century dates cannot be parsed by Tcl
# (start time, period, ... specified so that they can be parsed by the Tcl "clock scan" command)

# $Id: datasets.tcl,v 1.5 2006/01/14 20:04:05 jfontain Exp $


package provide datasets [lindex {$Revision: 1.5 $} 1]


namespace eval datasets {

    variable loadTime [clock seconds]

    variable list {
        airmiles AirPassengers austres BJsales co2 EuStockMarkets.DAX EuStockMarkets.SMI EuStockMarkets.CAC EuStockMarkets.FTSE
        ldeaths fdeaths mdeaths JohnsonJohnson lh nhtemp sinus square triangle UKDriverDeaths UKgas USAccDeaths WWWusage
    }                                                       ;# note: always append new series so old ones keep their instance number

    proc initialize {format minimum maximum} {
        variable dateTimeFormat $format
        variable minimumSeconds $minimum                                                    ;# absolute time range limits in seconds
        variable maximumSeconds $maximum
    }

    proc history {set startSeconds endSeconds last} {                                                 ;# replaces database retrieval
        variable dateTimeFormat
        variable loadTime

        set list {}
        set length [llength [set datasets::${set}(values)]]
        if {[info exists datasets::${set}(period)]} {
            set period [set datasets::${set}(period)]
            if {[info exists datasets::${set}(start)]} {
                set seconds [clock scan [set datasets::${set}(start)]]
                foreach value [set datasets::${set}(values)] {
                    if {($seconds >= $startSeconds) && ($seconds <= $endSeconds)} {
                        lappend list [clock format $seconds -format $dateTimeFormat] $value
                    }
                    set seconds [clock scan $period -base $seconds]
                }
            } else {
                set times {}
                for {set index 0; set seconds $loadTime} {$index < $length} {incr index} {
                    lappend times $seconds
                    set seconds [clock scan -$period -base $seconds]
                }
                foreach seconds [lsort -integer $times] value [set datasets::${set}(values)] {
                    if {($seconds >= $startSeconds) && ($seconds <= $endSeconds)} {
                        lappend list [clock format $seconds -format $dateTimeFormat] $value
                    }
                }
            }
        } else {                                                                                              ;# discrete timestamps
            foreach seconds [set datasets::${set}(times)] value [set datasets::${set}(values)] {
                if {($seconds >= $startSeconds) && ($seconds <= $endSeconds)} {
                    lappend list [clock format $seconds -format $dateTimeFormat] $value
                }
            }
        }
        if {$last} {
            return [lrange $list end-1 end]
        } else {
            return $list
        }
    }

    proc range {set {startSeconds {}} {endSeconds {}}} {     ;# replaces database procedure (not optimized but of little importance)
        variable minimumSeconds
        variable maximumSeconds

        if {$startSeconds eq ""} {set startSeconds $minimumSeconds}
        if {$endSeconds eq ""} {set endSeconds $maximumSeconds}
        set list [history $set $startSeconds $endSeconds 0]
        return [list [lindex $list 0] [lindex $list end-1]]
    }


    array set airmiles {
        description {Passenger Miles on Commercial US Airlines, 1937-1960}
        start 1937-01-01 period {1 year}
        values {412 480 683 1052 1385 1418 1634 2178 3362 5948 6109 5981 6753 8003 10566 12528 14760 16769 19819 22362 25340 25343 29269 30514}
    }

    array set AirPassengers {
        description {Monthly Airline Passenger Numbers 1949-1960}
        start 1949-01-01 period {1 month}
        values {112 118 132 129 121 135 148 148 136 119 104 118 115 126 141 135 125 149 170 170 158 133 114 140 145 150 178 163 172 178 199 199 184 162 146 166 171 180 193 181 183 218 230 242 209 191 172 194 196 196 236 235 229 243 264 272 237 211 180 201 204 188 235 227 234 264 302 293 259 229 203 229 242 233 267 269 270 315 364 347 312 274 237 278 284 277 317 313 318 374 413 405 355 306 271 306 315 301 356 348 355 422 465 467 404 347 305 336 340 318 362 348 363 435 491 505 404 359 310 337 360 342 406 396 420 472 548 559 463 407 362 405 417 391 419 461 472 535 622 606 508 461 390 432}
    }

    array set austres {
        description {Quarterly Time Series of the Number of Australian Residents}
        start 1971-04-01 period {3 months}
        values {13067.3 13130.5 13198.4 13254.2 13303.7 13353.9 13409.3 13459.2 13504.5 13552.6 13614.3 13669.5 13722.6 13772.1 13832 13862.6 13893 13926.8 13968.9 14004.7 14033.1 14066 14110.1 14155.6 14192.2 14231.7 14281.5 14330.3 14359.3 14396.6 14430.8 14478.4 14515.7 14554.9 14602.5 14646.4 14695.4 14746.6 14807.4 14874.4 14923.3 14988.7 15054.1 15121.7 15184.2 15239.3 15288.9 15346.2 15393.5 15439 15483.5 15531.5 15579.4 15628.5 15677.3 15736.7 15788.3 15839.7 15900.6 15961.5 16018.3 16076.9 16139 16203 16263.3 16327.9 16398.9 16478.3 16538.2 16621.6 16697 16777.2 16833.1 16891.6 16956.8 17026.3 17085.4 17106.9 17169.4 17239.4 17292 17354.2 17414.2 17447.3 17482.6 17526 17568.7 17627.1 17661.5}
    }


    # note: there is no start date for this series:
    array set BJsales {
        description {Sales Data with Leading Indicator}
        period {1 month}
        values {200.1 199.5 199.4 198.9 199 200.2 198.6 200 200.3 201.2 201.6 201.5 201.5 203.5 204.9 207.1 210.5 210.5 209.8 208.8 209.5 213.2 213.7 215.1 218.7 219.8 220.5 223.8 222.8 223.8 221.7 222.3 220.8 219.4 220.1 220.6 218.9 217.8 217.7 215 215.3 215.9 216.7 216.7 217.7 218.7 222.9 224.9 222.2 220.7 220 218.7 217 215.9 215.8 214.1 212.3 213.9 214.6 213.6 212.1 211.4 213.1 212.9 213.3 211.5 212.3 213 211 210.7 210.1 211.4 210 209.7 208.8 208.8 208.8 210.6 211.9 212.8 212.5 214.8 215.3 217.5 218.8 220.7 222.2 226.7 228.4 233.2 235.7 237.1 240.6 243.8 245.3 246 246.3 247.7 247.6 247.8 249.4 249 249.9 250.5 251.5 249 247.6 248.8 250.4 250.7 253 253.7 255 256.2 256 257.4 260.4 260 261.3 260.4 261.6 260.8 259.8 259 258.9 257.4 257.7 257.9 257.4 257.3 257.6 258.9 257.8 257.7 257.2 257.5 256.8 257.5 257 257.6 257.3 257.5 259.6 261.1 262.9 263.3 262.8 261.8 262.2 262.7}
    }

    array set co2 {
        description {Mauna Loa Atmospheric CO2 Concentration}
        start 1959-01-01 period {1 month}
        values {315.42 316.31 316.5 317.56 318.13 318 316.39 314.65 313.68 313.18 314.66 315.43 316.27 316.81 317.42 318.87 319.87 319.43 318.01 315.74 314 313.68 314.84 316.03 316.73 317.54 318.38 319.31 320.42 319.61 318.42 316.63 314.83 315.16 315.94 316.85 317.78 318.4 319.53 320.42 320.85 320.45 319.45 317.25 316.11 315.27 316.53 317.53 318.58 318.92 319.7 321.22 322.08 321.31 319.58 317.61 316.05 315.83 316.91 318.2 319.41 320.07 320.74 321.4 322.06 321.73 320.27 318.54 316.54 316.71 317.53 318.55 319.27 320.28 320.73 321.97 322 321.71 321.05 318.71 317.66 317.14 318.7 319.25 320.46 321.43 322.23 323.54 323.91 323.59 322.24 320.2 318.48 317.94 319.63 320.87 322.17 322.34 322.88 324.25 324.83 323.93 322.38 320.76 319.1 319.24 320.56 321.8 322.4 322.99 323.73 324.86 325.4 325.2 323.98 321.95 320.18 320.09 321.16 322.74 323.83 324.26 325.47 326.5 327.21 326.54 325.72 323.5 322.22 321.62 322.69 323.95 324.89 325.82 326.77 327.97 327.91 327.5 326.18 324.53 322.93 322.9 323.85 324.96 326.01 326.51 327.01 327.62 328.76 328.4 327.2 325.27 323.2 323.4 324.63 325.85 326.6 327.47 327.58 329.56 329.9 328.92 327.88 326.16 324.68 325.04 326.34 327.39 328.37 329.4 330.14 331.33 332.31 331.9 330.7 329.15 327.35 327.02 327.99 328.48 329.18 330.55 331.32 332.48 332.92 332.08 331.01 329.23 327.27 327.21 328.29 329.41 330.23 331.25 331.87 333.14 333.8 333.43 331.73 329.9 328.4 328.17 329.32 330.59 331.58 332.39 333.33 334.41 334.71 334.17 332.89 330.77 329.14 328.78 330.14 331.52 332.75 333.24 334.53 335.9 336.57 336.1 334.76 332.59 331.42 330.98 332.24 333.68 334.8 335.22 336.47 337.59 337.84 337.72 336.37 334.51 332.6 332.38 333.75 334.78 336.05 336.59 337.79 338.71 339.3 339.12 337.56 335.92 333.75 333.7 335.12 336.56 337.84 338.19 339.91 340.6 341.29 341 339.39 337.43 335.72 335.84 336.93 338.04 339.06 340.3 341.21 342.33 342.74 342.08 340.32 338.26 336.52 336.68 338.19 339.44 340.57 341.44 342.53 343.39 343.96 343.18 341.88 339.65 337.81 337.69 339.09 340.32 341.2 342.35 342.93 344.77 345.58 345.14 343.81 342.21 339.69 339.82 340.98 342.82 343.52 344.33 345.11 346.88 347.25 346.62 345.22 343.11 340.9 341.18 342.8 344.04 344.79 345.82 347.25 348.17 348.74 348.07 346.38 344.51 342.92 342.62 344.06 345.38 346.11 346.78 347.68 349.37 350.03 349.37 347.76 345.73 344.68 343.99 345.48 346.72 347.84 348.29 349.23 350.8 351.66 351.07 349.33 347.92 346.27 346.18 347.64 348.78 350.25 351.54 352.05 353.41 354.04 353.62 352.22 350.27 348.55 348.72 349.91 351.18 352.6 352.92 353.53 355.26 355.52 354.97 353.75 351.52 349.64 349.83 351.14 352.37 353.5 354.55 355.23 356.04 357 356.07 354.67 352.76 350.82 351.04 352.69 354.07 354.59 355.63 357.03 358.48 359.22 358.12 356.06 353.92 352.05 352.11 353.64 354.89 355.88 356.63 357.72 359.07 359.58 359.17 356.94 354.92 352.94 353.23 354.09 355.33 356.63 357.1 358.32 359.41 360.23 359.55 357.53 355.48 353.67 353.95 355.3 356.78 358.34 358.89 359.95 361.25 361.67 360.94 359.55 357.49 355.84 356 357.59 359.05 359.98 361.03 361.66 363.48 363.82 363.3 361.94 359.5 358.11 357.8 359.61 360.74 362.09 363.29 364.06 364.76 365.45 365.01 363.7 361.54 359.51 359.65 360.8 362.38 363.23 364.06 364.61 366.4 366.84 365.68 364.52 362.57 360.24 360.83 362.49 364.34}
    }

    # note: times are in seconds since 1970 for these irregular series
    set EuStockMarkets(times) {678326256 678452400 678578544 678704688 678830832 678925440 679051584 679177728 679303872 679430016 679556160 679650768 679776912 679903056 680029200 680155344 680281488 680407632 680502240 680628384 680754528 680880672 681006816 681132960 681227568 681353712 681479856 681606000 681732144 681858288 681984432 682079040 682205184 682331328 682457472 682583616 682709760 682804368 682930512 683056656 683182800 683308944 683435088 683561232 683655840 683781984 683908128 684034272 684160416 684286560 684381168 684507312 684633456 684759600 684885744 685011888 685138032 685232640 685358784 685484928 685611072 685737216 685863360 685957968 686084112 686210256 686336400 686462544 686588688 686714832 686809440 686935584 687061728 687187872 687314016 687440160 687534768 687660912 687787056 687913200 688039344 688165488 688291632 688386240 688512384 688638528 688764672 688890816 689016960 689111568 689237712 689363856 689490000 689616144 689742288 689868432 689963040 690089184 690215328 690341472 690467616 690593760 690688368 690814512 690940656 691066800 691192944 691319088 691445232 691539840 691665984 691792128 691918272 692044416 692170560 692265168 692391312 692517456 692643600 692769744 692895888 693022032 693116640 693242784 693368928 693495072 693621216 693747360 693841968 693968112 694094256 694220400 694346890 694473379 694599869 694694736 694821226 694947715 695074205 695200694 695327184 695422051 695548541 695675030 695801520 695928010 696054499 696180989 696275856 696402346 696528835 696655325 696781814 696908304 697003171 697129661 697256150 697382640 697509130 697635619 697762109 697856976 697983466 698109955 698236445 698362934 698489424 698584291 698710781 698837270 698963760 699090250 699216739 699343229 699438096 699564586 699691075 699817565 699944054 700070544 700165411 700291901 700418390 700544880 700671370 700797859 700924349 701019216 701145706 701272195 701398685 701525174 701651664 701746531 701873021 701999510 702126000 702252490 702378979 702505469 702600336 702726826 702853315 702979805 703106294 703232784 703327651 703454141 703580630 703707120 703833610 703960099 704086589 704181456 704307946 704434435 704560925 704687414 704813904 704908771 705035261 705161750 705288240 705414730 705541219 705667709 705762576 705889066 706015555 706142045 706268534 706395024 706489891 706616381 706742870 706869360 706995850 707122339 707248829 707343696 707470186 707596675 707723165 707849654 707976144 708071011 708197501 708323990 708450480 708576970 708703459 708829949 708924816 709051306 709177795 709304285 709430774 709557264 709652131 709778621 709905110 710031600 710158090 710284579 710411069 710505936 710632426 710758915 710885405 711011894 711138384 711233251 711359741 711486230 711612720 711739210 711865699 711992189 712087056 712213546 712340035 712466525 712593014 712719504 712814371 712940861 713067350 713193840 713320330 713446819 713573309 713668176 713794666 713921155 714047645 714174134 714300624 714395491 714521981 714648470 714774960 714901450 715027939 715154429 715249296 715375786 715502275 715628765 715755254 715881744 715976611 716103101 716229590 716356080 716482570 716609059 716735549 716830416 716956906 717083395 717209885 717336374 717462864 717557731 717684221 717810710 717937200 718063690 718190179 718316669 718411536 718538026 718664515 718791005 718917494 719043984 719138851 719265341 719391830 719518320 719644810 719771299 719897789 719992656 720119146 720245635 720372125 720498614 720625104 720719971 720846461 720972950 721099440 721225930 721352419 721478909 721573776 721700266 721826755 721953245 722079734 722206224 722301091 722427581 722554070 722680560 722807050 722933539 723060029 723154896 723281386 723407875 723534365 723660854 723787344 723882211 724008701 724135190 724261680 724388170 724514659 724641149 724736016 724862506 724988995 725115485 725241974 725368464 725463331 725589821 725716310 725842800 725968944 726095088 726221232 726315840 726441984 726568128 726694272 726820416 726946560 727041168 727167312 727293456 727419600 727545744 727671888 727798032 727892640 728018784 728144928 728271072 728397216 728523360 728617968 728744112 728870256 728996400 729122544 729248688 729374832 729469440 729595584 729721728 729847872 729974016 730100160 730194768 730320912 730447056 730573200 730699344 730825488 730951632 731046240 731172384 731298528 731424672 731550816 731676960 731771568 731897712 732023856 732150000 732276144 732402288 732528432 732623040 732749184 732875328 733001472 733127616 733253760 733348368 733474512 733600656 733726800 733852944 733979088 734105232 734199840 734325984 734452128 734578272 734704416 734830560 734925168 735051312 735177456 735303600 735429744 735555888 735682032 735776640 735902784 736028928 736155072 736281216 736407360 736501968 736628112 736754256 736880400 737006544 737132688 737258832 737353440 737479584 737605728 737731872 737858016 737984160 738078768 738204912 738331056 738457200 738583344 738709488 738835632 738930240 739056384 739182528 739308672 739434816 739560960 739655568 739781712 739907856 740034000 740160144 740286288 740412432 740507040 740633184 740759328 740885472 741011616 741137760 741232368 741358512 741484656 741610800 741736944 741863088 741989232 742083840 742209984 742336128 742462272 742588416 742714560 742809168 742935312 743061456 743187600 743313744 743439888 743566032 743660640 743786784 743912928 744039072 744165216 744291360 744385968 744512112 744638256 744764400 744890544 745016688 745142832 745237440 745363584 745489728 745615872 745742016 745868160 745962768 746088912 746215056 746341200 746467344 746593488 746719632 746814240 746940384 747066528 747192672 747318816 747444960 747539568 747665712 747791856 747918000 748044144 748170288 748296432 748391040 748517184 748643328 748769472 748895616 749021760 749116368 749242512 749368656 749494800 749620944 749747088 749873232 749967840 750093984 750220128 750346272 750472416 750598560 750693168 750819312 750945456 751071600 751197744 751323888 751450032 751544640 751670784 751796928 751923072 752049216 752175360 752269968 752396112 752522256 752648400 752774544 752900688 753026832 753121440 753247584 753373728 753499872 753626016 753752160 753846768 753972912 754099056 754225200 754351344 754477488 754603632 754698240 754824384 754950528 755076672 755202816 755328960 755423568 755549712 755675856 755802000 755928144 756054288 756180432 756275040 756401184 756527328 756653472 756779616 756905760 757000368 757126512 757252656 757378800 757504944 757631088 757757232 757851840 757977984 758104128 758230272 758356416 758482560 758577168 758703312 758829456 758955600 759081744 759207888 759334032 759428640 759554784 759680928 759807072 759933216 760059360 760153968 760280112 760406256 760532400 760658544 760784688 760910832 761005440 761131584 761257728 761383872 761510016 761636160 761730768 761856912 761983056 762109200 762235344 762361488 762487632 762582240 762708384 762834528 762960672 763086816 763212960 763307568 763433712 763559856 763686000 763812144 763938288 764064432 764159040 764285184 764411328 764537472 764663616 764789760 764884368 765010512 765136656 765262800 765388944 765515088 765641232 765735840 765861984 765988128 766114272 766240416 766366560 766461168 766587312 766713456 766839600 766965744 767091888 767218032 767312640 767438784 767564928 767691072 767817216 767943360 768037968 768164112 768290256 768416400 768542544 768668688 768794832 768889440 769015584 769141728 769267872 769394016 769520160 769614768 769740912 769867056 769993200 770119344 770245488 770371632 770466240 770592384 770718528 770844672 770970816 771096960 771191568 771317712 771443856 771570000 771696144 771822288 771948432 772043040 772169184 772295328 772421472 772547616 772673760 772768368 772894512 773020656 773146800 773272944 773399088 773525232 773619840 773745984 773872128 773998272 774124416 774250560 774345168 774471312 774597456 774723600 774849744 774975888 775102032 775196640 775322784 775448928 775575072 775701216 775827360 775921968 776048112 776174256 776300400 776426544 776552688 776678832 776773440 776899584 777025728 777151872 777278016 777404160 777498768 777624912 777751056 777877200 778003344 778129488 778255632 778350240 778476384 778602528 778728672 778854816 778980960 779075568 779201712 779327856 779454000 779580144 779706288 779832432 779927040 780053184 780179328 780305472 780431616 780557760 780652368 780778512 780904656 781030800 781156944 781283088 781409232 781503840 781629984 781756128 781882272 782008416 782134560 782229168 782355312 782481456 782607600 782733744 782859888 782986032 783080640 783206784 783332928 783459072 783585216 783711360 783805968 783932112 784058256 784184400 784310544 784436688 784562832 784657440 784783584 784909728 785035872 785162016 785288160 785382768 785508912 785635056 785761200 785887344 786013488 786139632 786234240 786360384 786486528 786612672 786738816 786864960 786959568 787085712 787211856 787338000 787464144 787590288 787716432 787811040 787937184 788063328 788189472 788315616 788441760 788536368 788662512 788788656 788914800 789040944 789167088 789293232 789387840 789513984 789640128 789766272 789892416 790018560 790113168 790239312 790365456 790491600 790617744 790743888 790870032 790964640 791090784 791216928 791343072 791469216 791595360 791689968 791816112 791942256 792068400 792194544 792320688 792446832 792541440 792667584 792793728 792919872 793046016 793172160 793266768 793392912 793519056 793645200 793771344 793897488 794023632 794118240 794244384 794370528 794496672 794622816 794748960 794843568 794969712 795095856 795222000 795348144 795474288 795600432 795695040 795821184 795947328 796073472 796199616 796325760 796420368 796546512 796672656 796798800 796924944 797051088 797177232 797271840 797397984 797524128 797650272 797776416 797902560 797997168 798123312 798249456 798375600 798501744 798627888 798754032 798848640 798974784 799100928 799227072 799353216 799479360 799573968 799700112 799826256 799952400 800078544 800204688 800330832 800425440 800551584 800677728 800803872 800930016 801056160 801150768 801276912 801403056 801529200 801655344 801781488 801907632 802002240 802128384 802254528 802380672 802506816 802632960 802727568 802853712 802979856 803106000 803232144 803358288 803484432 803579040 803705184 803831328 803957472 804083616 804209760 804304368 804430512 804556656 804682800 804808944 804935088 805061232 805155840 805281984 805408128 805534272 805660416 805786560 805881168 806007312 806133456 806259600 806385744 806511888 806638032 806732640 806858784 806984928 807111072 807237216 807363360 807457968 807584112 807710256 807836400 807962544 808088688 808214832 808309440 808435584 808561728 808687872 808814016 808940160 809034768 809160912 809287056 809413200 809539344 809665488 809791632 809886240 810012384 810138528 810264672 810390816 810516960 810611568 810737712 810863856 810990000 811116144 811242288 811368432 811463040 811589184 811715328 811841472 811967616 812093760 812188368 812314512 812440656 812566800 812692944 812819088 812945232 813039840 813165984 813292128 813418272 813544416 813670560 813765168 813891312 814017456 814143600 814269744 814395888 814522032 814616640 814742784 814868928 814995072 815121216 815247360 815341968 815468112 815594256 815720400 815846544 815972688 816098832 816193440 816319584 816445728 816571872 816698016 816824160 816918768 817044912 817171056 817297200 817423344 817549488 817675632 817770240 817896384 818022528 818148672 818274816 818400960 818495568 818621712 818747856 818874000 819000144 819126288 819252432 819347040 819473184 819599328 819725472 819851616 819977760 820072368 820198512 820324656 820450800 820577290 820703779 820830269 820925136 821051626 821178115 821304605 821431094 821557584 821652451 821778941 821905430 822031920 822158410 822284899 822411389 822506256 822632746 822759235 822885725 823012214 823138704 823233571 823360061 823486550 823613040 823739530 823866019 823992509 824087376 824213866 824340355 824466845 824593334 824719824 824814691 824941181 825067670 825194160 825320650 825447139 825573629 825668496 825794986 825921475 826047965 826174454 826300944 826395811 826522301 826648790 826775280 826901770 827028259 827154749 827249616 827376106 827502595 827629085 827755574 827882064 827976931 828103421 828229910 828356400 828482890 828609379 828735869 828830736 828957226 829083715 829210205 829336694 829463184 829558051 829684541 829811030 829937520 830064010 830190499 830316989 830411856 830538346 830664835 830791325 830917814 831044304 831139171 831265661 831392150 831518640 831645130 831771619 831898109 831992976 832119466 832245955 832372445 832498934 832625424 832720291 832846781 832973270 833099760 833226250 833352739 833479229 833574096 833700586 833827075 833953565 834080054 834206544 834301411 834427901 834554390 834680880 834807370 834933859 835060349 835155216 835281706 835408195 835534685 835661174 835787664 835882531 836009021 836135510 836262000 836388490 836514979 836641469 836736336 836862826 836989315 837115805 837242294 837368784 837463651 837590141 837716630 837843120 837969610 838096099 838222589 838317456 838443946 838570435 838696925 838823414 838949904 839044771 839171261 839297750 839424240 839550730 839677219 839803709 839898576 840025066 840151555 840278045 840404534 840531024 840625891 840752381 840878870 841005360 841131850 841258339 841384829 841479696 841606186 841732675 841859165 841985654 842112144 842207011 842333501 842459990 842586480 842712970 842839459 842965949 843060816 843187306 843313795 843440285 843566774 843693264 843788131 843914621 844041110 844167600 844294090 844420579 844547069 844641936 844768426 844894915 845021405 845147894 845274384 845369251 845495741 845622230 845748720 845875210 846001699 846128189 846223056 846349546 846476035 846602525 846729014 846855504 846950371 847076861 847203350 847329840 847456330 847582819 847709309 847804176 847930666 848057155 848183645 848310134 848436624 848531491 848657981 848784470 848910960 849037450 849163939 849290429 849385296 849511786 849638275 849764765 849891254 850017744 850112611 850239101 850365590 850492080 850618570 850745059 850871549 850966416 851092906 851219395 851345885 851472374 851598864 851693731 851820221 851946710 852073200 852199344 852325488 852451632 852546240 852672384 852798528 852924672 853050816 853176960 853271568 853397712 853523856 853650000 853776144 853902288 854028432 854123040 854249184 854375328 854501472 854627616 854753760 854848368 854974512 855100656 855226800 855352944 855479088 855605232 855699840 855825984 855952128 856078272 856204416 856330560 856425168 856551312 856677456 856803600 856929744 857055888 857182032 857276640 857402784 857528928 857655072 857781216 857907360 858001968 858128112 858254256 858380400 858506544 858632688 858758832 858853440 858979584 859105728 859231872 859358016 859484160 859578768 859704912 859831056 859957200 860083344 860209488 860335632 860430240 860556384 860682528 860808672 860934816 861060960 861155568 861281712 861407856 861534000 861660144 861786288 861912432 862007040 862133184 862259328 862385472 862511616 862637760 862732368 862858512 862984656 863110800 863236944 863363088 863489232 863583840 863709984 863836128 863962272 864088416 864214560 864309168 864435312 864561456 864687600 864813744 864939888 865066032 865160640 865286784 865412928 865539072 865665216 865791360 865885968 866012112 866138256 866264400 866390544 866516688 866642832 866737440 866863584 866989728 867115872 867242016 867368160 867462768 867588912 867715056 867841200 867967344 868093488 868219632 868314240 868440384 868566528 868692672 868818816 868944960 869039568 869165712 869291856 869418000 869544144 869670288 869796432 869891040 870017184 870143328 870269472 870395616 870521760 870616368 870742512 870868656 870994800 871120944 871247088 871373232 871467840 871593984 871720128 871846272 871972416 872098560 872193168 872319312 872445456 872571600 872697744 872823888 872950032 873044640 873170784 873296928 873423072 873549216 873675360 873769968 873896112 874022256 874148400 874274544 874400688 874526832 874621440 874747584 874873728 874999872 875126016 875252160 875346768 875472912 875599056 875725200 875851344 875977488 876103632 876198240 876324384 876450528 876576672 876702816 876828960 876923568 877049712 877175856 877302000 877428144 877554288 877680432 877775040 877901184 878027328 878153472 878279616 878405760 878500368 878626512 878752656 878878800 879004944 879131088 879257232 879351840 879477984 879604128 879730272 879856416 879982560 880077168 880203312 880329456 880455600 880581744 880707888 880834032 880928640 881054784 881180928 881307072 881433216 881559360 881653968 881780112 881906256 882032400 882158544 882284688 882410832 882505440 882631584 882757728 882883872 883010016 883136160 883230768 883356912 883483056 883609200 883735344 883861488 883987632 884082240 884208384 884334528 884460672 884586816 884712960 884807568 884933712 885059856 885186000 885312144 885438288 885564432 885659040 885785184 885911328 886037472 886163616 886289760 886384368 886510512 886636656 886762800 886888944 887015088 887141232 887235840 887361984 887488128 887614272 887740416 887866560 887961168 888087312 888213456 888339600 888465744 888591888 888718032 888812640 888938784 889064928 889191072 889317216 889443360 889537968 889664112 889790256 889916400 890042544 890168688 890294832 890389440 890515584 890641728 890767872 890894016 891020160 891114768 891240912 891367056 891493200 891619344 891745488 891871632 891966240 892092384 892218528 892344672 892470816 892596960 892691568 892817712 892943856 893070000 893196144 893322288 893448432 893543040 893669184 893795328 893921472 894047616 894173760 894268368 894394512 894520656 894646800 894772944 894899088 895025232 895119840 895245984 895372128 895498272 895624416 895750560 895845168 895971312 896097456 896223600 896349744 896475888 896602032 896696640 896822784 896948928 897075072 897201216 897327360 897421968 897548112 897674256 897800400 897926544 898052688 898178832 898273440 898399584 898525728 898651872 898778016 898904160 898998768 899124912 899251056 899377200 899503344 899629488 899755632 899850240 899976384 900102528 900228672 900354816 900480960 900575568 900701712 900827856 900954000 901080144 901206288 901332432 901427040 901553184 901679328 901805472 901931616 902057760 902152368 902278512 902404656 902530800 902656944 902783088 902909232 903003840 903129984 903256128 903382272 903508416 903634560 903729168 903855312 903981456}
    array set EuStockMarkets.DAX [list\
        description {Daily Closing Prices of DAX Stock Index, 1991-1998}\
        times $EuStockMarkets(times)\
        values {1628.75 1613.63 1606.51 1621.04 1618.16 1610.61 1630.75 1640.17 1635.47 1645.89 1647.84 1638.35 1629.93 1621.49 1624.74 1627.63 1631.99 1621.18 1613.42 1604.95 1605.75 1616.67 1619.29 1620.49 1619.67 1623.07 1613.98 1631.87 1630.37 1633.47 1626.55 1650.43 1650.06 1654.11 1653.60 1501.82 1524.28 1603.65 1622.49 1636.68 1652.10 1645.81 1650.36 1651.55 1649.88 1653.52 1657.51 1649.55 1649.09 1646.41 1638.65 1625.80 1628.64 1632.22 1633.65 1631.17 1635.80 1621.27 1624.70 1616.13 1618.12 1627.80 1625.79 1614.80 1612.80 1605.47 1609.32 1607.48 1607.48 1604.89 1589.12 1582.27 1567.99 1568.16 1569.71 1571.74 1585.41 1570.01 1561.89 1565.18 1570.34 1577.00 1590.29 1572.72 1572.07 1579.19 1588.73 1586.01 1579.77 1572.58 1568.09 1578.21 1573.94 1582.06 1610.18 1605.16 1623.84 1615.26 1627.08 1626.97 1605.70 1589.70 1589.70 1603.26 1599.75 1590.86 1603.50 1589.86 1587.92 1571.06 1549.81 1549.36 1554.65 1557.52 1555.31 1559.76 1548.44 1543.99 1550.21 1557.03 1551.78 1562.89 1570.28 1559.26 1545.87 1542.77 1542.77 1542.77 1542.77 1564.27 1577.26 1577.26 1577.26 1598.19 1604.05 1604.69 1593.65 1581.68 1599.14 1613.82 1620.45 1629.51 1663.70 1664.09 1669.29 1685.14 1687.07 1680.13 1671.84 1669.52 1686.71 1685.51 1671.01 1683.06 1685.70 1685.66 1678.77 1685.85 1683.71 1686.59 1683.73 1679.14 1685.03 1680.81 1676.17 1688.46 1696.55 1690.24 1711.35 1711.29 1729.86 1716.63 1743.36 1745.17 1746.76 1749.29 1763.86 1762.27 1762.29 1746.77 1753.50 1753.21 1739.88 1723.92 1734.42 1723.13 1732.92 1729.89 1725.74 1730.90 1714.17 1716.20 1719.06 1718.21 1698.84 1714.76 1718.35 1706.69 1723.37 1716.18 1738.78 1737.41 1714.77 1724.24 1733.77 1729.96 1734.46 1744.35 1746.88 1746.88 1746.88 1747.47 1753.10 1745.17 1745.72 1742.92 1731.68 1731.18 1728.09 1728.09 1731.29 1733.82 1745.78 1752.57 1748.13 1750.70 1747.91 1745.79 1735.34 1719.92 1763.59 1766.76 1785.40 1783.56 1804.42 1812.33 1799.51 1792.80 1792.80 1806.36 1798.23 1800.62 1786.19 1791.35 1789.05 1789.05 1784.71 1789.45 1779.74 1786.97 1773.25 1781.62 1773.75 1773.75 1776.34 1770.72 1772.39 1762.55 1764.35 1752.83 1755.98 1754.95 1759.90 1759.84 1776.50 1769.98 1766.98 1752.29 1760.17 1750.32 1731.44 1735.51 1733.84 1730.78 1699.46 1652.71 1654.09 1636.81 1622.81 1613.36 1617.78 1617.18 1637.62 1622.20 1608.49 1605.11 1609.61 1624.94 1618.07 1611.96 1578.95 1561.39 1547.87 1548.63 1560.16 1554.76 1531.87 1526.14 1509.03 1530.03 1484.97 1464.03 1475.11 1516.12 1519.69 1529.97 1516.44 1515.53 1543.89 1534.72 1538.66 1536.71 1523.83 1527.10 1530.20 1601.50 1580.29 1595.09 1579.47 1600.59 1566.00 1557.01 1542.74 1536.30 1510.66 1481.03 1483.83 1470.09 1484.78 1475.41 1402.34 1421.49 1434.61 1446.32 1437.65 1441.57 1471.64 1453.95 1453.79 1458.02 1479.59 1504.89 1496.54 1511.00 1528.86 1534.02 1536.60 1508.19 1493.54 1489.68 1482.44 1483.34 1470.57 1484.84 1487.71 1508.63 1515.27 1509.84 1542.28 1541.79 1542.48 1550.27 1550.27 1543.37 1547.84 1523.62 1526.68 1513.42 1523.02 1529.69 1545.12 1546.82 1528.12 1530.65 1526.25 1519.48 1506.65 1504.30 1480.65 1476.70 1478.07 1479.62 1477.55 1472.59 1495.60 1517.45 1520.93 1527.06 1527.06 1527.06 1547.51 1545.82 1538.43 1538.43 1538.43 1538.04 1554.03 1551.17 1538.37 1529.10 1522.26 1533.79 1510.18 1526.91 1555.52 1581.49 1572.61 1572.69 1580.64 1593.35 1571.28 1575.59 1561.78 1572.68 1574.04 1590.33 1584.14 1605.91 1615.98 1643.83 1646.85 1639.12 1642.80 1659.07 1649.64 1674.93 1651.60 1656.35 1670.90 1683.30 1679.41 1658.09 1652.92 1661.96 1680.02 1691.37 1701.46 1690.48 1685.46 1686.15 1702.27 1711.91 1714.48 1708.65 1688.74 1705.05 1700.28 1689.71 1696.38 1686.57 1656.59 1653.20 1666.72 1663.27 1667.26 1675.18 1686.64 1676.84 1670.04 1657.06 1658.36 1667.64 1654.60 1658.13 1658.13 1658.13 1671.54 1674.95 1674.67 1678.65 1687.14 1680.06 1666.49 1680.01 1656.03 1643.53 1636.59 1630.88 1618.60 1626.83 1632.00 1619.92 1628.88 1617.74 1607.70 1616.45 1613.46 1632.99 1636.02 1632.35 1630.37 1619.26 1606.64 1606.64 1613.98 1608.58 1623.05 1617.18 1633.18 1627.21 1627.21 1625.59 1628.53 1630.56 1638.47 1660.88 1662.28 1679.69 1679.69 1685.85 1686.44 1684.57 1689.93 1681.47 1687.14 1697.26 1698.33 1690.96 1692.16 1699.52 1712.33 1703.05 1700.93 1698.36 1697.39 1694.83 1705.66 1739.48 1798.63 1798.36 1808.74 1806.52 1815.63 1807.12 1829.36 1835.09 1826.45 1821.28 1828.53 1830.61 1859.49 1846.02 1832.20 1823.40 1823.07 1818.10 1857.36 1861.22 1870.80 1878.94 1870.24 1864.51 1894.56 1908.69 1917.69 1903.44 1918.75 1930.29 1937.77 1909.53 1893.48 1907.65 1915.59 1890.58 1909.54 1929.56 1931.88 1923.67 1928.63 1920.43 1911.15 1878.77 1870.32 1870.46 1868.28 1874.38 1869.25 1852.81 1862.62 1884.67 1922.69 1922.05 1899.54 1902.14 1890.18 1914.40 1915.61 1908.97 1910.23 1920.46 1934.99 1973.45 1994.09 1991.95 2015.71 2009.28 2004.11 1999.60 1995.35 2023.26 2032.25 2036.93 2029.87 2048.05 2083.62 2061.18 2048.75 2040.97 2053.66 2064.98 2068.51 2086.40 2086.22 2057.10 2013.65 2007.50 2040.74 2020.11 2021.81 2023.06 2047.20 2070.17 2070.17 2075.99 2072.13 2020.36 2027.99 2036.45 2057.80 2045.25 2052.09 2052.92 2089.77 2099.76 2128.30 2118.01 2127.39 2165.95 2166.58 2165.50 2163.83 2128.07 2129.52 2144.03 2154.76 2188.18 2183.78 2209.17 2227.63 2227.63 2266.70 2236.91 2229.62 2255.29 2255.29 2274.62 2249.85 2233.61 2220.63 2224.95 2225.00 2228.10 2182.06 2142.37 2151.05 2115.56 2130.35 2132.52 2098.36 2073.94 2107.29 2090.78 2128.66 2123.31 2156.61 2192.60 2181.88 2184.05 2137.08 2143.90 2095.11 2099.57 2116.43 2119.69 2108.77 2101.93 2130.71 2135.25 2162.29 2133.85 2108.06 2113.64 2140.25 2082.90 2075.33 2103.24 2057.20 2018.69 2044.45 2076.76 2132.12 2125.47 2118.01 2124.51 2101.89 2169.40 2178.91 2168.11 2160.45 2140.39 2130.55 2141.70 2162.96 2144.36 2144.00 2167.72 2162.82 2151.84 2142.88 2142.88 2142.88 2177.09 2184.89 2202.57 2203.18 2224.85 2211.19 2215.19 2198.24 2211.92 2218.37 2193.89 2194.09 2194.41 2218.13 2208.68 2241.36 2256.98 2237.82 2252.51 2266.72 2261.71 2241.85 2249.78 2233.55 2218.77 2241.34 2248.02 2248.02 2257.33 2272.96 2268.11 2254.21 2245.79 2238.97 2238.97 2188.01 2137.56 2146.00 2112.80 2129.76 2137.34 2113.62 2120.23 2158.88 2163.59 2131.80 2143.93 2131.14 2143.58 2084.42 2088.44 2073.21 2047.29 2031.80 1986.42 1957.08 2004.93 2032.52 2005.07 2000.48 2022.25 2042.45 2020.85 2040.69 2061.70 2034.64 2031.33 2049.10 2047.83 2069.46 2048.57 2051.25 2070.71 2103.54 2116.96 2129.86 2120.97 2126.75 2148.23 2144.21 2163.32 2135.93 2134.12 2152.19 2161.50 2193.63 2190.83 2176.66 2188.81 2182.32 2160.56 2166.51 2156.31 2133.74 2134.34 2152.65 2166.56 2151.44 2143.84 2113.37 2121.25 2132.98 2153.48 2190.58 2215.72 2205.82 2207.09 2185.78 2197.38 2173.60 2158.07 2167.68 2178.10 2155.58 2155.81 2157.15 2118.17 2129.36 2097.45 2100.55 2059.15 2067.17 2072.81 2097.33 2057.83 2056.89 2070.36 2016.08 2002.30 2002.30 1988.67 1946.49 1965.41 1977.67 2048.56 2087.71 2072.68 2108.08 2118.52 2095.58 2069.58 2055.94 2048.15 2016.60 2022.64 1995.85 2009.45 2026.37 2064.86 2061.58 2066.18 2039.91 2061.37 2069.39 2043.30 2056.36 2090.78 2098.04 2073.67 2091.04 2112.21 2112.21 2091.94 2089.48 2097.20 2073.29 2040.05 2058.79 2056.27 2050.82 2036.24 2057.08 2045.54 2042.38 2067.26 2046.99 2044.04 2041.85 2024.19 2003.64 2019.13 2040.94 2054.23 2069.90 2066.59 2080.16 2095.50 2102.25 2100.98 2100.98 2110.77 2097.34 2074.68 2097.51 2079.19 2068.92 2072.90 2051.46 2058.20 2053.41 2062.08 2061.76 2059.68 2064.14 2088.25 2081.39 2085.62 2079.60 2050.86 2013.03 2027.67 2026.97 2040.79 2030.56 2026.68 2024.82 2047.44 2038.46 2077.94 2085.08 2093.01 2087.78 2117.80 2127.56 2118.96 2128.33 2135.49 2112.06 2119.29 2101.98 2101.82 2096.26 2126.44 2117.59 2106.05 2097.85 2127.21 2116.64 2094.16 2069.95 2041.26 2029.38 1989.20 2008.85 1974.14 2022.50 1995.22 2016.15 1979.52 1984.99 1984.15 1978.07 1935.08 1935.87 1928.82 1911.70 1918.85 1949.76 1914.69 1934.96 1965.68 1972.59 1976.52 1978.71 1980.81 1979.26 1991.08 1982.99 1982.99 1982.99 1954.62 1943.88 1955.33 1987.70 1978.97 2007.57 2028.52 2024.25 2017.95 2017.95 2036.47 2037.99 2034.15 2021.22 2030.65 2050.66 2064.41 2086.41 2102.40 2092.00 2109.36 2088.99 2083.80 2064.42 2096.87 2083.55 2105.88 2105.88 2065.71 2069.26 2081.46 2099.95 2119.24 2132.72 2132.72 2154.17 2136.72 2137.21 2107.16 2127.79 2124.25 2124.84 2130.78 2125.06 2154.13 2140.36 2144.88 2144.88 2144.88 2137.36 2128.99 2091.30 2094.68 2089.04 2099.68 2110.13 2117.63 2111.35 2163.37 2184.39 2194.15 2200.32 2193.72 2191.25 2201.36 2183.71 2195.31 2183.52 2196.54 2232.02 2232.23 2237.73 2239.36 2217.91 2222.51 2211.26 2249.60 2234.86 2236.68 2244.56 2236.89 2222.28 2236.72 2228.44 2208.41 2236.73 2251.64 2253.93 2265.86 2269.60 2258.97 2262.66 2253.91 2257.34 2240.31 2230.27 2251.30 2234.23 2242.76 2263.08 2266.57 2266.37 2263.27 2266.77 2268.26 2266.07 2300.72 2300.29 2305.58 2289.49 2306.66 2294.15 2275.72 2204.44 2212.97 2232.02 2190.90 2184.24 2201.27 2197.01 2197.01 2211.60 2191.54 2176.35 2157.41 2137.01 2153.56 2159.73 2206.11 2193.85 2185.52 2197.57 2176.09 2148.12 2116.86 2114.88 2140.74 2142.74 2112.01 2149.71 2165.76 2182.11 2180.49 2169.69 2168.57 2174.27 2184.05 2182.47 2174.78 2195.15 2197.34 2196.64 2199.29 2196.59 2204.01 2201.90 2194.53 2197.26 2210.92 2247.97 2242.91 2249.75 2254.95 2266.56 2261.08 2269.34 2274.94 2260.62 2277.70 2277.70 2278.64 2283.26 2286.21 2283.84 2240.24 2241.43 2268.07 2268.35 2280.81 2280.81 2280.81 2280.44 2273.90 2260.69 2260.69 2307.70 2326.18 2332.81 2315.66 2336.76 2351.47 2340.31 2330.98 2353.89 2361.38 2379.43 2375.63 2389.62 2400.58 2391.74 2393.12 2436.68 2437.02 2449.71 2437.70 2456.09 2463.00 2472.53 2459.81 2416.84 2437.98 2442.43 2411.49 2435.07 2428.59 2436.09 2427.77 2426.51 2423.60 2387.60 2373.01 2401.59 2421.93 2449.52 2438.73 2449.09 2486.95 2485.18 2488.85 2486.83 2472.52 2471.38 2479.38 2448.80 2419.72 2415.29 2417.00 2432.46 2458.00 2472.55 2484.74 2491.73 2505.32 2479.84 2514.80 2505.78 2523.81 2502.94 2489.35 2500.75 2508.11 2489.52 2498.75 2498.75 2498.75 2510.81 2525.59 2508.12 2526.74 2540.79 2547.32 2519.82 2524.18 2535.86 2549.27 2549.12 2524.84 2538.68 2539.88 2505.97 2492.63 2492.63 2465.49 2472.43 2473.52 2476.79 2466.21 2470.57 2499.02 2496.33 2528.20 2534.40 2534.40 2552.29 2541.98 2564.12 2548.79 2546.55 2549.53 2549.53 2559.15 2548.53 2535.78 2523.81 2543.99 2550.42 2551.03 2559.02 2536.03 2552.96 2571.10 2569.90 2566.13 2544.90 2549.71 2554.12 2546.04 2532.22 2547.78 2562.19 2578.36 2572.06 2551.80 2570.44 2573.44 2565.32 2566.32 2578.74 2568.79 2561.51 2562.24 2573.00 2561.95 2548.97 2529.50 2475.98 2497.69 2505.56 2498.35 2468.32 2488.82 2459.13 2464.51 2469.51 2472.25 2466.84 2491.50 2504.16 2520.26 2520.52 2527.73 2541.41 2534.63 2532.96 2529.14 2545.65 2544.61 2542.75 2567.96 2556.25 2562.12 2534.44 2559.42 2562.12 2554.42 2563.59 2556.64 2548.84 2534.49 2538.34 2528.73 2526.18 2532.55 2542.74 2549.71 2568.77 2568.70 2588.04 2614.50 2630.24 2626.43 2621.20 2625.46 2641.50 2624.18 2643.42 2663.10 2664.96 2659.86 2655.49 2654.34 2685.29 2685.29 2704.25 2705.21 2702.60 2685.23 2683.52 2692.69 2700.83 2718.73 2705.06 2717.50 2727.56 2733.67 2721.74 2678.89 2681.94 2690.79 2703.33 2675.50 2664.72 2671.40 2670.19 2678.73 2716.16 2735.28 2724.25 2732.29 2730.44 2764.00 2770.61 2784.39 2800.60 2768.68 2781.54 2766.08 2765.29 2769.47 2800.52 2808.62 2793.86 2829.68 2848.84 2853.46 2900.76 2880.89 2894.43 2832.53 2870.30 2890.95 2836.36 2846.94 2815.77 2841.16 2808.50 2814.23 2824.83 2835.54 2844.09 2844.09 2844.09 2844.09 2859.22 2880.07 2880.07 2880.07 2820.81 2863.26 2890.20 2876.34 2904.08 2936.69 2915.81 2956.78 2978.84 2976.56 2996.12 3006.87 2999.19 3000.66 3026.63 3037.28 2982.63 2992.55 3028.27 2997.95 3018.58 3037.70 3064.70 3067.48 3114.73 3124.78 3161.36 3185.72 3191.45 3211.01 3256.86 3249.17 3260.30 3230.83 3209.04 3197.09 3203.79 3180.63 3233.34 3245.02 3272.58 3261.04 3258.74 3345.09 3375.45 3396.55 3419.51 3426.77 3430.95 3382.40 3367.82 3404.29 3337.11 3289.59 3305.72 3247.03 3288.52 3302.57 3374.93 3439.22 3407.83 3407.83 3407.83 3281.46 3210.94 3212.82 3235.35 3342.77 3328.13 3364.76 3352.58 3319.24 3297.52 3369.26 3347.54 3361.80 3361.20 3328.41 3348.90 3366.87 3396.49 3357.57 3372.96 3425.86 3438.09 3438.09 3491.08 3565.69 3548.52 3537.45 3537.45 3533.21 3593.14 3559.29 3588.57 3564.85 3569.26 3569.26 3516.20 3600.40 3575.44 3621.72 3669.31 3665.43 3626.60 3635.38 3562.73 3596.40 3655.59 3651.59 3684.60 3700.53 3668.61 3671.16 3671.87 3737.16 3752.37 3750.02 3721.18 3730.56 3777.56 3788.54 3748.79 3761.07 3819.52 3820.16 3809.92 3766.89 3834.84 3867.53 3939.73 3946.73 4003.35 4030.10 4026.97 4000.65 4074.30 4142.19 4139.68 4223.69 4203.91 4131.94 4139.96 4297.64 4384.82 4320.52 4368.54 4400.30 4377.70 4458.66 4405.52 4336.98 4302.50 4325.86 4364.25 4428.08 4342.31 4333.15 4377.51 4237.06 4195.53 4077.59 4080.55 4190.45 4251.93 4204.81 4090.14 4076.75 3993.70 3992.03 3897.43 3919.79 4001.81 4127.28 4062.13 4093.43 4073.71 4131.26 4104.57 4028.00 3890.24 3796.61 3869.53 3995.69 3970.44 4004.04 3983.06 4096.85 4091.77 4150.95 4104.93 4135.09 4116.52 4154.89 4262.98 4266.17 4266.17 4326.35 4311.13 4267.40 4179.92 4164.62 4225.27 4215.23 4168.62 4149.92 4049.16 4069.25 4172.47 4124.86 3976.38 3981.44 3871.39 3645.69 3806.66 3748.88 3753.66 3847.73 3784.80 3841.39 3813.88 3715.38 3728.37 3734.79 3697.48 3701.94 3676.65 3816.71 3844.14 3876.90 3931.81 3941.91 3832.10 3850.14 3926.93 3961.97 3972.08 4125.92 4096.40 4074.55 4159.72 4191.81 4208.14 4187.13 4116.70 4016.70 4061.91 4029.08 4150.31 4154.57 4162.92 4055.35 4125.54 4132.79 4132.79 4132.79 4132.79 4266.02 4224.30 4224.30 4224.30 4364.32 4416.95 4360.05 4339.98 4293.64 4237.75 4134.64 4150.01 4145.41 4140.22 4216.24 4290.05 4310.83 4250.47 4238.77 4222.16 4266.34 4316.05 4385.29 4444.53 4442.53 4529.88 4529.18 4509.25 4494.72 4536.91 4519.56 4558.62 4552.46 4509.37 4522.42 4535.56 4627.42 4611.66 4581.08 4583.03 4610.66 4604.55 4704.58 4695.78 4693.86 4781.62 4759.62 4690.52 4676.42 4762.71 4828.89 4852.22 4862.41 4838.67 4872.24 4905.59 4945.91 4908.55 4949.91 5045.16 5014.13 5064.35 5114.13 5029.00 5066.90 5069.89 5097.25 5135.35 5179.04 5254.32 5345.89 5309.67 5267.35 5312.25 5312.25 5312.25 5367.98 5359.24 5292.97 5326.63 5407.93 5373.80 5312.28 5262.57 5144.42 5002.71 5110.88 5083.80 5241.23 5241.23 5337.75 5226.20 5264.62 5164.89 5270.61 5348.75 5307.82 5371.99 5374.11 5414.31 5343.66 5441.00 5514.51 5514.51 5530.19 5592.46 5639.89 5466.88 5507.36 5556.99 5556.99 5583.83 5640.42 5605.38 5724.75 5787.05 5773.77 5799.22 5799.22 5631.34 5581.24 5621.71 5742.83 5689.89 5644.22 5648.11 5748.34 5784.40 5886.72 5870.49 5933.73 5841.83 5910.51 5905.15 5961.45 5942.06 5975.88 6018.89 6000.84 6001.24 6023.31 6101.90 6106.10 6108.00 6162.86 6186.09 6184.10 6081.11 6043.82 6040.58 5854.35 5867.52 5828.74 5906.33 5861.19 5774.38 5718.70 5614.77 5528.12 5598.32 5460.43 5285.78 5386.94 5355.03 5473.72}\
    ]
    array set EuStockMarkets.SMI [list\
        description {Daily Closing Prices of SMI Stock Index, 1991-1998}\
        times $EuStockMarkets(times)\
        values {1678.1 1688.5 1678.6 1684.1 1686.6 1671.6 1682.9 1703.6 1697.5 1716.3 1723.8 1730.5 1727.4 1733.3 1734.0 1728.3 1737.1 1723.1 1723.6 1719.0 1721.2 1725.3 1727.2 1727.2 1731.6 1724.1 1716.9 1723.4 1723.0 1728.4 1722.1 1724.5 1733.6 1739.0 1726.2 1587.4 1630.6 1685.5 1701.3 1718.0 1726.2 1716.6 1725.8 1737.4 1736.6 1732.4 1731.2 1726.9 1727.8 1720.2 1715.4 1708.7 1713.0 1713.5 1718.0 1701.7 1701.7 1684.9 1687.2 1690.6 1684.3 1679.9 1672.9 1663.1 1669.3 1664.7 1672.3 1687.7 1686.8 1686.6 1675.8 1677.4 1673.2 1665.0 1671.3 1672.4 1676.2 1692.6 1696.5 1716.1 1713.3 1705.1 1711.3 1709.8 1688.6 1698.9 1700.0 1693.0 1683.9 1679.2 1673.9 1683.9 1688.4 1693.9 1720.9 1717.9 1733.6 1729.7 1735.6 1734.1 1699.3 1678.6 1675.5 1670.1 1652.2 1635.0 1654.9 1642.0 1638.7 1622.6 1596.1 1612.4 1625.0 1610.5 1606.6 1610.7 1603.1 1591.5 1605.2 1621.4 1622.5 1626.6 1627.4 1614.9 1602.3 1598.3 1627.0 1627.0 1627.0 1655.7 1670.1 1670.1 1670.1 1670.1 1704.0 1711.8 1700.5 1690.3 1715.4 1723.5 1719.4 1734.4 1772.8 1760.3 1747.2 1750.2 1755.3 1754.6 1751.2 1752.5 1769.4 1767.6 1750.0 1747.1 1753.5 1752.8 1752.9 1764.7 1776.8 1779.3 1785.1 1798.2 1794.1 1795.2 1780.4 1789.5 1794.2 1784.4 1800.1 1804.0 1816.2 1810.5 1821.9 1828.2 1840.6 1841.1 1846.3 1850.0 1839.0 1820.2 1815.2 1820.6 1807.1 1791.4 1806.2 1798.7 1818.2 1820.5 1833.3 1837.1 1818.2 1824.1 1830.1 1835.6 1828.7 1839.2 1837.2 1826.7 1838.0 1829.1 1843.1 1850.5 1827.1 1829.1 1848.0 1840.5 1853.8 1874.1 1871.3 1871.3 1871.3 1860.5 1874.7 1880.1 1874.7 1875.6 1859.5 1874.2 1880.1 1880.1 1907.7 1920.5 1937.3 1936.8 1949.1 1963.7 1950.8 1953.5 1945.0 1921.1 1939.1 1928.0 1933.4 1925.7 1931.7 1928.7 1924.5 1914.2 1914.2 1920.6 1923.3 1930.4 1915.2 1916.9 1913.8 1913.8 1899.7 1888.0 1868.8 1879.9 1865.7 1881.3 1873.1 1862.5 1869.3 1846.9 1847.1 1838.3 1845.8 1835.5 1846.6 1854.8 1845.3 1854.5 1870.5 1862.6 1856.6 1837.6 1846.7 1856.5 1841.8 1835.0 1844.4 1838.9 1805.6 1756.6 1786.1 1757.1 1762.8 1756.8 1761.9 1778.5 1812.7 1806.1 1798.1 1794.9 1805.4 1820.3 1819.6 1809.6 1799.9 1800.3 1793.3 1784.8 1791.7 1800.2 1788.6 1775.7 1753.5 1768.2 1727.9 1709.6 1704.6 1740.6 1745.7 1751.7 1747.3 1757.8 1774.2 1774.4 1788.3 1788.0 1779.1 1792.8 1812.0 1872.1 1851.4 1873.4 1889.6 1897.5 1888.8 1900.4 1913.4 1909.9 1910.8 1879.2 1880.2 1878.3 1885.2 1867.6 1788.0 1820.5 1858.2 1870.3 1878.4 1881.5 1893.2 1889.3 1877.3 1884.0 1904.7 1922.7 1908.5 1911.4 1921.1 1930.8 1927.8 1908.3 1905.9 1911.1 1921.6 1933.6 1942.0 1951.5 1955.7 1957.4 1962.3 1946.1 1950.2 1929.7 1913.4 1889.5 1882.8 1895.4 1897.9 1891.5 1880.1 1887.0 1891.4 1914.6 1931.2 1929.2 1924.3 1927.0 1935.0 1955.4 1962.2 1980.7 1987.7 1993.7 2015.7 2005.0 2023.9 2028.5 2044.9 2045.8 2057.3 2061.7 2061.7 2061.7 2092.3 2090.1 2105.4 2105.4 2105.4 2117.7 2128.2 2124.7 2079.9 2074.9 2046.4 2079.8 2076.7 2104.5 2101.3 2084.0 2063.9 2062.7 2089.9 2102.9 2086.0 2085.9 2064.1 2072.7 2091.0 2120.2 2120.4 2117.6 2123.7 2132.2 2137.0 2134.8 2121.4 2127.5 2135.5 2144.8 2131.0 2112.9 2131.3 2117.8 2096.1 2051.5 2065.7 2061.0 2100.6 2120.5 2130.9 2142.4 2139.5 2134.6 2132.2 2150.1 2157.0 2165.0 2127.2 2157.2 2150.8 2139.1 2154.7 2182.4 2161.3 2166.3 2146.8 2135.7 2160.9 2175.7 2188.9 2190.3 2188.4 2190.3 2184.0 2196.0 2184.9 2188.3 2188.3 2188.3 2181.7 2165.7 2160.5 2162.9 2166.3 2170.8 2178.1 2177.3 2162.3 2140.4 2124.7 2138.2 2123.2 2129.7 2152.8 2160.0 2165.8 2165.4 2162.6 2179.1 2191.1 2191.5 2183.3 2186.3 2205.2 2227.2 2227.0 2227.0 2232.9 2237.4 2243.5 2247.5 2267.1 2271.6 2271.6 2253.8 2259.8 2269.0 2284.2 2309.7 2294.5 2312.5 2309.2 2308.6 2293.8 2274.0 2294.6 2323.4 2318.0 2329.5 2335.0 2323.9 2335.8 2347.3 2369.8 2371.9 2376.2 2375.5 2368.8 2364.4 2390.9 2372.0 2397.1 2403.0 2408.7 2418.1 2410.5 2399.9 2396.4 2381.8 2324.5 2313.7 2340.2 2350.5 2388.7 2398.0 2408.7 2401.5 2400.9 2400.7 2429.8 2431.5 2424.9 2423.3 2420.5 2384.5 2411.1 2449.5 2461.0 2478.7 2464.9 2488.0 2480.6 2474.5 2467.3 2472.4 2499.7 2475.2 2478.6 2481.0 2488.5 2470.4 2463.8 2438.8 2392.5 2403.4 2397.9 2382.1 2363.6 2365.6 2388.3 2361.8 2374.4 2385.2 2418.4 2431.8 2414.6 2425.8 2445.9 2482.4 2478.3 2485.4 2473.1 2481.6 2490.3 2521.3 2534.8 2528.0 2533.6 2548.5 2552.5 2571.0 2586.4 2612.6 2663.6 2660.7 2668.4 2690.1 2701.4 2685.0 2700.2 2715.9 2720.9 2723.2 2727.1 2742.9 2749.9 2706.6 2654.2 2675.3 2727.1 2742.1 2720.1 2721.1 2717.4 2712.3 2711.1 2727.1 2733.9 2696.5 2702.1 2717.8 2726.8 2741.9 2738.2 2738.5 2774.3 2787.2 2819.0 2836.4 2834.8 2843.8 2858.4 2861.2 2876.9 2867.8 2850.6 2867.7 2889.5 2909.0 2918.0 2942.4 2967.4 2967.4 2972.6 2930.9 2934.2 2957.6 2957.6 2996.2 2999.2 3009.4 3021.9 3042.9 3015.9 3026.6 2999.0 2949.9 2990.6 3011.1 3037.5 3049.4 3045.9 3039.3 3041.3 3066.5 3091.3 3095.2 3140.7 3178.4 3148.7 3169.1 3151.9 3166.6 3089.8 3097.2 3054.2 3012.2 3012.2 2947.1 2947.1 2947.1 3033.4 3025.9 2997.6 2982.8 3027.5 2958.4 2929.3 2888.2 2847.4 2768.5 2803.4 2865.2 2918.7 2902.4 2858.5 2861.2 2831.6 2870.5 2906.9 2887.1 2887.7 2843.5 2804.3 2824.7 2857.0 2850.8 2831.8 2862.4 2827.4 2814.5 2794.8 2794.8 2794.8 2807.3 2830.2 2860.9 2873.4 2884.0 2887.8 2877.7 2843.7 2872.9 2869.1 2837.1 2818.7 2781.6 2787.9 2763.5 2776.2 2791.3 2768.9 2736.3 2765.2 2755.9 2705.6 2682.2 2641.4 2569.5 2629.0 2633.0 2633.0 2678.7 2709.0 2727.1 2740.6 2725.8 2732.7 2732.7 2692.0 2673.1 2689.2 2711.9 2742.9 2722.9 2731.5 2725.9 2728.4 2781.4 2778.1 2801.1 2777.2 2763.8 2744.2 2740.0 2701.8 2666.6 2627.6 2545.0 2544.2 2595.7 2626.7 2577.2 2561.4 2604.4 2631.2 2608.8 2588.9 2636.4 2609.7 2598.9 2590.5 2560.3 2562.5 2508.0 2474.5 2528.3 2514.9 2494.7 2521.6 2562.3 2579.4 2599.2 2601.2 2604.4 2544.9 2559.3 2579.5 2579.5 2629.7 2620.7 2615.3 2600.8 2617.5 2585.5 2591.4 2588.8 2580.5 2570.6 2588.7 2599.3 2582.7 2557.0 2532.4 2530.9 2541.1 2551.1 2581.3 2635.2 2635.7 2645.6 2628.4 2672.0 2674.5 2652.4 2662.5 2664.1 2642.1 2643.0 2658.8 2637.7 2628.6 2603.3 2614.7 2593.0 2594.0 2602.9 2609.1 2586.1 2581.0 2590.0 2558.0 2534.4 2499.5 2522.5 2480.4 2484.8 2496.9 2553.4 2570.2 2562.7 2593.3 2585.3 2575.6 2542.5 2529.8 2530.9 2508.6 2525.2 2494.7 2477.2 2458.6 2490.5 2506.5 2503.9 2500.0 2534.8 2541.0 2557.9 2543.9 2590.4 2602.7 2597.8 2590.9 2591.9 2609.4 2600.9 2606.5 2607.0 2571.8 2546.1 2568.6 2572.5 2579.4 2584.7 2594.9 2576.0 2577.5 2611.0 2589.1 2584.3 2589.3 2576.0 2562.8 2566.0 2581.4 2598.0 2604.7 2600.7 2614.7 2636.1 2649.8 2651.0 2651.0 2673.5 2656.2 2628.8 2628.8 2628.8 2612.3 2632.4 2613.1 2622.3 2617.3 2600.4 2597.0 2600.5 2600.2 2591.6 2582.8 2593.6 2595.5 2574.6 2525.3 2534.8 2525.5 2554.7 2552.4 2540.2 2536.9 2560.8 2563.7 2589.4 2621.4 2628.0 2617.4 2636.9 2635.4 2636.0 2642.4 2636.0 2621.0 2601.0 2597.2 2609.6 2610.0 2632.1 2623.3 2607.5 2600.4 2619.2 2617.2 2586.5 2553.2 2543.1 2515.8 2461.5 2467.7 2450.3 2463.3 2475.0 2491.7 2497.5 2516.0 2505.5 2501.9 2480.7 2526.6 2518.1 2495.6 2495.0 2533.1 2508.9 2511.1 2536.2 2536.3 2556.1 2559.3 2561.2 2563.1 2562.4 2560.3 2560.3 2560.3 2547.1 2516.8 2544.2 2560.8 2554.8 2561.1 2574.4 2567.0 2597.2 2597.2 2621.4 2618.6 2641.7 2650.5 2658.8 2655.6 2661.4 2701.0 2728.2 2739.0 2746.6 2739.8 2727.7 2701.0 2751.2 2750.4 2778.5 2778.5 2751.7 2762.5 2779.4 2786.4 2807.8 2805.3 2805.3 2809.7 2819.6 2806.8 2795.3 2802.3 2800.9 2798.1 2800.2 2775.4 2803.0 2805.6 2816.8 2843.3 2837.7 2830.3 2845.1 2836.8 2845.6 2825.3 2827.5 2847.5 2843.7 2833.7 2848.4 2842.9 2847.3 2867.9 2869.7 2863.4 2892.3 2875.1 2856.2 2822.2 2830.2 2836.6 2833.2 2827.1 2824.7 2833.1 2826.8 2826.8 2852.3 2814.3 2807.1 2800.6 2797.4 2781.3 2802.6 2820.1 2807.2 2838.4 2880.6 2880.0 2889.5 2886.7 2900.8 2896.8 2887.1 2893.8 2883.3 2887.3 2915.7 2908.8 2918.3 2952.7 2971.1 2999.6 2989.1 2989.2 2986.3 2990.5 3016.8 3011.6 3025.4 3008.1 3033.5 3029.0 3039.1 2994.7 3015.3 3045.5 3009.8 3000.6 3014.8 3039.0 3049.0 3036.5 3036.6 3054.0 3064.0 3050.0 3073.0 3083.7 3109.2 3116.3 3112.3 3135.4 3124.5 3120.4 3041.7 3062.3 3078.9 3080.0 3026.6 3057.9 3108.2 3128.5 3123.4 3137.6 3123.2 3133.8 3132.6 3150.1 3130.4 3121.4 3133.0 3124.8 3148.4 3152.5 3182.2 3183.9 3201.0 3209.2 3220.1 3251.0 3229.5 3245.9 3251.6 3261.2 3278.4 3278.5 3281.5 3245.7 3246.6 3261.6 3280.9 3273.4 3272.2 3289.8 3251.6 3217.7 3254.3 3255.0 3277.9 3277.9 3277.9 3317.1 3297.7 3297.7 3297.7 3297.7 3384.5 3395.8 3376.1 3368.1 3361.5 3314.8 3284.1 3292.9 3233.9 3254.8 3219.3 3257.8 3253.1 3249.9 3219.0 3221.3 3211.2 3191.4 3198.2 3236.8 3248.7 3279.9 3270.0 3221.7 3253.8 3262.7 3245.6 3242.7 3253.1 3260.1 3266.1 3272.2 3280.3 3270.6 3243.5 3253.0 3264.4 3296.2 3298.6 3309.9 3360.5 3354.8 3385.5 3400.6 3383.4 3408.3 3581.9 3549.7 3506.2 3540.5 3558.3 3578.4 3561.1 3583.6 3605.1 3611.1 3629.3 3647.7 3677.0 3658.1 3671.0 3640.3 3646.5 3654.3 3648.3 3606.7 3611.1 3611.1 3611.1 3594.4 3601.5 3566.9 3575.5 3589.6 3607.5 3603.9 3601.2 3628.1 3673.2 3647.3 3668.4 3668.1 3696.0 3658.2 3651.0 3651.0 3635.8 3611.2 3591.6 3583.6 3534.1 3521.6 3556.9 3557.8 3582.4 3572.5 3572.5 3593.0 3584.0 3587.3 3550.0 3547.4 3572.2 3572.2 3583.8 3558.2 3537.7 3556.1 3544.6 3568.1 3572.0 3585.5 3547.9 3575.8 3575.6 3600.9 3607.4 3593.9 3600.1 3628.3 3643.6 3655.4 3687.3 3701.5 3699.6 3723.5 3703.0 3732.9 3728.2 3750.8 3757.0 3771.3 3731.2 3732.8 3773.4 3788.0 3810.0 3785.8 3755.4 3678.1 3656.2 3649.2 3661.5 3566.1 3585.2 3482.6 3537.4 3537.8 3549.5 3518.2 3494.4 3494.4 3604.6 3623.3 3629.0 3662.6 3653.4 3643.2 3623.1 3630.2 3650.7 3646.3 3670.5 3664.4 3698.2 3683.8 3713.0 3725.4 3714.6 3722.4 3692.6 3682.6 3646.3 3654.7 3623.9 3634.2 3627.1 3637.8 3684.4 3694.2 3678.9 3687.2 3716.8 3722.7 3690.0 3690.1 3694.9 3694.9 3635.5 3661.2 3697.7 3699.5 3696.6 3736.4 3737.5 3760.2 3771.6 3771.6 3813.7 3818.1 3807.9 3797.3 3805.2 3813.7 3822.2 3790.2 3793.2 3803.3 3788.2 3775.3 3742.9 3753.6 3758.7 3763.5 3732.0 3723.1 3725.3 3736.1 3749.2 3784.8 3815.4 3811.5 3812.0 3827.6 3829.8 3825.7 3856.4 3897.9 3888.3 3892.6 3883.3 3861.5 3882.8 3891.7 3878.2 3845.3 3884.1 3902.5 3874.0 3924.5 3910.6 3913.8 3835.5 3874.0 3885.4 3857.6 3860.0 3839.3 3869.2 3850.4 3874.0 3880.8 3890.1 3875.5 3869.8 3869.8 3869.8 3922.2 3948.3 3942.2 3942.2 3942.2 3940.1 3923.8 3922.9 3944.9 3966.2 3947.4 3975.5 3983.6 3979.6 4007.1 4019.9 4009.5 4023.1 4115.4 4161.0 4125.5 4127.3 4182.3 4169.7 4209.1 4272.2 4282.8 4296.5 4305.5 4309.8 4357.9 4384.3 4408.4 4444.1 4436.3 4464.2 4514.6 4490.7 4525.5 4530.8 4522.5 4463.2 4503.9 4539.0 4519.7 4487.6 4460.1 4513.7 4547.1 4605.2 4638.9 4684.4 4677.1 4676.2 4609.9 4636.2 4556.5 4519.9 4535.1 4442.9 4491.3 4497.3 4558.6 4620.5 4659.2 4659.2 4659.2 4501.7 4488.7 4463.9 4471.5 4588.0 4582.6 4634.9 4626.6 4604.2 4586.3 4643.4 4625.6 4665.7 4699.1 4740.1 4752.3 4781.1 4836.1 4772.3 4793.3 4855.1 4897.6 4897.6 4953.5 5029.6 4988.4 5016.0 5016.0 5004.7 5042.5 5084.2 5134.3 5141.7 5157.5 5157.5 5081.0 5178.6 5176.4 5181.0 5196.7 5190.0 5133.1 5132.1 5041.6 5150.0 5207.2 5238.5 5251.2 5320.0 5368.8 5361.9 5308.6 5364.2 5384.6 5362.0 5345.9 5405.0 5510.3 5561.8 5587.8 5576.1 5662.4 5669.9 5700.3 5620.6 5654.8 5674.3 5804.9 5846.5 5947.0 6012.6 5977.1 5885.4 5801.5 5845.8 5844.7 5927.5 5868.3 5737.1 5620.5 5677.1 5869.9 5849.2 5847.0 5888.0 5842.1 5929.5 5898.2 5898.2 5771.0 5765.2 5812.1 5922.1 5864.8 5825.6 5808.4 5682.1 5579.5 5498.5 5405.6 5580.1 5690.1 5668.8 5475.8 5473.9 5363.3 5409.6 5217.3 5216.7 5271.5 5447.5 5478.6 5478.1 5532.9 5505.3 5445.1 5356.7 5280.8 5281.9 5321.7 5417.8 5550.4 5629.0 5611.0 5705.1 5730.4 5732.5 5667.1 5716.6 5691.8 5673.6 5754.7 5825.0 5929.0 5897.4 5846.9 5822.3 5732.2 5699.5 5792.8 5836.3 5815.9 5806.8 5751.6 5777.2 5862.9 5803.2 5651.8 5689.5 5533.5 5279.7 5479.0 5370.9 5467.2 5581.6 5538.2 5601.6 5557.4 5438.6 5459.7 5483.9 5434.0 5418.2 5437.0 5565.0 5574.2 5571.7 5650.4 5725.5 5645.7 5666.3 5738.3 5772.4 5775.9 5875.1 5919.9 5922.7 5969.5 6009.0 6095.3 6103.2 6056.6 6021.8 6018.7 5986.6 6092.7 6122.1 6115.1 5989.9 6049.3 6044.7 6046.7 6046.7 6046.7 6190.4 6267.6 6265.5 6265.5 6265.5 6397.0 6375.7 6390.0 6330.2 6251.8 6062.1 6169.3 6149.8 6148.5 6274.0 6340.4 6397.5 6391.4 6356.1 6391.0 6411.0 6424.0 6508.7 6530.4 6582.6 6688.0 6720.7 6708.9 6772.0 6857.1 6828.4 6860.8 6931.6 6856.0 6898.9 6905.3 6990.5 6966.2 6953.2 6986.7 6986.1 6945.0 7065.4 7118.6 7153.1 7273.0 7259.5 7130.5 7077.3 7197.2 7187.5 7246.5 7276.7 7267.9 7328.0 7261.2 7236.5 7132.4 7143.8 7300.5 7341.0 7407.4 7472.1 7415.9 7530.3 7536.3 7585.5 7615.5 7638.8 7725.9 7827.7 7744.3 7588.1 7624.1 7624.1 7624.1 7662.9 7616.3 7500.1 7453.7 7500.1 7369.1 7308.9 7265.5 7232.3 7053.5 7180.1 7241.8 7401.4 7401.4 7640.8 7596.2 7610.8 7536.0 7587.1 7677.5 7627.3 7582.8 7550.6 7519.4 7371.4 7483.2 7495.8 7495.8 7542.7 7657.1 7731.9 7633.5 7605.0 7656.1 7656.1 7657.5 7676.3 7592.9 7699.5 7743.4 7716.8 7652.6 7498.4 7417.4 7342.7 7388.7 7562.7 7488.0 7518.6 7511.8 7624.8 7667.9 7794.7 7816.9 7881.9 7882.0 8038.2 8047.3 8099.0 8166.0 8160.0 8227.2 8205.0 8192.4 8141.9 8180.5 8158.1 8126.5 8288.2 8400.8 8412.0 8340.7 8229.2 8205.7 7998.7 8093.0 8102.7 8205.5 8239.5 8139.2 8170.2 7943.2 7846.2 7952.9 7721.3 7447.9 7607.5 7552.6 7676.3}\
    ]
    array set EuStockMarkets.CAC [list\
        description {Daily Closing Prices of CAC Stock Index, 1991-1998}\
        times $EuStockMarkets(times)\
        values {1772.8 1750.5 1718.0 1708.1 1723.1 1714.3 1734.5 1757.4 1754.0 1754.3 1759.8 1755.5 1758.1 1757.5 1763.5 1762.8 1768.9 1778.1 1780.1 1767.7 1757.9 1756.6 1754.7 1766.8 1766.5 1762.2 1759.5 1782.4 1789.5 1783.5 1780.4 1808.8 1820.3 1820.3 1820.3 1687.5 1725.6 1792.9 1819.1 1833.5 1853.4 1849.7 1851.8 1857.7 1864.3 1863.5 1873.2 1860.8 1868.7 1860.4 1855.9 1840.5 1842.6 1861.2 1876.2 1878.3 1878.4 1869.4 1880.4 1885.5 1888.4 1885.2 1877.9 1876.5 1883.8 1880.6 1887.4 1878.3 1867.1 1851.9 1843.6 1848.1 1843.4 1843.6 1833.8 1833.4 1856.9 1863.4 1855.5 1864.2 1846.0 1836.8 1830.4 1831.6 1834.8 1852.1 1849.8 1861.8 1856.7 1856.7 1841.5 1846.9 1836.1 1838.6 1857.6 1857.6 1858.4 1846.8 1868.5 1863.2 1808.3 1765.1 1763.5 1766.0 1741.3 1743.3 1769.0 1757.9 1754.9 1739.7 1708.8 1722.2 1713.9 1703.2 1685.7 1663.4 1636.9 1645.6 1671.6 1688.3 1696.8 1711.7 1706.2 1684.2 1648.5 1633.6 1699.1 1699.1 1722.5 1720.7 1741.9 1765.7 1765.7 1749.9 1770.3 1787.6 1778.7 1785.6 1833.9 1837.4 1824.3 1843.8 1873.6 1860.2 1860.2 1865.9 1867.9 1841.3 1838.7 1849.9 1869.3 1890.6 1879.6 1873.9 1875.3 1857.0 1856.5 1865.8 1860.6 1861.6 1865.6 1864.1 1861.6 1876.5 1865.1 1882.1 1912.2 1915.4 1951.2 1962.4 1976.5 1953.5 1981.3 1985.1 1983.4 1979.7 1983.8 1988.1 1973.0 1966.9 1976.3 1993.9 1968.0 1941.8 1947.1 1929.2 1943.6 1928.2 1922.0 1919.1 1884.6 1896.3 1928.3 1934.8 1923.5 1943.8 1942.4 1928.1 1942.0 1942.7 1974.8 1975.4 1907.5 1943.6 1974.1 1963.3 1972.3 1990.7 1978.2 1978.2 1978.2 1980.4 1983.7 1978.1 1984.9 1995.7 2006.6 2036.7 2031.1 2031.1 2041.6 2046.9 2047.2 2063.4 2063.4 2077.5 2063.6 2053.2 2017.0 2024.0 2051.6 2023.1 2030.8 2016.8 2045.1 2046.3 2029.6 2014.1 2014.1 2033.3 2017.4 2024.9 1992.6 1994.9 1981.6 1981.6 1962.2 1953.7 1928.8 1928.3 1918.1 1931.4 1908.8 1891.8 1913.9 1885.8 1895.8 1899.6 1920.3 1915.3 1907.3 1900.6 1880.9 1873.5 1883.6 1868.5 1879.1 1847.8 1861.8 1859.4 1859.4 1859.4 1853.3 1851.2 1801.8 1767.9 1762.7 1727.5 1734.6 1734.6 1755.4 1769.0 1801.6 1782.6 1754.7 1784.4 1787.6 1798.0 1793.8 1777.3 1755.2 1737.8 1730.1 1722.4 1753.5 1757.3 1736.7 1734.2 1724.2 1744.2 1689.7 1667.7 1667.8 1687.6 1687.5 1684.9 1674.2 1711.4 1780.5 1779.0 1779.3 1763.7 1756.8 1774.2 1802.0 1873.6 1836.2 1859.8 1852.7 1882.9 1826.1 1832.8 1828.9 1829.5 1843.5 1770.3 1731.9 1736.7 1724.0 1683.3 1611.0 1612.5 1654.2 1673.9 1657.3 1655.1 1685.1 1667.9 1650.0 1664.2 1679.1 1731.3 1722.2 1730.7 1766.4 1770.7 1774.5 1749.9 1730.9 1742.4 1742.4 1786.9 1804.1 1804.7 1793.6 1786.7 1798.5 1798.5 1821.5 1796.8 1772.7 1764.4 1759.2 1722.3 1724.2 1674.8 1720.6 1721.0 1739.7 1749.7 1771.4 1792.3 1783.3 1799.4 1781.7 1788.6 1765.9 1791.2 1769.5 1758.7 1738.3 1744.8 1736.7 1735.2 1760.1 1786.3 1824.4 1821.1 1854.6 1854.6 1857.5 1870.3 1858.8 1857.8 1857.8 1843.1 1850.8 1859.6 1844.5 1852.6 1814.6 1796.8 1782.5 1803.5 1827.1 1837.5 1837.7 1818.8 1812.2 1820.4 1779.9 1792.6 1777.4 1780.6 1772.2 1785.9 1787.3 1824.1 1854.4 1908.2 1904.7 1894.1 1893.3 1905.6 1912.0 1899.5 1878.2 1905.0 1926.5 1937.2 1959.2 1944.1 1953.4 1944.6 1983.7 1998.8 2001.5 1995.2 1986.8 1995.1 2004.3 2009.7 1992.4 1988.9 1965.2 1986.0 1975.3 1967.3 1963.5 1962.7 1939.3 1952.2 1954.6 2001.4 2025.8 2033.9 2035.9 2031.4 2005.9 1990.8 1974.7 1995.3 1984.0 1986.9 1986.9 1986.9 2018.1 2015.4 1988.6 1986.7 1968.9 1949.3 1931.9 1944.5 1916.6 1911.6 1927.4 1942.5 1920.6 1939.0 1937.0 1923.6 1926.3 1920.5 1878.6 1877.2 1854.5 1872.7 1879.9 1851.7 1835.7 1846.4 1836.8 1836.8 1836.8 1861.4 1891.1 1890.4 1904.6 1888.7 1888.7 1872.8 1875.8 1867.9 1859.7 1887.9 1893.7 1915.2 1911.2 1920.4 1916.8 1897.9 1918.8 1900.3 1910.3 1929.2 1935.3 1942.4 1963.3 1960.8 1991.0 1977.5 1971.9 1960.2 1941.2 1925.4 1935.1 1943.7 1980.4 1985.7 1992.2 1991.2 1991.2 1963.1 1974.9 1981.7 1968.4 1947.5 1965.7 1995.0 2006.2 1998.1 1989.5 2036.0 2085.9 2129.0 2110.6 2101.4 2115.3 2149.8 2138.5 2139.8 2167.4 2161.9 2148.0 2148.0 2136.3 2160.8 2139.2 2128.2 2111.4 2123.4 2159.3 2173.6 2183.9 2205.7 2216.5 2191.9 2185.1 2156.1 2158.0 2137.3 2129.4 2108.8 2108.4 2119.1 2134.1 2078.5 2075.6 2099.5 2107.4 2094.4 2080.0 2057.5 2092.6 2108.6 2120.0 2126.8 2114.6 2116.7 2128.7 2158.8 2164.5 2147.4 2156.4 2138.7 2126.9 2127.3 2113.9 2139.3 2145.0 2147.3 2149.7 2199.7 2231.9 2227.7 2210.4 2192.0 2196.0 2182.0 2182.0 2169.7 2171.2 2136.0 2081.0 2084.8 2112.9 2087.3 2087.3 2096.9 2117.9 2115.9 2148.0 2149.7 2145.2 2082.6 2071.5 2070.6 2118.4 2120.6 2119.3 2110.1 2154.0 2160.5 2188.4 2186.7 2176.1 2205.3 2211.4 2198.1 2196.3 2156.5 2162.6 2160.3 2196.4 2223.5 2215.9 2225.8 2243.0 2251.5 2276.6 2264.6 2281.9 2281.2 2268.2 2290.6 2274.3 2249.6 2275.1 2307.6 2317.3 2331.3 2281.9 2252.2 2262.3 2234.8 2247.4 2274.7 2257.8 2244.0 2274.5 2278.3 2282.4 2281.0 2313.2 2334.4 2331.3 2355.9 2322.0 2329.2 2287.1 2299.9 2302.1 2296.8 2275.1 2281.6 2258.0 2258.0 2281.2 2251.8 2215.2 2226.7 2252.0 2208.3 2199.0 2238.1 2183.1 2144.7 2144.7 2178.7 2219.9 2216.4 2199.7 2184.6 2175.0 2215.0 2258.5 2242.7 2247.8 2221.3 2202.7 2200.7 2200.2 2152.6 2136.6 2144.5 2123.4 2083.9 2081.9 2081.9 2081.9 2100.3 2128.2 2119.6 2114.8 2145.3 2148.6 2152.4 2139.1 2159.6 2160.1 2136.0 2102.7 2092.0 2135.2 2116.3 2130.9 2147.3 2150.3 2166.0 2186.2 2179.0 2141.6 2162.6 2158.2 2139.4 2165.0 2176.7 2176.7 2187.0 2187.8 2195.2 2184.0 2165.4 2155.4 2155.4 2133.3 2084.4 2091.9 2050.7 2052.5 2029.9 1979.7 2007.4 2041.7 2037.2 2023.7 2046.8 2028.4 2020.7 1977.7 1992.0 1966.4 1942.8 1936.0 1903.0 1890.8 1917.0 1939.0 1907.0 1911.6 1925.8 1936.3 1892.0 1872.9 1866.2 1878.7 1889.0 1920.8 1920.8 1949.8 1942.1 1974.6 1974.6 1974.6 2025.1 2052.3 2043.7 2053.8 2041.4 2059.8 2076.8 2055.7 2053.4 2075.0 2069.6 2117.2 2115.0 2096.5 2107.1 2106.3 2074.5 2064.2 2038.9 2007.0 2007.0 2012.4 2035.0 2010.5 2001.3 1972.6 2000.6 2006.3 2026.5 2062.7 2075.3 2060.4 2069.1 2034.9 2020.4 1998.2 1961.5 1964.2 1983.4 1948.8 1966.8 1969.4 1952.9 1977.3 1924.6 1922.9 1919.3 1897.2 1899.4 1927.4 1902.7 1901.3 1905.0 1876.2 1879.3 1852.8 1876.1 1833.7 1843.4 1856.4 1898.3 1919.0 1918.1 1955.6 1933.0 1906.4 1898.6 1876.3 1867.4 1842.1 1841.6 1824.4 1831.5 1858.1 1905.7 1905.7 1905.7 1873.6 1911.1 1931.7 1906.1 1921.5 1943.9 1948.4 1948.4 1941.1 1954.5 1950.2 1927.5 1926.5 1927.8 1911.4 1893.1 1934.7 1945.9 1952.4 1940.0 1975.9 1964.0 1982.7 1973.7 1969.0 1969.8 1954.1 1937.0 1919.3 1917.1 1930.2 1931.1 1924.2 1928.1 1924.7 1940.9 1952.1 1949.9 1966.6 1956.0 1927.8 1894.2 1881.2 1881.2 1885.9 1901.8 1871.5 1886.4 1864.2 1859.2 1849.1 1844.1 1854.0 1872.8 1856.9 1860.3 1837.1 1813.3 1772.8 1780.2 1802.5 1826.3 1814.1 1813.4 1797.9 1827.8 1816.1 1842.4 1872.1 1870.4 1850.9 1874.4 1869.4 1850.1 1856.1 1861.9 1835.0 1822.5 1802.2 1805.6 1804.4 1827.0 1805.7 1802.2 1776.9 1808.4 1807.4 1795.2 1773.3 1748.6 1756.8 1727.1 1743.1 1721.8 1769.0 1738.6 1785.8 1788.8 1811.6 1813.8 1818.0 1795.7 1817.3 1836.1 1837.2 1852.4 1893.0 1859.5 1864.0 1882.8 1872.9 1890.2 1900.4 1880.9 1869.3 1871.9 1881.1 1881.1 1881.1 1855.7 1874.9 1882.0 1928.4 1918.5 1945.9 1942.3 1931.0 1918.5 1918.5 1936.9 1971.1 1946.2 1927.4 1927.4 1988.7 1996.7 2003.6 2017.3 1991.5 2001.3 2004.7 1989.8 1965.5 1979.5 1965.3 1960.7 1960.7 1919.1 1927.8 1927.3 1948.0 1960.8 1971.3 1971.3 1965.9 1974.7 1951.2 1897.0 1907.8 1922.8 1893.7 1920.7 1905.0 1918.4 1896.1 1902.8 1925.2 1895.1 1902.4 1877.7 1865.5 1861.9 1858.8 1879.7 1889.4 1907.8 1889.5 1950.8 1949.4 1961.3 1961.7 1948.3 1948.3 1951.5 1929.0 1931.2 1905.9 1908.9 1926.6 1941.3 1942.5 1947.9 1932.8 1920.0 1917.8 1960.0 1950.8 1957.1 1950.4 1954.7 1943.7 1949.2 1946.2 1946.2 1946.2 1970.4 1967.3 1971.6 1984.4 1963.3 1957.4 1939.2 1937.8 1890.1 1899.5 1921.4 1883.4 1881.7 1900.5 1902.5 1884.8 1861.7 1854.9 1869.7 1873.6 1898.2 1892.4 1880.0 1871.7 1882.5 1863.8 1853.9 1790.7 1800.6 1816.3 1791.8 1767.6 1788.4 1780.7 1800.7 1803.9 1800.3 1809.6 1785.7 1778.0 1794.4 1803.8 1817.0 1790.5 1779.7 1770.7 1757.3 1740.7 1721.1 1724.2 1764.1 1754.1 1742.4 1795.3 1814.0 1814.0 1828.7 1832.1 1822.6 1857.4 1866.3 1852.6 1839.8 1838.2 1838.2 1875.2 1905.1 1890.5 1881.4 1872.9 1875.5 1867.1 1891.0 1889.8 1870.3 1857.3 1828.3 1820.9 1774.9 1814.9 1834.8 1846.8 1856.3 1849.6 1848.9 1833.8 1875.0 1859.3 1820.5 1810.2 1831.2 1834.4 1873.4 1873.4 1866.7 1877.0 1879.1 1872.0 1872.0 1908.4 1943.0 1931.2 1917.7 1916.6 1916.3 1910.1 1897.9 1907.6 1924.4 1952.1 1966.0 1960.1 1964.3 1954.4 1934.1 1946.0 1950.2 1966.7 1980.7 2003.1 2021.0 2024.1 2022.2 1985.6 1988.5 1983.3 1956.2 1960.7 1967.7 1983.3 1956.4 1964.2 1952.5 1939.7 1932.4 1936.9 1953.3 1976.9 1960.9 1974.5 1996.9 1990.8 2017.2 2015.3 2001.9 2005.9 2008.0 1975.5 1948.9 1932.1 1944.3 1962.4 1950.1 1965.0 1967.2 1969.8 1976.9 1974.4 2003.8 2008.0 2030.5 2020.0 2044.8 2055.6 2070.4 2064.0 2075.0 2075.0 2075.0 2081.7 2093.9 2072.5 2074.7 2080.6 2097.3 2075.1 2086.0 2092.5 2116.5 2112.0 2122.1 2116.4 2138.4 2130.8 2146.8 2146.8 2136.8 2115.4 2090.4 2083.7 2083.7 2085.4 2114.8 2100.9 2122.1 2124.6 2124.6 2136.9 2120.6 2129.7 2103.5 2114.5 2117.7 2117.7 2132.9 2117.1 2108.4 2110.1 2121.1 2111.0 2115.4 2133.2 2102.0 2120.8 2137.5 2137.3 2126.2 2111.8 2113.0 2107.9 2100.7 2077.1 2084.2 2097.6 2118.4 2113.3 2112.9 2123.7 2118.8 2111.8 2114.0 2126.8 2098.8 2079.1 2076.6 2081.9 2073.4 2050.6 2029.5 1989.5 1995.1 2007.3 1992.8 1960.3 1982.2 1954.1 1974.7 1962.9 1961.1 1968.5 1995.9 2009.9 2023.4 2013.2 1999.3 1996.7 1997.4 1985.4 1978.2 1980.6 1979.5 1979.5 1979.5 1986.2 2019.3 2000.6 2017.8 2020.8 2020.4 2018.0 2002.9 1977.6 1970.6 1977.0 1971.4 1984.8 1996.2 2004.8 2020.3 2042.1 2038.1 2065.4 2080.4 2086.2 2080.9 2072.7 2082.3 2079.5 2067.1 2081.5 2103.4 2104.1 2107.1 2132.8 2123.2 2141.8 2136.1 2152.7 2151.6 2161.5 2146.7 2135.6 2147.1 2143.6 2168.3 2158.5 2165.3 2185.2 2180.2 2175.6 2148.9 2151.6 2162.4 2150.4 2125.7 2124.8 2140.5 2140.5 2142.2 2187.3 2213.4 2211.8 2205.2 2205.2 2229.1 2217.2 2218.2 2240.3 2228.8 2240.1 2233.4 2233.6 2255.5 2277.1 2275.1 2270.8 2290.3 2315.7 2318.6 2349.1 2308.7 2292.5 2240.7 2255.8 2251.4 2213.3 2212.1 2203.4 2222.6 2193.7 2218.9 2248.7 2278.5 2287.4 2289.6 2289.6 2303.8 2307.0 2318.6 2315.7 2315.7 2257.0 2282.8 2306.7 2301.7 2331.6 2349.1 2327.5 2361.3 2402.1 2388.0 2407.8 2425.1 2406.1 2409.9 2442.5 2461.3 2430.3 2435.2 2482.8 2465.0 2503.1 2516.6 2508.6 2503.1 2541.3 2558.4 2597.5 2595.4 2582.1 2599.3 2628.4 2627.4 2634.5 2617.5 2594.8 2575.2 2562.8 2567.9 2607.7 2602.2 2629.4 2607.8 2600.3 2651.7 2666.2 2698.9 2708.3 2709.2 2686.2 2641.7 2632.1 2645.6 2588.4 2574.0 2596.8 2553.7 2587.1 2579.3 2624.3 2648.7 2656.7 2656.7 2656.7 2581.8 2530.3 2514.5 2518.0 2572.3 2579.0 2617.6 2608.0 2574.6 2566.1 2620.6 2621.0 2615.2 2547.6 2522.7 2514.7 2533.6 2539.8 2536.3 2550.3 2602.9 2639.5 2639.5 2655.3 2672.8 2651.9 2643.3 2643.3 2633.9 2693.1 2719.6 2774.6 2776.0 2784.3 2784.3 2751.1 2786.4 2741.7 2762.9 2654.7 2680.3 2583.2 2579.2 2583.9 2601.5 2624.5 2635.4 2690.9 2719.3 2686.2 2664.2 2696.2 2760.3 2808.5 2795.9 2762.6 2751.7 2739.7 2757.1 2762.2 2784.8 2867.4 2893.6 2891.0 2858.3 2944.0 2909.5 2937.0 2934.5 2947.7 2929.8 2950.6 2929.1 2941.6 2941.6 2950.7 2988.0 2958.6 2876.7 2874.1 2921.1 3003.5 2973.5 3025.9 3022.2 3023.6 3069.3 3075.7 3049.5 2992.4 2984.1 3037.1 3056.3 2996.3 2983.4 2998.6 2924.0 2921.8 2921.8 2870.1 2936.2 2979.3 2957.2 2904.2 2898.6 2869.3 2871.7 2828.4 2770.5 2805.8 2921.2 2918.0 2927.0 2924.5 2940.9 2919.7 2874.6 2843.6 2834.1 2898.6 2940.6 2944.0 2978.4 2977.2 3017.5 2997.2 3023.7 3005.4 2985.6 2989.0 3008.3 3054.9 3052.1 3094.0 3078.0 3064.4 3024.1 2960.7 2955.1 3002.9 3002.5 2992.2 2992.9 2958.0 2946.7 2989.9 2958.1 2856.9 2849.0 2769.6 2651.3 2818.0 2739.5 2739.3 2788.0 2774.9 2822.4 2781.8 2707.1 2707.1 2707.1 2694.5 2700.7 2698.9 2773.0 2782.6 2790.6 2821.2 2861.7 2802.5 2786.3 2811.7 2829.0 2854.4 2918.5 2913.1 2902.4 2914.5 2910.1 2932.5 2959.4 2932.2 2828.5 2830.3 2838.3 2912.2 2893.3 2894.5 2822.9 2869.7 2858.1 2874.1 2874.1 2875.1 2939.5 2975.5 2998.9 2998.9 3038.7 3072.8 3037.7 3006.7 2954.9 2919.8 2862.5 2902.9 2919.8 2932.8 2976.1 2987.0 3008.3 2998.1 2988.6 2966.2 3000.5 3052.0 3088.3 3133.8 3172.1 3187.5 3188.4 3166.3 3189.6 3216.7 3220.9 3235.8 3240.0 3178.7 3187.7 3225.1 3280.5 3281.7 3250.6 3262.5 3273.5 3262.3 3348.2 3397.0 3421.9 3446.7 3414.9 3381.3 3395.8 3483.2 3525.9 3521.5 3539.4 3526.6 3540.2 3598.3 3661.3 3652.5 3688.7 3688.9 3680.1 3738.5 3818.7 3783.8 3810.2 3800.2 3875.3 3883.3 3935.9 3932.0 3986.8 3903.3 3873.9 3894.5 3894.5 3894.5 3867.7 3884.6 3845.9 3861.6 3885.7 3860.4 3835.1 3822.1 3788.7 3689.4 3777.2 3726.2 3867.9 3867.9 3979.3 3945.5 3947.5 3912.8 3912.8 4007.3 3986.1 4018.5 4012.0 3990.2 3945.3 3980.8 4047.9 4047.9 4049.8 4108.7 4115.9 4017.4 4014.9 4041.2 4041.2 4087.0 4149.4 4119.0 4185.1 4204.6 4201.9 4208.6 4141.6 4050.8 4005.3 4013.3 4092.9 4052.3 4027.3 4018.6 4065.0 4126.3 4203.8 4215.7 4248.2 4203.5 4260.7 4252.1 4304.4 4311.1 4333.1 4339.9 4319.2 4256.4 4256.4 4256.4 4344.3 4358.1 4388.5 4368.9 4322.1 4220.1 4235.9 4205.4 4139.5 4122.4 4139.2 4197.6 4177.3 4095.0 4047.9 3976.4 3968.6 4041.9 3939.5 3846.0 3945.7 3951.7 3995.0}\
    ]
    array set EuStockMarkets.FTSE [list\
        description {Daily Closing Prices of FTSE Stock Index, 1991-1998}\
        times $EuStockMarkets(times)\
        values {2443.6 2460.2 2448.2 2470.4 2484.7 2466.8 2487.9 2508.4 2510.5 2497.4 2532.5 2556.8 2561.0 2547.3 2541.5 2558.5 2587.9 2580.5 2579.6 2589.3 2595.0 2595.6 2588.8 2591.7 2601.7 2585.4 2573.3 2597.4 2600.6 2570.6 2569.4 2584.9 2608.8 2617.2 2621.0 2540.5 2554.5 2601.9 2623.0 2640.7 2640.7 2619.8 2624.2 2638.2 2645.7 2679.6 2669.0 2664.6 2663.3 2667.4 2653.2 2630.8 2626.6 2641.9 2625.8 2606.0 2594.4 2583.6 2588.7 2600.3 2579.5 2576.6 2597.8 2595.6 2599.0 2621.7 2645.6 2644.2 2625.6 2624.6 2596.2 2599.5 2584.1 2570.8 2555.0 2574.5 2576.7 2579.0 2588.7 2601.1 2575.7 2559.5 2561.1 2528.3 2514.7 2558.5 2553.3 2577.1 2566.0 2549.5 2527.8 2540.9 2534.2 2538.0 2559.0 2554.9 2575.5 2546.5 2561.6 2546.6 2502.9 2463.1 2472.6 2463.5 2446.3 2456.2 2471.5 2447.5 2428.6 2420.2 2414.9 2420.2 2423.8 2407.0 2388.7 2409.6 2392.0 2380.2 2423.3 2451.6 2440.8 2432.9 2413.6 2391.6 2358.1 2345.4 2384.4 2384.4 2384.4 2418.7 2420.0 2493.1 2493.1 2492.8 2504.1 2493.2 2482.9 2467.1 2497.9 2477.9 2490.1 2516.3 2537.1 2541.6 2536.7 2544.9 2543.4 2522.0 2525.3 2510.4 2539.9 2552.0 2546.5 2550.8 2571.2 2560.2 2556.8 2547.1 2534.3 2517.2 2538.4 2537.1 2523.7 2522.6 2513.9 2541.0 2555.9 2536.7 2543.4 2542.3 2559.7 2546.8 2565.0 2562.0 2562.1 2554.3 2565.4 2558.4 2538.3 2533.1 2550.7 2574.8 2522.4 2493.3 2476.0 2470.7 2491.2 2464.7 2467.6 2456.6 2441.0 2458.7 2464.9 2472.2 2447.9 2452.9 2440.1 2408.6 2405.4 2382.7 2400.9 2404.2 2393.2 2436.4 2572.6 2591.0 2600.5 2640.2 2638.6 2638.6 2638.6 2625.8 2607.8 2609.8 2643.0 2658.2 2651.0 2664.9 2654.1 2659.8 2659.8 2662.2 2698.7 2701.9 2725.7 2737.8 2722.4 2720.5 2694.7 2682.6 2703.6 2700.6 2711.9 2702.0 2715.0 2715.0 2704.6 2698.6 2694.2 2707.6 2697.6 2705.9 2680.9 2681.9 2668.5 2645.8 2635.4 2636.1 2614.1 2603.7 2593.6 2616.3 2598.4 2562.7 2584.8 2550.3 2560.6 2532.6 2557.3 2534.1 2515.8 2521.2 2493.9 2476.1 2497.1 2469.0 2493.7 2472.6 2497.9 2490.8 2478.3 2484.0 2486.4 2483.4 2431.9 2403.7 2415.6 2387.9 2399.5 2377.2 2348.0 2373.4 2423.2 2411.6 2399.6 2420.2 2407.5 2392.8 2377.6 2350.1 2325.7 2309.6 2303.1 2318.0 2356.8 2376.1 2354.7 2363.5 2359.4 2365.7 2311.1 2281.0 2285.0 2311.6 2312.6 2312.6 2298.4 2313.0 2381.9 2362.2 2372.2 2337.7 2327.5 2340.6 2370.9 2422.1 2370.0 2378.3 2483.9 2567.0 2560.1 2586.0 2580.5 2621.2 2601.0 2560.0 2565.5 2553.0 2572.3 2549.7 2446.3 2488.4 2517.1 2538.8 2541.2 2557.2 2584.7 2574.7 2546.6 2563.9 2562.2 2617.0 2645.7 2658.1 2669.7 2661.6 2669.8 2650.4 2642.3 2658.3 2687.8 2705.6 2691.7 2711.1 2702.7 2695.4 2714.6 2696.8 2726.4 2697.5 2679.6 2679.2 2704.0 2706.2 2732.4 2722.9 2727.1 2709.6 2741.8 2760.1 2778.8 2792.0 2764.1 2771.0 2759.4 2754.5 2769.8 2750.7 2726.5 2716.2 2721.8 2717.9 2732.8 2740.3 2789.7 2807.7 2842.0 2827.4 2827.5 2827.5 2827.5 2847.8 2832.5 2846.5 2846.5 2861.5 2833.6 2826.0 2816.5 2799.2 2773.4 2757.9 2745.3 2759.2 2765.1 2763.1 2737.6 2748.7 2773.3 2781.2 2771.9 2835.7 2832.5 2816.9 2807.2 2851.6 2834.4 2873.8 2865.9 2862.9 2870.0 2831.3 2816.4 2834.3 2843.0 2845.9 2812.2 2814.0 2837.7 2840.0 2838.3 2818.0 2817.0 2828.7 2868.0 2882.6 2882.3 2918.6 2904.8 2916.6 2957.3 2949.9 2956.7 2953.4 2915.9 2922.4 2919.3 2889.9 2883.3 2900.1 2863.9 2861.1 2860.6 2852.8 2852.9 2846.5 2861.0 2878.7 2878.4 2869.9 2838.8 2832.2 2822.1 2821.8 2821.8 2821.8 2846.8 2842.1 2839.7 2824.4 2830.0 2856.1 2869.6 2881.1 2843.8 2822.3 2832.7 2797.3 2786.8 2813.1 2813.1 2812.6 2796.5 2786.3 2793.7 2829.7 2836.1 2860.8 2849.3 2847.0 2858.1 2847.3 2819.7 2816.8 2812.2 2825.6 2837.7 2846.9 2855.3 2840.7 2840.7 2849.2 2863.0 2852.8 2829.9 2844.8 2844.4 2866.9 2860.0 2861.8 2885.5 2870.0 2883.0 2875.7 2879.4 2903.4 2907.6 2900.7 2894.7 2887.5 2897.0 2886.0 2900.0 2888.8 2857.7 2838.5 2848.1 2848.3 2845.9 2843.2 2830.9 2837.1 2832.3 2831.7 2833.0 2842.9 2823.9 2814.1 2820.1 2827.7 2844.2 2879.4 2884.2 2917.6 2926.5 2941.7 2945.0 2941.3 2943.4 2969.8 2986.4 2971.6 3006.1 3009.1 3010.1 3008.3 3025.0 3073.6 3065.5 3057.6 3042.0 3049.3 3079.2 3079.2 3100.6 3100.6 3100.0 3085.1 3072.6 3057.3 3055.4 3038.6 3035.4 3031.2 3037.0 3024.8 3028.0 2989.4 3003.9 3005.5 3004.5 3001.6 3007.5 3001.3 3005.2 3026.3 3036.9 3030.1 3037.5 3039.3 3067.7 3085.2 3100.8 3092.4 3108.6 3102.2 3094.7 3080.9 3086.3 3120.8 3137.6 3129.6 3156.3 3188.3 3199.0 3184.8 3165.3 3154.3 3163.0 3171.0 3164.4 3164.1 3162.3 3149.0 3085.6 3077.6 3096.0 3098.5 3099.7 3099.1 3093.3 3097.5 3120.0 3125.5 3108.0 3070.6 3069.3 3067.2 3093.1 3111.4 3135.8 3166.9 3233.2 3223.9 3234.2 3237.3 3237.3 3277.4 3271.6 3261.3 3254.6 3248.4 3278.8 3311.2 3337.1 3364.9 3342.4 3355.7 3396.5 3412.3 3412.3 3412.3 3462.0 3428.8 3418.4 3418.4 3408.5 3379.2 3403.0 3446.0 3440.6 3413.8 3372.0 3360.0 3400.6 3407.8 3437.0 3475.1 3470.0 3484.2 3481.4 3444.0 3436.1 3427.3 3447.4 3491.8 3481.5 3520.3 3491.5 3475.4 3419.1 3440.2 3429.1 3407.0 3378.9 3363.5 3393.2 3417.7 3425.3 3382.6 3350.3 3333.7 3341.9 3267.5 3281.2 3328.1 3270.6 3248.1 3246.5 3278.0 3305.9 3264.4 3246.7 3233.9 3191.9 3233.4 3267.4 3242.9 3255.7 3218.1 3198.0 3201.5 3155.3 3121.7 3129.0 3129.5 3123.4 3092.4 3086.4 3086.4 3086.4 3116.2 3131.5 3129.0 3120.8 3149.4 3159.1 3145.8 3131.7 3168.3 3138.2 3128.0 3098.3 3101.2 3133.7 3106.1 3125.3 3150.0 3129.9 3125.3 3125.3 3100.0 3070.5 3106.0 3106.0 3097.8 3136.3 3130.5 3137.8 3119.2 3115.6 3123.5 3116.5 3122.8 3127.3 3108.4 3089.1 3020.7 3019.7 2966.4 2966.4 2970.5 2931.9 2980.8 2997.8 3009.4 3004.8 3038.2 3028.9 3055.9 3016.3 3039.6 3045.8 3030.1 3022.9 2971.1 2940.2 2960.4 2942.4 2876.6 2899.9 2909.0 2946.3 2919.2 2936.4 2970.4 2965.0 2946.7 2964.4 2962.4 2983.8 2963.9 3005.3 3050.4 3074.8 3082.0 3091.3 3077.2 3095.1 3114.7 3106.1 3117.2 3082.3 3095.9 3082.6 3097.4 3157.5 3160.4 3150.5 3167.5 3171.9 3168.6 3167.0 3138.2 3142.3 3142.2 3147.3 3190.3 3182.6 3191.4 3171.3 3175.1 3205.2 3234.2 3265.1 3265.1 3249.6 3251.3 3216.5 3222.7 3241.5 3205.4 3203.9 3180.0 3139.3 3128.8 3121.4 3079.8 3112.7 3065.1 3079.1 3037.3 3014.8 3021.2 3028.2 2999.8 3008.5 3038.7 2992.5 3026.3 2983.5 3001.8 2956.3 2984.4 2998.7 3032.3 3073.0 3100.5 3141.9 3106.7 3120.2 3085.3 3060.8 3063.2 3032.8 3029.1 3000.9 2999.9 3029.6 3083.8 3097.4 3096.3 3081.3 3104.4 3097.6 3065.8 3063.8 3099.6 3103.5 3075.9 3095.3 3135.4 3146.5 3127.5 3131.0 3121.0 3078.7 3027.5 3036.6 3033.5 3047.1 3061.1 3081.4 3039.6 3017.3 3033.5 3016.1 3012.5 3013.8 2977.3 2943.4 2946.4 2980.6 2973.4 3013.6 3034.4 3058.1 3070.4 3091.7 3083.4 3083.4 3083.4 3095.8 3065.6 3065.5 3065.5 3065.7 3051.6 3032.3 3065.0 3055.8 3060.4 3049.4 3033.2 3048.3 3076.7 3054.0 3054.9 3028.6 2995.9 2954.2 2969.0 2982.2 3007.3 3022.2 2995.9 2991.6 3017.3 3034.7 3059.7 3062.0 3072.7 3072.5 3099.0 3109.9 3081.1 3071.3 3074.9 3051.1 3044.2 3018.6 3023.4 3019.5 3049.3 3037.7 3025.3 3009.3 3041.2 3038.2 3025.1 3001.9 2977.0 2992.1 2986.9 3021.1 3011.8 3050.6 3047.0 3094.1 3089.3 3124.2 3135.0 3139.7 3136.4 3153.4 3149.8 3128.3 3142.3 3176.2 3137.9 3143.1 3188.1 3190.2 3200.9 3210.9 3204.2 3190.9 3209.8 3208.8 3208.8 3208.8 3194.5 3170.1 3174.7 3199.9 3209.3 3214.9 3226.2 3217.6 3216.7 3220.4 3248.2 3262.6 3264.3 3251.7 3251.7 3261.2 3290.1 3317.9 3310.3 3310.7 3300.8 3297.4 3285.8 3261.0 3284.5 3291.8 3327.3 3328.2 3311.1 3311.1 3309.9 3319.4 3340.6 3345.0 3376.6 3380.0 3370.8 3380.8 3337.7 3344.6 3348.0 3339.8 3370.4 3366.1 3381.3 3377.2 3378.3 3403.8 3379.4 3309.2 3313.2 3282.7 3294.0 3314.6 3323.7 3349.2 3394.9 3388.3 3462.9 3455.0 3464.0 3450.6 3447.2 3429.2 3442.6 3420.7 3405.3 3400.4 3413.1 3431.6 3432.9 3454.3 3458.3 3468.9 3463.3 3449.9 3499.9 3475.6 3482.4 3483.5 3468.8 3468.3 3474.7 3467.5 3441.4 3444.4 3465.1 3470.6 3509.8 3535.7 3530.2 3515.9 3520.0 3524.9 3524.9 3502.6 3504.0 3477.8 3509.4 3522.7 3532.4 3557.7 3545.6 3554.5 3549.3 3535.9 3570.8 3565.4 3564.6 3533.3 3541.4 3561.5 3557.9 3514.8 3507.0 3523.3 3485.0 3479.0 3508.2 3520.2 3524.2 3544.1 3544.4 3526.5 3510.3 3460.1 3474.3 3523.8 3568.0 3557.3 3562.2 3593.0 3578.6 3551.4 3531.5 3535.3 3537.8 3519.6 3497.9 3510.0 3529.1 3518.7 3523.0 3500.4 3514.8 3522.4 3537.1 3541.6 3523.4 3536.8 3547.9 3571.4 3610.8 3609.2 3628.8 3604.1 3632.4 3602.5 3624.0 3649.0 3648.8 3655.5 3664.3 3664.3 3669.7 3664.2 3662.8 3639.5 3630.0 3652.1 3654.9 3662.4 3671.6 3642.6 3596.1 3576.9 3613.7 3633.3 3658.3 3658.3 3658.3 3676.4 3676.7 3689.3 3689.3 3687.9 3715.6 3714.1 3704.5 3720.6 3720.6 3671.5 3654.9 3657.3 3662.7 3710.6 3704.2 3748.7 3748.4 3754.2 3735.0 3758.2 3734.2 3734.7 3734.6 3735.3 3759.3 3752.8 3781.3 3746.6 3747.5 3726.1 3708.4 3716.3 3726.6 3747.6 3745.0 3779.8 3770.9 3744.3 3714.6 3725.6 3740.0 3740.3 3704.2 3715.9 3738.2 3727.6 3752.7 3768.6 3777.1 3758.9 3758.2 3710.3 3674.5 3639.5 3640.3 3681.8 3644.8 3669.6 3693.0 3685.4 3698.3 3707.0 3681.9 3660.9 3672.4 3672.6 3699.7 3718.4 3728.5 3725.1 3755.6 3755.6 3755.6 3758.6 3767.4 3744.2 3766.8 3790.5 3825.3 3805.6 3820.7 3857.1 3852.7 3833.0 3817.6 3819.3 3832.8 3809.2 3817.9 3806.0 3776.4 3751.6 3751.6 3723.0 3707.3 3728.3 3754.4 3739.2 3759.7 3776.2 3753.6 3789.6 3778.2 3789.4 3764.2 3747.0 3752.1 3752.1 3760.2 3775.7 3746.7 3747.8 3739.2 3755.2 3753.4 3760.3 3706.8 3728.8 3755.7 3769.2 3761.7 3753.6 3761.5 3756.4 3753.2 3727.5 3722.3 3710.8 3679.5 3695.5 3678.8 3711.0 3725.6 3725.7 3714.1 3760.6 3743.2 3741.5 3752.3 3765.8 3749.0 3728.3 3698.3 3632.3 3658.2 3693.4 3710.5 3681.3 3708.4 3668.8 3684.7 3673.3 3678.8 3668.5 3703.2 3734.4 3770.6 3788.3 3788.4 3811.1 3811.4 3810.7 3803.3 3823.4 3830.3 3837.4 3872.9 3863.7 3883.2 3872.1 3891.1 3907.5 3907.5 3905.7 3918.7 3885.0 3867.6 3884.4 3855.9 3872.7 3887.2 3893.0 3910.8 3916.1 3905.6 3932.6 3967.9 3977.2 3972.3 3955.7 3974.3 3964.1 3919.7 3910.5 3935.7 3933.2 3946.4 3953.7 3992.2 4015.1 4000.0 4024.8 4031.5 4035.6 4009.3 3994.7 4028.1 4038.7 4050.8 4024.4 4042.1 4053.1 4073.1 4057.2 4028.4 3999.4 4022.4 4025.3 3993.5 3963.9 3979.1 3948.5 3928.1 3921.1 3935.7 3900.4 3910.8 3914.4 3934.3 3926.9 3926.1 3958.2 3962.1 3978.1 3962.8 3953.8 4018.7 4054.6 4068.4 4049.2 4050.2 4058.0 4038.5 4061.5 4045.2 4051.2 3963.0 4011.6 4035.7 3982.5 3990.7 3972.4 3993.8 3979.6 4018.2 4051.3 4077.6 4087.2 4092.5 4092.5 4092.5 4091.0 4115.7 4118.5 4118.5 4057.4 4089.5 4106.5 4078.8 4087.5 4087.0 4056.6 4107.3 4168.2 4158.9 4197.5 4207.7 4194.0 4195.5 4219.1 4219.1 4218.8 4212.0 4237.4 4207.5 4228.4 4275.8 4257.8 4260.9 4281.5 4265.9 4307.8 4307.7 4304.3 4304.3 4327.1 4341.0 4337.8 4332.3 4357.4 4356.1 4336.8 4331.1 4344.7 4329.3 4339.2 4308.3 4307.1 4357.7 4360.1 4399.3 4420.3 4437.4 4444.3 4422.5 4397.7 4424.3 4373.3 4356.8 4332.2 4258.1 4254.8 4214.8 4270.7 4301.5 4312.9 4312.9 4312.9 4248.1 4236.6 4214.6 4236.6 4271.7 4269.3 4292.3 4313.2 4270.7 4251.7 4286.8 4294.6 4298.9 4310.5 4328.7 4346.1 4387.7 4388.5 4369.7 4389.7 4433.2 4436.0 4445.0 4455.6 4455.6 4519.3 4537.5 4580.4 4630.9 4669.6 4691.0 4686.9 4681.2 4693.9 4645.2 4607.5 4642.0 4651.8 4661.8 4661.8 4681.6 4677.5 4672.3 4621.3 4562.8 4557.8 4557.1 4576.2 4645.0 4686.7 4739.6 4724.8 4757.4 4783.1 4745.1 4682.2 4657.0 4653.7 4593.9 4575.8 4596.3 4640.0 4657.9 4640.3 4604.6 4728.3 4751.4 4831.7 4812.8 4810.7 4758.5 4762.4 4767.8 4799.5 4857.4 4899.3 4964.2 4949.0 4877.2 4805.7 4846.7 4874.5 4862.9 4851.5 4862.6 4876.6 4927.3 4907.5 4899.3 4895.7 4960.6 5026.2 5086.8 5031.3 5031.9 5075.8 5003.6 4991.3 4865.8 4835.0 4914.2 4958.4 4978.0 4901.1 4901.1 4886.3 4906.9 4845.4 4817.5 4870.2 4952.2 4976.9 4991.3 4994.2 4985.2 4950.5 4905.2 4854.8 4848.2 4902.9 4976.4 5013.1 5046.2 5023.8 5075.7 5027.5 5077.2 5065.5 5226.3 5220.3 5244.2 5317.1 5296.1 5330.8 5300.0 5305.6 5262.1 5217.8 5227.3 5300.1 5298.9 5263.7 5287.9 5271.1 5211.0 5225.9 5148.8 4991.5 4970.2 4840.7 4755.4 4871.8 4801.9 4842.3 4906.4 4897.4 4908.3 4863.8 4764.3 4806.8 4793.7 4720.4 4711.0 4741.8 4867.0 4845.4 4830.1 4908.4 4985.8 4898.6 4863.5 4891.2 4889.0 4831.8 4921.8 4977.6 4970.7 5082.3 5142.9 5187.4 5177.1 5130.7 5035.9 5045.2 5121.8 5203.4 5190.8 5168.3 5020.2 5018.2 5049.8 5013.9 5013.9 5013.9 5112.4 5132.3 5135.5 5135.5 5193.5 5262.5 5264.4 5224.1 5237.1 5138.3 5068.8 5083.9 5106.9 5165.8 5263.1 5273.6 5278.2 5272.3 5253.1 5181.4 5237.2 5326.3 5372.6 5422.4 5458.5 5599.0 5612.8 5595.8 5606.4 5629.7 5600.9 5613.3 5607.9 5552.5 5582.3 5619.9 5709.5 5723.4 5718.5 5718.5 5702.8 5651.0 5745.1 5764.8 5767.3 5820.6 5807.7 5733.1 5695.6 5782.9 5818.9 5828.5 5829.8 5794.8 5782.3 5785.1 5834.9 5903.6 5997.9 5956.3 5947.0 5983.7 5967.8 5905.6 5939.3 5911.9 5932.2 6017.6 6052.8 6064.2 6105.8 6094.0 6055.2 6105.5 6105.5 6105.5 6104.1 6074.1 6002.0 5922.2 5954.1 5955.0 5931.1 5898.1 5863.9 5722.4 5806.6 5833.1 5928.3 6011.3 6011.3 5986.5 5992.4 5938.0 5969.8 6028.3 5956.7 5972.9 5948.5 5917.8 5826.2 5877.8 5907.4 5935.6 5955.6 5955.6 5970.7 5870.2 5862.3 5870.7 5837.9 5842.3 5898.4 5860.8 5947.3 6037.8 6019.8 5987.4 5852.5 5769.8 5715.7 5729.7 5832.7 5812.1 5748.1 5712.4 5772.0 5804.9 5858.9 5877.4 5884.5 5832.5 5919.9 5960.2 5988.4 5990.3 6003.4 6009.6 5969.7 5927.9 5958.2 6100.2 6151.5 6116.8 6174.0 6179.0 6132.7 5989.6 5976.2 5892.3 5836.1 5835.8 5844.1 5910.7 5837.0 5809.7 5736.1 5632.5 5594.1 5680.4 5587.6 5432.8 5462.2 5399.5 5455.0}\
    ]

    array set ldeaths {
        description {Monthly Deaths from Lung Diseases in the UK}
        start 1974-01-01 period {1 month}
        values {3035 2552 2704 2554 2014 1655 1721 1524 1596 2074 2199 2512 2933 2889 2938 2497 1870 1726 1607 1545 1396 1787 2076 2837 2787 3891 3179 2011 1636 1580 1489 1300 1356 1653 2013 2823 3102 2294 2385 2444 1748 1554 1498 1361 1346 1564 1640 2293 2815 3137 2679 1969 1870 1633 1529 1366 1357 1570 1535 2491 3084 2605 2573 2143 1693 1504 1461 1354 1333 1492 1781 1915}
    }
    array set fdeaths {
        description {Monthly Deaths from Lung Diseases in the UK (females)}
        start 1974-01-01 period {1 month}
        values {901 689 827 677 522 406 441 393 387 582 578 666 830 752 785 664 467 438 421 412 343 440 531 771 767 1141 896 532 447 420 376 330 357 445 546 764 862 660 663 643 502 392 411 348 387 385 411 638 796 853 737 546 530 446 431 362 387 430 425 679 821 785 727 612 478 429 405 379 393 411 487 574}
    }
    array set mdeaths {
        description {Monthly Deaths from Lung Diseases in the UK (males)}
        start 1974-01-01 period {1 month}
        values {2134 1863 1877 1877 1492 1249 1280 1131 1209 1492 1621 1846 2103 2137 2153 1833 1403 1288 1186 1133 1053 1347 1545 2066 2020 2750 2283 1479 1189 1160 1113 970 999 1208 1467 2059 2240 1634 1722 1801 1246 1162 1087 1013 959 1179 1229 1655 2019 2284 1942 1423 1340 1187 1098 1004 970 1140 1110 1812 2263 1820 1846 1531 1215 1075 1056 975 940 1081 1294 1341}
    }

    array set JohnsonJohnson {
        description {Quarterly Earnings per Johnson & Johnson Share}
        start 1960-01-01 period {3 months}
        values {0.71 0.63 0.85 0.44 0.61 0.69 0.92 0.55 0.72 0.77 0.92 0.6 0.83 0.8 1 0.77 0.92 1 1.24 1 1.16 1.3 1.45 1.25 1.26 1.38 1.86 1.56 1.53 1.59 1.83 1.86 1.53 2.07 2.34 2.25 2.16 2.43 2.7 2.25 2.79 3.42 3.69 3.6 3.6 4.32 4.32 4.05 4.86 5.04 5.04 4.41 5.58 5.85 6.57 5.31 6.03 6.39 6.93 5.85 6.93 7.74 7.83 6.12 7.74 8.91 8.28 6.84 9.54 10.26 9.54 8.73 11.88 12.06 12.15 8.91 14.04 12.96 14.85 9.99 16.2 14.67 16.02 11.61}
    }

    array set lh {
        description {Luteinizing Hormone in Blood Samples}
        period {10 minutes}
        values {2.4 2.4 2.4 2.2 2.1 1.5 2.3 2.3 2.5 2 1.9 1.7 2.2 1.8 3.2 3.2 2.7 2.2 2.2 1.9 1.9 1.8 2.7 3 2.3 2 2 2.9 2.9 2.7 2.7 2.3 2.6 2.4 1.8 1.7 1.5 1.4 2.1 3.3 3.5 3.5 3.1 2.6 2.1 3.4 3 2.9}
    }

    array set nhtemp {
        description {Average Yearly Temperatures in New Haven}
        start 1912-01-01 period {1 year}
        values {49.9 52.3 49.4 51.1 49.4 47.9 49.8 50.9 49.3 51.9 50.8 49.6 49.3 50.6 48.4 50.7 50.9 50.6 51.5 52.8 51.8 51.1 49.8 50.2 50.4 51.6 51.8 50.9 48.8 51.7 51 50.6 51.7 51.5 52.1 51.3 51 54 51.4 52.7 53.1 54.6 52 52 50.9 52.6 50.2 52.6 51.6 51.9 50.5 50.9 51.7 51.4 51.7 50.8 51.9 51.8 51.9 53}
    }

    array set UKDriverDeaths {
        description {Road Casualties in Great Britain 1969-84}
        start 1969-01-01 period {1 month}
        values {1687 1508 1507 1385 1632 1511 1559 1630 1579 1653 2152 2148 1752 1765 1717 1558 1575 1520 1805 1800 1719 2008 2242 2478 2030 1655 1693 1623 1805 1746 1795 1926 1619 1992 2233 2192 2080 1768 1835 1569 1976 1853 1965 1689 1778 1976 2397 2654 2097 1963 1677 1941 2003 1813 2012 1912 2084 2080 2118 2150 1608 1503 1548 1382 1731 1798 1779 1887 2004 2077 2092 2051 1577 1356 1652 1382 1519 1421 1442 1543 1656 1561 1905 2199 1473 1655 1407 1395 1530 1309 1526 1327 1627 1748 1958 2274 1648 1401 1411 1403 1394 1520 1528 1643 1515 1685 2000 2215 1956 1462 1563 1459 1446 1622 1657 1638 1643 1683 2050 2262 1813 1445 1762 1461 1556 1431 1427 1554 1645 1653 2016 2207 1665 1361 1506 1360 1453 1522 1460 1552 1548 1827 1737 1941 1474 1458 1542 1404 1522 1385 1641 1510 1681 1938 1868 1726 1456 1445 1456 1365 1487 1558 1488 1684 1594 1850 1998 2079 1494 1057 1218 1168 1236 1076 1174 1139 1427 1487 1483 1513 1357 1165 1282 1110 1297 1185 1222 1284 1444 1575 1737 1763}
    }

    array set UKgas {
        description {UK Quarterly Gas Consumption}
        start 1960-01-01 period {3 months}
        values {160.1 129.7 84.8 120.1 160.1 124.9 84.8 116.9 169.7 140.9 89.7 123.3 187.3 144.1 92.9 120.1 176.1 147.3 89.7 123.3 185.7 155.3 99.3 131.3 200.1 161.7 102.5 136.1 204.9 176.1 112.1 140.9 227.3 195.3 115.3 142.5 244.9 214.5 118.5 153.7 244.9 216.1 188.9 142.5 301 196.9 136.1 267.3 317 230.5 152.1 336.2 371.4 240.1 158.5 355.4 449.9 286.6 179.3 403.4 491.5 321.8 177.7 409.8 593.9 329.8 176.1 483.5 584.3 395.4 187.3 485.1 669.2 421 216.1 509.1 827.7 467.5 209.7 542.7 840.5 414.6 217.7 670.8 848.5 437 209.7 701.2 925.3 443.4 214.5 683.6 917.3 515.5 224.1 694.8 989.4 477.1 233.7 730 1087 534.7 281.8 787.6 1163.9 613.1 347.4 782.8}
    }

    array set USAccDeaths {
        description {Accidental Deaths in the US 1973-1979}
        start 1973-01-01 period {1 month}
        values {9007 8106 8928 9137 10017 10826 11317 10744 9713 9938 9161 8927 7750 6981 8038 8422 8714 9512 10120 9823 8743 9129 8710 8680 8162 7306 8124 7870 9387 9556 10093 9620 8285 8466 8160 8034 7717 7461 7767 7925 8623 8945 10078 9179 8037 8488 7874 8647 7792 6957 7726 8106 8890 9299 10625 9302 8314 8850 8265 8796 7836 6892 7791 8192 9115 9434 10484 9827 9110 9070 8633 9240 7798 7406 8363 8460 9217 9316}
    }

    array set WWWusage {
        description {number of users connected to the Internet through a server every minute}
        period {1 minute}
        values {88 84 85 85 84 85 83 85 88 89 91 99 104 112 126 138 146 151 150 148 147 149 143 132 131 139 147 150 148 145 140 134 131 131 129 126 126 132 137 140 142 150 159 167 170 171 172 172 174 175 172 172 174 174 169 165 156 142 131 121 112 104 102 99 99 95 88 84 84 87 89 88 85 86 89 91 91 94 101 110 121 135 145 149 156 165 171 175 177 182 193 204 208 210 215 222 228 226 222 220}
    }


}


namespace eval datasets {                                                                             ;# some commonly shaped series

    array set sinus {
        description {sine wave (period: 1 hour)}
        period {30 seconds}
        values {
0.5 0.526167978121 0.552264231634 0.57821723252 0.603955845409 0.629409522551 0.654508497187 0.679183974773 0.703368321538 0.72699524987 0.75 0.772319517508 0.793892626146 0.814660195525 0.834565303179 0.853553390593 0.871572412739 0.888572980728 0.904508497187 0.919335283973 0.933012701892 0.945503262094 0.956772728821 0.966790213249 0.975528258148 0.982962913145 0.989073800367 0.993844170298 0.997260947684 0.999314767377 1.0 0.999314767377 0.997260947684 0.993844170298 0.989073800367 0.982962913145 0.975528258148 0.966790213249 0.956772728821 0.945503262094 0.933012701892 0.919335283973 0.904508497187 0.888572980728 0.871572412739 0.853553390593 0.834565303179 0.814660195525 0.793892626146 0.772319517508 0.75 0.72699524987 0.703368321538 0.679183974773 0.654508497187 0.629409522551 0.603955845409 0.57821723252 0.552264231634 0.526167978121 0.5 0.473832021879 0.447735768366 0.42178276748 0.396044154591 0.370590477449 0.345491502813 0.320816025227 0.296631678462 0.27300475013 0.25 0.227680482492 0.206107373854 0.185339804475 0.165434696821 0.146446609407 0.128427587261 0.111427019272 0.0954915028125 0.0806647160273 0.0669872981078 0.0544967379058 0.0432272711787 0.0332097867514 0.0244717418524 0.0170370868555 0.0109261996331 0.00615582970243 0.00273905231586 0.000685232622713 0.0 0.000685232622713 0.00273905231586 0.00615582970243 0.0109261996331 0.0170370868555 0.0244717418524 0.0332097867514 0.0432272711787 0.0544967379058 0.0669872981078 0.0806647160273 0.0954915028125 0.111427019272 0.128427587261 0.146446609407 0.165434696821 0.185339804475 0.206107373854 0.227680482492 0.25 0.27300475013 0.296631678462 0.320816025227 0.345491502813 0.370590477449 0.396044154591 0.42178276748 0.447735768366 0.473832021879 0.5 0.526167978121 0.552264231634 0.57821723252 0.603955845409 0.629409522551 0.654508497187 0.679183974773 0.703368321538 0.72699524987 0.75 0.772319517508 0.793892626146 0.814660195525 0.834565303179 0.853553390593 0.871572412739 0.888572980728 0.904508497187 0.919335283973 0.933012701892 0.945503262094 0.956772728821 0.966790213249 0.975528258148 0.982962913145 0.989073800367 0.993844170298 0.997260947684 0.999314767377 1.0 0.999314767377 0.997260947684 0.993844170298 0.989073800367 0.982962913145 0.975528258148 0.966790213249 0.956772728821 0.945503262094 0.933012701892 0.919335283973 0.904508497187 0.888572980728 0.871572412739 0.853553390593 0.834565303179 0.814660195525 0.793892626146 0.772319517508 0.75 0.72699524987 0.703368321538 0.679183974773 0.654508497187 0.629409522551 0.603955845409 0.57821723252 0.552264231634 0.526167978121 0.5 0.473832021879 0.447735768366 0.42178276748 0.396044154591 0.370590477449 0.345491502813 0.320816025227 0.296631678462 0.27300475013 0.25 0.227680482492 0.206107373854 0.185339804475 0.165434696821 0.146446609407 0.128427587261 0.111427019272 0.0954915028125 0.0806647160273 0.0669872981078 0.0544967379058 0.0432272711787 0.0332097867514 0.0244717418524 0.0170370868555 0.0109261996331 0.00615582970243 0.00273905231586 0.000685232622713 0.0 0.000685232622713 0.00273905231586 0.00615582970243 0.0109261996331 0.0170370868555 0.0244717418524 0.0332097867514 0.0432272711787 0.0544967379058 0.0669872981078 0.0806647160273 0.0954915028125 0.111427019272 0.128427587261 0.146446609407 0.165434696821 0.185339804475 0.206107373854 0.227680482492 0.25 0.27300475013 0.296631678462 0.320816025227 0.345491502813 0.370590477449 0.396044154591 0.42178276748 0.447735768366 0.473832021879 0.5 0.526167978121 0.552264231634 0.57821723252 0.603955845409 0.629409522551 0.654508497187 0.679183974773 0.703368321538 0.72699524987 0.75 0.772319517508 0.793892626146 0.814660195525 0.834565303179 0.853553390593 0.871572412739 0.888572980728 0.904508497187 0.919335283973 0.933012701892 0.945503262094 0.956772728821 0.966790213249 0.975528258148 0.982962913145 0.989073800367 0.993844170298 0.997260947684 0.999314767377 1.0 0.999314767377 0.997260947684 0.993844170298 0.989073800367 0.982962913145 0.975528258148 0.966790213249 0.956772728821 0.945503262094 0.933012701892 0.919335283973 0.904508497187 0.888572980728 0.871572412739 0.853553390593 0.834565303179 0.814660195525 0.793892626146 0.772319517508 0.75 0.72699524987 0.703368321538 0.679183974773 0.654508497187 0.629409522551 0.603955845409 0.57821723252 0.552264231634 0.526167978121 0.5 0.473832021879 0.447735768366 0.42178276748 0.396044154591 0.370590477449 0.345491502813 0.320816025227 0.296631678462 0.27300475013 0.25 0.227680482492 0.206107373854 0.185339804475 0.165434696821 0.146446609407 0.128427587261 0.111427019272 0.0954915028125 0.0806647160273 0.0669872981078 0.0544967379058 0.0432272711787 0.0332097867514 0.0244717418524 0.0170370868555 0.0109261996331 0.00615582970243 0.00273905231586 0.000685232622713 0.0 0.000685232622713 0.00273905231586 0.00615582970243 0.0109261996331 0.0170370868555 0.0244717418524 0.0332097867514 0.0432272711787 0.0544967379058 0.0669872981078 0.0806647160273 0.0954915028125 0.111427019272 0.128427587261 0.146446609407 0.165434696821 0.185339804475 0.206107373854 0.227680482492 0.25 0.27300475013 0.296631678462 0.320816025227 0.345491502813 0.370590477449 0.396044154591 0.42178276748 0.447735768366 0.473832021879 0.5 0.526167978121 0.552264231634 0.57821723252 0.603955845409 0.629409522551 0.654508497187 0.679183974773 0.703368321538 0.72699524987 0.75 0.772319517508 0.793892626146 0.814660195525 0.834565303179 0.853553390593 0.871572412739 0.888572980728 0.904508497187 0.919335283973 0.933012701892 0.945503262094 0.956772728821 0.966790213249 0.975528258148 0.982962913145 0.989073800367 0.993844170298 0.997260947684 0.999314767377 1.0 0.999314767377 0.997260947684 0.993844170298 0.989073800367 0.982962913145 0.975528258148 0.966790213249 0.956772728821 0.945503262094 0.933012701892 0.919335283973 0.904508497187 0.888572980728 0.871572412739 0.853553390593 0.834565303179 0.814660195525 0.793892626146 0.772319517508 0.75 0.72699524987 0.703368321538 0.679183974773 0.654508497187 0.629409522551 0.603955845409 0.57821723252 0.552264231634 0.526167978121 0.5 0.473832021879 0.447735768366 0.42178276748 0.396044154591 0.370590477449 0.345491502813 0.320816025227 0.296631678462 0.27300475013 0.25 0.227680482492 0.206107373854 0.185339804475 0.165434696821 0.146446609407 0.128427587261 0.111427019272 0.0954915028125 0.0806647160273 0.0669872981078 0.0544967379058 0.0432272711787 0.0332097867514 0.0244717418524 0.0170370868555 0.0109261996331 0.00615582970243 0.00273905231586 0.000685232622713 0.0 0.000685232622713 0.00273905231586 0.00615582970243 0.0109261996331 0.0170370868555 0.0244717418524 0.0332097867514 0.0432272711787 0.0544967379058 0.0669872981078 0.0806647160273 0.0954915028125 0.111427019272 0.128427587261 0.146446609407 0.165434696821 0.185339804475 0.206107373854 0.227680482492 0.25 0.27300475013 0.296631678462 0.320816025227 0.345491502813 0.370590477449 0.396044154591 0.42178276748 0.447735768366 0.473832021879 0.5 0.526167978121 0.552264231634 0.57821723252 0.603955845409 0.629409522551 0.654508497187 0.679183974773 0.703368321538 0.72699524987 0.75 0.772319517508 0.793892626146 0.814660195525 0.834565303179 0.853553390593 0.871572412739 0.888572980728 0.904508497187 0.919335283973 0.933012701892 0.945503262094 0.956772728821 0.966790213249 0.975528258148 0.982962913145 0.989073800367 0.993844170298 0.997260947684 0.999314767377 1.0 0.999314767377 0.997260947684 0.993844170298 0.989073800367 0.982962913145 0.975528258148 0.966790213249 0.956772728821 0.945503262094 0.933012701892 0.919335283973 0.904508497187 0.888572980728 0.871572412739 0.853553390593 0.834565303179 0.814660195525 0.793892626146 0.772319517508 0.75 0.72699524987 0.703368321538 0.679183974773 0.654508497187 0.629409522551 0.603955845409 0.57821723252 0.552264231634 0.526167978121 0.5 0.473832021879 0.447735768366 0.42178276748 0.396044154591 0.370590477449 0.345491502813 0.320816025227 0.296631678462 0.27300475013 0.25 0.227680482492 0.206107373854 0.185339804475 0.165434696821 0.146446609407 0.128427587261 0.111427019272 0.0954915028125 0.0806647160273 0.0669872981078 0.0544967379058 0.0432272711787 0.0332097867514 0.0244717418524 0.0170370868555 0.0109261996331 0.00615582970243 0.00273905231586 0.000685232622713 0.0 0.000685232622713 0.00273905231586 0.00615582970243 0.0109261996331 0.0170370868555 0.0244717418524 0.0332097867514 0.0432272711787 0.0544967379058 0.0669872981078 0.0806647160273 0.0954915028125 0.111427019272 0.128427587261 0.146446609407 0.165434696821 0.185339804475 0.206107373854 0.227680482492 0.25 0.27300475013 0.296631678462 0.320816025227 0.345491502813 0.370590477449 0.396044154591 0.42178276748 0.447735768366 0.473832021879 0.5 0.526167978121 0.552264231634 0.57821723252 0.603955845409 0.629409522551 0.654508497187 0.679183974773 0.703368321538 0.72699524987 0.75 0.772319517508 0.793892626146 0.814660195525 0.834565303179 0.853553390593 0.871572412739 0.888572980728 0.904508497187 0.919335283973 0.933012701892 0.945503262094 0.956772728821 0.966790213249 0.975528258148 0.982962913145 0.989073800367 0.993844170298 0.997260947684 0.999314767377 1.0 0.999314767377 0.997260947684 0.993844170298 0.989073800367 0.982962913145 0.975528258148 0.966790213249 0.956772728821 0.945503262094 0.933012701892 0.919335283973 0.904508497187 0.888572980728 0.871572412739 0.853553390593 0.834565303179 0.814660195525 0.793892626146 0.772319517508 0.75 0.72699524987 0.703368321538 0.679183974773 0.654508497187 0.629409522551 0.603955845409 0.57821723252 0.552264231634 0.526167978121 0.5 0.473832021879 0.447735768366 0.42178276748 0.396044154591 0.370590477449 0.345491502813 0.320816025227 0.296631678462 0.27300475013 0.25 0.227680482492 0.206107373854 0.185339804475 0.165434696821 0.146446609407 0.128427587261 0.111427019272 0.0954915028125 0.0806647160273 0.0669872981078 0.0544967379058 0.0432272711787 0.0332097867514 0.0244717418524 0.0170370868555 0.0109261996331 0.00615582970243 0.00273905231586 0.000685232622713 0.0 0.000685232622713 0.00273905231586 0.00615582970243 0.0109261996331 0.0170370868555 0.0244717418524 0.0332097867514 0.0432272711787 0.0544967379058 0.0669872981078 0.0806647160273 0.0954915028125 0.111427019272 0.128427587261 0.146446609407 0.165434696821 0.185339804475 0.206107373854 0.227680482492 0.25 0.27300475013 0.296631678462 0.320816025227 0.345491502813 0.370590477449 0.396044154591 0.42178276748 0.447735768366 0.473832021879
        }
    }

    array set square {
        description {square wave (period: 1 hour)}
        period {30 seconds}
        values {1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0}
    }

    array set triangle {
        description {triangular wave (period: 1 hour)}
        period {30 seconds}
        values {0.0 0.0166666666667 0.0333333333333 0.05 0.0666666666667 0.0833333333333 0.1 0.116666666667 0.133333333333 0.15 0.166666666667 0.183333333333 0.2 0.216666666667 0.233333333333 0.25 0.266666666667 0.283333333333 0.3 0.316666666667 0.333333333333 0.35 0.366666666667 0.383333333333 0.4 0.416666666667 0.433333333333 0.45 0.466666666667 0.483333333333 0.5 0.516666666667 0.533333333333 0.55 0.566666666667 0.583333333333 0.6 0.616666666667 0.633333333333 0.65 0.666666666667 0.683333333333 0.7 0.716666666667 0.733333333333 0.75 0.766666666667 0.783333333333 0.8 0.816666666667 0.833333333333 0.85 0.866666666667 0.883333333333 0.9 0.916666666667 0.933333333333 0.95 0.966666666667 0.983333333333 1.0 0.983333333333 0.966666666667 0.95 0.933333333333 0.916666666667 0.9 0.883333333333 0.866666666667 0.85 0.833333333333 0.816666666667 0.8 0.783333333333 0.766666666667 0.75 0.733333333333 0.716666666667 0.7 0.683333333333 0.666666666667 0.65 0.633333333333 0.616666666667 0.6 0.583333333333 0.566666666667 0.55 0.533333333333 0.516666666667 0.5 0.483333333333 0.466666666667 0.45 0.433333333333 0.416666666667 0.4 0.383333333333 0.366666666667 0.35 0.333333333333 0.316666666667 0.3 0.283333333333 0.266666666667 0.25 0.233333333333 0.216666666667 0.2 0.183333333333 0.166666666667 0.15 0.133333333333 0.116666666667 0.1 0.0833333333333 0.0666666666667 0.05 0.0333333333333 0.0166666666667 0.0 0.0166666666667 0.0333333333333 0.05 0.0666666666667 0.0833333333333 0.1 0.116666666667 0.133333333333 0.15 0.166666666667 0.183333333333 0.2 0.216666666667 0.233333333333 0.25 0.266666666667 0.283333333333 0.3 0.316666666667 0.333333333333 0.35 0.366666666667 0.383333333333 0.4 0.416666666667 0.433333333333 0.45 0.466666666667 0.483333333333 0.5 0.516666666667 0.533333333333 0.55 0.566666666667 0.583333333333 0.6 0.616666666667 0.633333333333 0.65 0.666666666667 0.683333333333 0.7 0.716666666667 0.733333333333 0.75 0.766666666667 0.783333333333 0.8 0.816666666667 0.833333333333 0.85 0.866666666667 0.883333333333 0.9 0.916666666667 0.933333333333 0.95 0.966666666667 0.983333333333 1.0 0.983333333333 0.966666666667 0.95 0.933333333333 0.916666666667 0.9 0.883333333333 0.866666666667 0.85 0.833333333333 0.816666666667 0.8 0.783333333333 0.766666666667 0.75 0.733333333333 0.716666666667 0.7 0.683333333333 0.666666666667 0.65 0.633333333333 0.616666666667 0.6 0.583333333333 0.566666666667 0.55 0.533333333333 0.516666666667 0.5 0.483333333333 0.466666666667 0.45 0.433333333333 0.416666666667 0.4 0.383333333333 0.366666666667 0.35 0.333333333333 0.316666666667 0.3 0.283333333333 0.266666666667 0.25 0.233333333333 0.216666666667 0.2 0.183333333333 0.166666666667 0.15 0.133333333333 0.116666666667 0.1 0.0833333333333 0.0666666666667 0.05 0.0333333333333 0.0166666666667 0.0 0.0166666666667 0.0333333333333 0.05 0.0666666666667 0.0833333333333 0.1 0.116666666667 0.133333333333 0.15 0.166666666667 0.183333333333 0.2 0.216666666667 0.233333333333 0.25 0.266666666667 0.283333333333 0.3 0.316666666667 0.333333333333 0.35 0.366666666667 0.383333333333 0.4 0.416666666667 0.433333333333 0.45 0.466666666667 0.483333333333 0.5 0.516666666667 0.533333333333 0.55 0.566666666667 0.583333333333 0.6 0.616666666667 0.633333333333 0.65 0.666666666667 0.683333333333 0.7 0.716666666667 0.733333333333 0.75 0.766666666667 0.783333333333 0.8 0.816666666667 0.833333333333 0.85 0.866666666667 0.883333333333 0.9 0.916666666667 0.933333333333 0.95 0.966666666667 0.983333333333 1.0 0.983333333333 0.966666666667 0.95 0.933333333333 0.916666666667 0.9 0.883333333333 0.866666666667 0.85 0.833333333333 0.816666666667 0.8 0.783333333333 0.766666666667 0.75 0.733333333333 0.716666666667 0.7 0.683333333333 0.666666666667 0.65 0.633333333333 0.616666666667 0.6 0.583333333333 0.566666666667 0.55 0.533333333333 0.516666666667 0.5 0.483333333333 0.466666666667 0.45 0.433333333333 0.416666666667 0.4 0.383333333333 0.366666666667 0.35 0.333333333333 0.316666666667 0.3 0.283333333333 0.266666666667 0.25 0.233333333333 0.216666666667 0.2 0.183333333333 0.166666666667 0.15 0.133333333333 0.116666666667 0.1 0.0833333333333 0.0666666666667 0.05 0.0333333333333 0.0166666666667 0.0 0.0166666666667 0.0333333333333 0.05 0.0666666666667 0.0833333333333 0.1 0.116666666667 0.133333333333 0.15 0.166666666667 0.183333333333 0.2 0.216666666667 0.233333333333 0.25 0.266666666667 0.283333333333 0.3 0.316666666667 0.333333333333 0.35 0.366666666667 0.383333333333 0.4 0.416666666667 0.433333333333 0.45 0.466666666667 0.483333333333 0.5 0.516666666667 0.533333333333 0.55 0.566666666667 0.583333333333 0.6 0.616666666667 0.633333333333 0.65 0.666666666667 0.683333333333 0.7 0.716666666667 0.733333333333 0.75 0.766666666667 0.783333333333 0.8 0.816666666667 0.833333333333 0.85 0.866666666667 0.883333333333 0.9 0.916666666667 0.933333333333 0.95 0.966666666667 0.983333333333 1.0 0.983333333333 0.966666666667 0.95 0.933333333333 0.916666666667 0.9 0.883333333333 0.866666666667 0.85 0.833333333333 0.816666666667 0.8 0.783333333333 0.766666666667 0.75 0.733333333333 0.716666666667 0.7 0.683333333333 0.666666666667 0.65 0.633333333333 0.616666666667 0.6 0.583333333333 0.566666666667 0.55 0.533333333333 0.516666666667 0.5 0.483333333333 0.466666666667 0.45 0.433333333333 0.416666666667 0.4 0.383333333333 0.366666666667 0.35 0.333333333333 0.316666666667 0.3 0.283333333333 0.266666666667 0.25 0.233333333333 0.216666666667 0.2 0.183333333333 0.166666666667 0.15 0.133333333333 0.116666666667 0.1 0.0833333333333 0.0666666666667 0.05 0.0333333333333 0.0166666666667 0.0 0.0166666666667 0.0333333333333 0.05 0.0666666666667 0.0833333333333 0.1 0.116666666667 0.133333333333 0.15 0.166666666667 0.183333333333 0.2 0.216666666667 0.233333333333 0.25 0.266666666667 0.283333333333 0.3 0.316666666667 0.333333333333 0.35 0.366666666667 0.383333333333 0.4 0.416666666667 0.433333333333 0.45 0.466666666667 0.483333333333 0.5 0.516666666667 0.533333333333 0.55 0.566666666667 0.583333333333 0.6 0.616666666667 0.633333333333 0.65 0.666666666667 0.683333333333 0.7 0.716666666667 0.733333333333 0.75 0.766666666667 0.783333333333 0.8 0.816666666667 0.833333333333 0.85 0.866666666667 0.883333333333 0.9 0.916666666667 0.933333333333 0.95 0.966666666667 0.983333333333 1.0 0.983333333333 0.966666666667 0.95 0.933333333333 0.916666666667 0.9 0.883333333333 0.866666666667 0.85 0.833333333333 0.816666666667 0.8 0.783333333333 0.766666666667 0.75 0.733333333333 0.716666666667 0.7 0.683333333333 0.666666666667 0.65 0.633333333333 0.616666666667 0.6 0.583333333333 0.566666666667 0.55 0.533333333333 0.516666666667 0.5 0.483333333333 0.466666666667 0.45 0.433333333333 0.416666666667 0.4 0.383333333333 0.366666666667 0.35 0.333333333333 0.316666666667 0.3 0.283333333333 0.266666666667 0.25 0.233333333333 0.216666666667 0.2 0.183333333333 0.166666666667 0.15 0.133333333333 0.116666666667 0.1 0.0833333333333 0.0666666666667 0.05 0.0333333333333 0.0166666666667 0.0 0.0166666666667 0.0333333333333 0.05 0.0666666666667 0.0833333333333 0.1 0.116666666667 0.133333333333 0.15 0.166666666667 0.183333333333 0.2 0.216666666667 0.233333333333 0.25 0.266666666667 0.283333333333 0.3 0.316666666667 0.333333333333 0.35 0.366666666667 0.383333333333 0.4 0.416666666667 0.433333333333 0.45 0.466666666667 0.483333333333 0.5 0.516666666667 0.533333333333 0.55 0.566666666667 0.583333333333 0.6 0.616666666667 0.633333333333 0.65 0.666666666667 0.683333333333 0.7 0.716666666667 0.733333333333 0.75 0.766666666667 0.783333333333 0.8 0.816666666667 0.833333333333 0.85 0.866666666667 0.883333333333 0.9 0.916666666667 0.933333333333 0.95 0.966666666667 0.983333333333 1.0 0.983333333333 0.966666666667 0.95 0.933333333333 0.916666666667 0.9 0.883333333333 0.866666666667 0.85 0.833333333333 0.816666666667 0.8 0.783333333333 0.766666666667 0.75 0.733333333333 0.716666666667 0.7 0.683333333333 0.666666666667 0.65 0.633333333333 0.616666666667 0.6 0.583333333333 0.566666666667 0.55 0.533333333333 0.516666666667 0.5 0.483333333333 0.466666666667 0.45 0.433333333333 0.416666666667 0.4 0.383333333333 0.366666666667 0.35 0.333333333333 0.316666666667 0.3 0.283333333333 0.266666666667 0.25 0.233333333333 0.216666666667 0.2 0.183333333333 0.166666666667 0.15 0.133333333333 0.116666666667 0.1 0.0833333333333 0.0666666666667 0.05 0.0333333333333 0.0166666666667}
    }

}
