# copyright (C) 1997-2006 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: kernmods.tcl,v 1.34 2006/04/04 18:33:31 jfontain Exp $


package provide kernmods [lindex {$Revision: 1.34 $} 1]
package require network 1
package require stooop 4.1
namespace import stooop::*
package require switched
if {[catch {package require Thread 2.5}]} {
    namespace eval kernmods {variable threads 0}
} else {                                                                                  ;# load thread worker class implementation
    package require threads 1
    namespace eval kernmods {variable threads 1}
}
package require linetask 1
package require hashes


namespace eval kernmods {

    variable flag                                                                                                ;# module flags map
    array set flag {autoclean C deleted D uninitialized I unused U}

    array set data {
        updates 0
        0,label name 0,type ascii 0,message {Linux kernel module name}
        1,label size 1,type integer 1,message {size in bytes}
        2,label used 2,type integer 2,message {use count}
        3,label state 3,type ascii 3,message "2.6 kernel: Unloading, Loading or Live\n2.4 kernel: auto(C)lean, (D)eleted, un(I)nitialized, (U)nused"
        4,label referring 4,type ascii 4,message {list of referring modules} 4,anchor left
        sort {0 increasing}
        persistent 1 64Bits 1
        switches {-C 0 --daemon 0 -i 1 -p 1 -r 1 --remote 1}
    }
    set file [open kernmods.htm]
    set data(helpText) [::read $file]                                                         ;# initialize HTML help data from file
    close $file
    unset file

    proc initialize {optionsName} {
        upvar 1 $optionsName options
        variable local
        variable remote
        variable data
        variable threads                                                                     ;# whether threads package is available

        catch {set locator $options(-r)}
        catch {set locator $options(--remote)}                                                                  ;# favor long option
        if {[info exists locator]} {                                                                                  ;# remote host
            set data(pollTimes) {60 10 20 30 120 300 600}                                ;# poll less often when remotely monitoring
        } else {                                                                                                       ;# local host
            set data(pollTimes) {30 5 10 20 60 120 300 600}
            set local(modules) [open /proc/modules]                                   ;# keep local file open for better performance
            return                                                                                               ;# local monitoring
        }
        # for remote monitoring, decode protocol, remote user and host
        foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
        set remote(rsh) [string equal $remote(protocol) rsh]
        set data(identifier) kernmods($remote(host))
        # important: pack data in a single line using special control separator characters
        set remote(command) {cat /proc/modules 2>&1 | tr '\n' '\v'}
        if {$::tcl_platform(platform) eq "unix"} {
            if {$remote(rsh)} {
                set command "rsh -n -l $remote(user) $remote(host) {$remote(command)}"
            } else {
                set command ssh
                if {[info exists options(-C)]} {append command { -C}}                                            ;# data compression
                if {[info exists options(-i)]} {append command " -i \"$options(-i)\""}                              ;# identity file
                if {[info exists options(-p)]} {append command " -p $options(-p)"}                                           ;# port
                append command " -T -l $remote(user) $remote(host)"
            }
        } else {                                                                                                          ;# windows
            if {$remote(rsh)} {
                error {use -r(--remote) ssh://session syntax (see help)}
            }
            set remote(rsh) 0
            set command "plink -ssh -batch -T $remote(host)"       ;# note: host must be a putty session and pageant must be running
        }
        if {$remote(rsh)} {
            set access r                                                                            ;# writing to pipe is not needed
        } else {
            set access r+                                                                                     ;# bi-directional pipe
            # terminate remote command output by a newline so that the buffered stream flushes it through the pipe as soon as the
            # remote data becomes available:
            append remote(command) {; echo}
        }
        set remote(task) [new lineTask\
            -command $command -callback kernmods::read -begin 0 -access $access -translation lf -threaded $threads\
        ]
        if {![info exists options(--daemon)] && !$remote(rsh)} {             ;# for ssh, detect errors early when not in daemon mode
            lineTask::begin $remote(task)
        }                                                       ;# note: for rsh, shell and command need be restarted at each update
        set remote(busy) 0
    }

    proc update {} {
        variable remote
        variable local

        if {[info exists remote]} {
            if {$remote(busy)} return                                               ;# core invocation while waiting for remote data
            set remote(busy) 1
            if {[lineTask::end $remote(task)]} {                                                           ;# rsh or ssh daemon mode
                lineTask::begin $remote(task)                       ;# note: for rsh, shell and command are restarted here each time
            }
            if {!$remote(rsh)} {
                lineTask::write $remote(task) $remote(command)             ;# start data retrieval by sending command to remote side
            }
        } else {
            seek $local(modules) 0                                                                  ;# rewind before retrieving data
            process [split [::read -nonewline $local(modules)] \n]
        }
    }

    proc process {lines} {                                                          ;# process modules data lines and update display
        variable flag
        variable data

        foreach line $lines {
            if {[scan $line {%s %u %s %s %s %x} name size count modules state address] == 6} {                         ;# 2.6 kernel
                # from kernel/module.c:
                #   - count = number or -
                #   - users = comma separated list of module names or -
                #   - state = Unloading, Loading or Live
                if {$count eq "-"} {set count ?}
                if {$modules eq "-"} {
                    set modules {}
                } else {
                    set modules [split [string trimright $modules ,] ,]
                }
            } elseif {[regexp {^([^ ]+) +(\d+) +(\d+) *(.*)$} $line dummy name size count comments]} {                 ;# 2.4 kernel
                set state {}
                set modules {}
                regsub {\[} $comments \{ comments                         ;# replace brackets around modules in order to make a list
                regsub {\]} $comments \} comments                    ;# as comments format is: (flag) ... (flag) [module ... module]
                foreach item $comments {
                    if {[scan $item {(%[^)]} literal]} {                                                                     ;# flag
                        append state $flag($literal)
                    } else {
                        set modules $item                                                                  ;# referring modules list
                    }
                }
            } else continue                                                               ;# unrecognizable line (should not happen)
            if {[catch {set row [hash64::string $name 1]} message]} {                                             ;# repeatable mode
                flashMessage "warning: duplicate row ($name) removed: $message"                               ;# should be very rare
                continue
            }
            if {![info exists data($row,0)]} {                                                                          ;# new entry
                set data($row,0) $name                                                                     ;# initialize static data
                set data($row,1) $size
            }
            set data($row,2) $count
            set data($row,3) $state
            set data($row,4) $modules
            set current($row) {}
        }
        foreach name [array names data *,0] {                                                         ;# cleanup disappeared entries
            set row [lindex [split $name ,] 0]
            if {![info exists current($row)]} {array unset data $row,\[0-9\]*}
        }
        if {![info exists current] && ([string length [lindex $lines 0]] > 0)} {
            set message "invalid data: [lindex $lines 0]"
            if {[llength $lines] > 1} {append message "..."}
            flashMessage $message
        }
        incr data(updates)
    }

    proc read {line} {                                       ;# read remote data now that it is available and possibly handle errors
        variable remote

        switch $lineTask::($remote(task),event) {
            end {
                # either valid data availability as rsh connection was closed, or connection broken for ssh, in which case remote
                # shell command will be attempted to be restarted at next update
            }
            error {                                                                              ;# some communication error occured
                set message "error on remote data: $lineTask::($remote(task),error)"
            }
            timeout {                                                                         ;# remote host did not respond in time
                set message "timeout on remote host: $remote(host)"
            }
        }
        # note: in case of an unexpected event, task insures that line is empty
        if {[info exists message]} {
            flashMessage $message
        }
        process [split [string trimright $line \v] \v]
        set remote(busy) 0
    }

}
