# copyright (C) 1997-2006 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

# $Id: hddtemp.tcl,v 1.16 2006/01/28 19:16:59 jfontain Exp $


package provide hddtemp [lindex {$Revision: 1.16 $} 1]
package require network 1
package require stooop 4.1
namespace import stooop::*
package require switched
if {[catch {package require Thread 2.5}]} {
    namespace eval hddtemp {variable threads 0}
} else {                                                                                  ;# load thread worker class implementation
    package require threads 1
    namespace eval hddtemp {variable threads 1}
}
package require linetask 1


namespace eval hddtemp {

    array set data {
        updates 0
        0,label disk 0,type dictionary 0,message {disk device (in the /dev directory)}
        1,label temperature 1,type integer 1,message {temperature in degrees Celsius}
        sort {0 increasing}
        pollTimes {60 10 20 30 120 300 600}
        persistent 1
        switches {-C 0 --daemon 0 --devices 1 -i 1 -p 1 --path 1 -r 1 --remote 1}
    }

    set file [open hddtemp.htm]
    set data(helpText) [::read $file]                                                         ;# initialize HTML help data from file
    close $file
    unset file

    proc initialize {optionsName} {
        upvar 1 $optionsName options
        variable local
        variable remote
        variable data
        variable threads                                                                     ;# whether threads package is available
        variable devices

        set path /usr/sbin                                                                       ;# default path for hddtemp command
        catch {set path $options(--path)}                                                                ;# may be overriden by user
        set path [file join $path hddtemp]
        catch {set locator $options(-r)}
        catch {set locator $options(--remote)}                                                                  ;# favor long option
        if {![info exists options(--devices)]} {
            error {missing devices (use --devices option)}
        }
        append path { -qn}
        set devices [split $options(--devices) ,]
        foreach device $devices {
            append path " [file join /dev $device]"
        }
        if {![info exists locator]} {                                                                                  ;# local host
            eval exec $path                                            ;# detect errors early by attempting immediate data retrieval
            set local(command) $path
            return                                                                                               ;# local monitoring
        }
        # for remote monitoring, decode protocol, remote user and host
        foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
        set remote(rsh) [string equal $remote(protocol) rsh]
        set data(identifier) hddtemp($remote(host))
        # important: pack data in a single line using special control separator characters
        set remote(command) "$path 2>&1 | tr '\\n' '\\v'"
        if {[string equal $::tcl_platform(platform) unix]} {
            if {$remote(rsh)} {
                set command "rsh -n -l $remote(user) $remote(host) {$remote(command)}"
            } else {
                set command ssh
                if {[info exists options(-C)]} {append command { -C}}                                            ;# data compression
                if {[info exists options(-i)]} {append command " -i \"$options(-i)\""}                              ;# identity file
                if {[info exists options(-p)]} {append command " -p $options(-p)"}                                           ;# port
                # note: redirect standard error to pipe output in order to be able to detect remote errors
                append command " -T -l $remote(user) $remote(host) 2>@ stdout"
            }
        } else {                                                                                                          ;# windows
            if {$remote(rsh)} {
                error {use -r(--remote) ssh://session syntax (see help)}
            }
            set remote(rsh) 0                                      ;# note: host must be a putty session and pageant must be running
            set command "plink -ssh -batch -T $remote(host) 2>@ stdout"
        }
        if {$remote(rsh)} {
            set access r                                                                            ;# writing to pipe is not needed
        } else {
            set access r+                                                                                     ;# bi-directional pipe
            # terminate remote command output by a newline so that the buffered stream flushes it through the pipe as soon as the
            # remote data becomes available:
            append remote(command) {; echo}
        }
        set remote(task) [new lineTask\
            -command $command -callback hddtemp::read -begin 0 -access $access -translation lf -threaded $threads\
        ]
        if {![info exists options(--daemon)] && !$remote(rsh)} {             ;# for ssh, detect errors early when not in daemon mode
            lineTask::begin $remote(task)
        }                                                       ;# note: for rsh, shell and command need be restarted at each update
        set remote(busy) 0
    }

    proc update {} {
        variable remote
        variable local

        if {[info exists remote]} {
            if {$remote(busy)} return                                               ;# core invocation while waiting for remote data
            set remote(busy) 1
            if {[lineTask::end $remote(task)]} {                                                           ;# rsh or ssh daemon mode
                lineTask::begin $remote(task)                       ;# note: for rsh, shell and command are restarted here each time
            }
            if {!$remote(rsh)} {
                lineTask::write $remote(task) $remote(command)             ;# start data retrieval by sending command to remote side
            }
        } elseif {[catch {set result [eval exec $local(command)]} message]} {           ;# immediate retrieval failure on local host
            flashMessage "hddtemp error: $message"
        } else {
            process [split $result \n]
        }
    }

    proc process {lines} {                                           ;# process hard disks temperature data lines and update display
        variable data
        variable devices

        set row 0
        foreach device $devices {
            set data($row,0) $device
            set data($row,1) ?                                                                     ;# in case there is no valid data
            incr row
        }
        set row 0
        foreach value $lines {
            if {[string is integer -strict $value]} {                        ;# valid lines only contain the temperature as a number
                set data($row,1) $value
            } else {
                flashMessage "invalid line: $value"            ;# can happen if hddtemp cannot be run by user or drive has no sensor
            }
            incr row
        }
        incr data(updates)
    }

    proc read {line} {                                       ;# read remote data now that it is available and possibly handle errors
        variable remote

        switch $lineTask::($remote(task),event) {
            end {
                # either valid data availability as rsh connection was closed, or connection broken for ssh, in which case remote
                # shell command will be attempted to be restarted at next update
            }
            error {                                                                              ;# some communication error occured
                set message "error on remote data: $lineTask::($remote(task),error)"
            }
            timeout {                                                                         ;# remote host did not respond in time
                set message "timeout on remote host: $remote(host)"
            }
        }
        # unpack list while removing extra last separator without copying to a variable for better performance, as data could be big
        # note: in case of an unexpected event, task insures that line is empty
        if {[info exists message]} {
            flashMessage $message
        }
        process [split [string trimright $line \v] \v]
        set remote(busy) 0
    }

}
