/**
 *    Copyright (C) 2021-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include <algorithm>
#include <boost/optional.hpp>
#include <iosfwd>
#include <string>

#include "mongo/base/string_data.h"
#include "mongo/bson/util/builder.h"
#include "mongo/db/namespace_string.h"
#include "mongo/db/tenant_database_name.h"
#include "mongo/db/tenant_id.h"
#include "mongo/logv2/log_attr.h"

namespace mongo {

class TenantNamespace {
public:
    TenantNamespace(const TenantNamespace& tenantNs) = default;

    /**
     * Constructs an empty TenantNamespace.
     */
    TenantNamespace() = default;

    /**
     * Constructs a TenantNamespace from the given tenantId and NamespaceString.
     *
     * If featureFlagRequireTenantID is set, tenantId is required.
     */
    TenantNamespace(boost::optional<mongo::TenantId> tenantId, NamespaceString nss);

    /**
     * Create a TenantDatabaseName from the TenantNamespace.
     */
    TenantDatabaseName createTenantDatabaseName() const;

    /**
     * Constructs a TenantNamespace from the string "ns". When the server parameter
     * "multitenancySupport” is enabled, the tenantId will be parsed separately from the database
     * name. If it is disabled, the tenantId will be parsed as a prefix of the database name, and
     * the tenantId field will be empty. For example:
     * if “multitenancySupport” is enabled, "tenant1_dbA.collA" will be parsed as:
     * _tenantId = tenant1
     * _nss = NamespaceString(dbA.collA)
     *
     * if “multitenancySupport” is disabled, "tenant1_dbA.collA" will be parsed as:
     * _tenantId = boost::none
     * _nss = NamespaceString(tenant1_dbA.collA), and the _nss,db()
     *
     * This method should only be used when reading a namespace from disk. To construct a
     * TenantNamespace otherwise, use the standard constructor above.
     *
     * If featureFlagRequireTenantID is set, tenantId is required.
     */
    static TenantNamespace parseTenantNamespaceFromDisk(StringData ns);

    boost::optional<TenantId> tenantId() const {
        return _tenantId;
    }

    StringData db() const {
        return _nss.db();
    }

    StringData coll() const {
        return _nss.coll();
    }

    const NamespaceString& getNss() const {
        return _nss;
    }

    std::string toString() const {
        if (_tenantNsStr)
            return *_tenantNsStr;

        invariant(!_tenantId);
        return _nss.ns();
    }

    // Relops among `TenantNamespace`.
    friend bool operator==(const TenantNamespace& a, const TenantNamespace& b) {
        return a.toString() == b.toString();
    }
    friend bool operator!=(const TenantNamespace& a, const TenantNamespace& b) {
        return a.toString() != b.toString();
    }
    friend bool operator<(const TenantNamespace& a, const TenantNamespace& b) {
        return a.toString() < b.toString();
    }
    friend bool operator>(const TenantNamespace& a, const TenantNamespace& b) {
        return a.toString() > b.toString();
    }
    friend bool operator<=(const TenantNamespace& a, const TenantNamespace& b) {
        return a.toString() <= b.toString();
    }
    friend bool operator>=(const TenantNamespace& a, const TenantNamespace& b) {
        return a.toString() >= b.toString();
    }

    template <typename H>
    friend H AbslHashValue(H h, const TenantNamespace& tenantNs) {
        return H::combine(std::move(h), tenantNs.toString());
    }

    friend auto logAttrs(const TenantNamespace& nss) {
        return "namespace"_attr = nss;
    }

private:
    boost::optional<TenantId> _tenantId;
    NamespaceString _nss{};
    boost::optional<std::string> _tenantNsStr;  // Only set if _tenantId exists
};

std::ostream& operator<<(std::ostream& stream, const TenantNamespace& tenantNs);
StringBuilder& operator<<(StringBuilder& builder, const TenantNamespace& tenantNs);

}  // namespace mongo
