/**
 *    Copyright (C) 2021-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#define MONGO_LOGV2_DEFAULT_COMPONENT ::mongo::logv2::LogComponent::kSharding

#include "mongo/db/s/balancer/balancer_chunk_merger_impl.h"

#include "mongo/db/s/balancer/balancer_policy.h"
#include "mongo/db/s/config/sharding_catalog_manager.h"
#include "mongo/logv2/log.h"
#include "mongo/s/balancer_configuration.h"
#include "mongo/s/grid.h"

namespace mongo {

using Progress = BalancerChunkMerger::Progress;

BalancerChunkMergerImpl::BalancerChunkMergerImpl(BalancerCommandsScheduler& scheduler,
                                                 ClusterStatistics& stats)
    : BalancerChunkMerger(), _cmdScheduler(scheduler), _clusterStats(stats) {}

BalancerChunkMergerImpl::~BalancerChunkMergerImpl() {}

void BalancerChunkMergerImpl::waitForStop() {}

void BalancerChunkMergerImpl::onStepDown() {}

bool BalancerChunkMergerImpl::isMergeCandidate(OperationContext* opCtx,
                                               NamespaceString const& nss) const {
    try {
        auto collection = Grid::get(opCtx)->catalogClient()->getCollection(opCtx, nss);
        return collection.getBalancerShouldMergeChunks();
    } catch (const ExceptionFor<ErrorCodes::NamespaceNotFound>&) {
        // prevent error from bubbling up
        return false;
    }
}

std::vector<CollectionType> BalancerChunkMergerImpl::selectCollections(OperationContext* opCtx) {

    // TODO use a more sophisticate filtered method
    auto collections = Grid::get(opCtx)->catalogClient()->getCollections(opCtx, {});
    if (collections.empty()) {
        return {};
    }

    std::vector<CollectionType> result;
    for (auto& coll : collections) {
        if (coll.getTimeseriesFields()) {
            continue;
        }

        if (coll.getBalancerShouldMergeChunks()) {
            LOGV2_INFO(8423325,
                       "Selecting collection {namespace} for chunk merging.",
                       "namespace"_attr = coll.getNss());
            result.emplace_back(std::move(coll));
        }
    }

    return result;
}

/**
 * Schedule chunk data size estimation requests
 */
StatusWith<Progress> BalancerChunkMergerImpl::mergeChunksOnShards(OperationContext* opCtx,
                                                                  CollectionType const& coll) {
    invariant(!coll.getTimeseriesFields());
    return Progress::Done;
}

/**
 * Schedule chunk move & merge requests
 */
StatusWith<Progress> BalancerChunkMergerImpl::moveMergeOrSplitChunks(OperationContext* opCtx,
                                                                     CollectionType const& coll) {

    return Progress::Done;
}

bool BalancerChunkMergerImpl::isConverged(OperationContext* opCtx, CollectionType const& coll) {
    return true;
}

}  // namespace mongo
