/**
 * Copyright (C) 2017 MongoDB Inc.
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License, version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * As a special exception, the copyright holders give permission to link the
 * code of portions of this program with the OpenSSL library under certain
 * conditions as described in each individual source file and distribute
 * linked combinations including the program with the OpenSSL library. You
 * must comply with the GNU Affero General Public License in all respects
 * for all of the code used other than as permitted herein. If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so. If you do not
 * wish to do so, delete this exception statement from your version. If you
 * delete this exception statement from all source files in the program,
 * then also delete it in the license file.
 */

#include "mongo/platform/basic.h"

#include "mongo/scripting/mozjs/status.h"

#include "mongo/scripting/mozjs/implscope.h"
#include "mongo/scripting/mozjs/internedstring.h"
#include "mongo/scripting/mozjs/objectwrapper.h"
#include "mongo/scripting/mozjs/valuereader.h"
#include "mongo/scripting/mozjs/wrapconstrainedmethod.h"

namespace mongo {
namespace mozjs {

const char* const MongoStatusInfo::className = "MongoStatus";
const char* const MongoStatusInfo::inheritFrom = "Error";

Status MongoStatusInfo::toStatus(JSContext* cx, JS::HandleObject object) {
    return *static_cast<Status*>(JS_GetPrivate(object));
}

Status MongoStatusInfo::toStatus(JSContext* cx, JS::HandleValue value) {
    return *static_cast<Status*>(JS_GetPrivate(value.toObjectOrNull()));
}

void MongoStatusInfo::fromStatus(JSContext* cx, Status status, JS::MutableHandleValue value) {
    auto scope = getScope(cx);

    JS::RootedValue undef(cx);
    undef.setUndefined();

    JS::AutoValueArray<1> args(cx);
    ValueReader(cx, args[0]).fromStringData(status.reason());
    JS::RootedObject error(cx);
    scope->getProto<ErrorInfo>().newInstance(args, &error);

    JS::RootedObject thisv(cx);
    scope->getProto<MongoStatusInfo>().newObjectWithProto(&thisv, error);
    ObjectWrapper thisvObj(cx, thisv);
    thisvObj.defineProperty(
        InternedString::code,
        undef,
        JSPROP_ENUMERATE | JSPROP_SHARED,
        smUtils::wrapConstrainedMethod<Functions::code, false, MongoStatusInfo>);

    thisvObj.defineProperty(
        InternedString::reason,
        undef,
        JSPROP_ENUMERATE | JSPROP_SHARED,
        smUtils::wrapConstrainedMethod<Functions::reason, false, MongoStatusInfo>);

    JS_SetPrivate(thisv, scope->trackedNew<Status>(std::move(status)));

    value.setObjectOrNull(thisv);
}

void MongoStatusInfo::construct(JSContext* cx, JS::CallArgs args) {
    auto code = args.get(0).toInt32();
    auto reason = JSStringWrapper(cx, args.get(1).toString()).toString();

    JS::RootedValue out(cx);
    fromStatus(cx, Status(ErrorCodes::Error(code), std::move(reason)), &out);

    args.rval().set(out);
}

void MongoStatusInfo::finalize(JSFreeOp* fop, JSObject* obj) {
    auto status = static_cast<Status*>(JS_GetPrivate(obj));

    if (status)
        getScope(fop)->trackedDelete(status);
}

void MongoStatusInfo::Functions::code::call(JSContext* cx, JS::CallArgs args) {
    args.rval().setInt32(toStatus(cx, args.thisv()).code());
}

void MongoStatusInfo::Functions::reason::call(JSContext* cx, JS::CallArgs args) {
    ValueReader(cx, args.rval()).fromStringData(toStatus(cx, args.thisv()).reason());
}

void MongoStatusInfo::postInstall(JSContext* cx, JS::HandleObject global, JS::HandleObject proto) {
    auto scope = getScope(cx);

    JS_SetPrivate(proto, scope->trackedNew<Status>(Status::OK()));
}

}  // namespace mozjs
}  // namespace mongo
