# MonArch - Groundwork Monitor Architect
# MonarchInstrument.pm
#
############################################################################
# Release 2.5
# 4-Mar-2008
############################################################################
# Author: Carlos McEvilly
#
# Copyright 2008 GroundWork Open Source, Inc. (GroundWork)  
# All rights reserved. This program is free software; you can redistribute
# it and/or modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#

use strict;
package Instrument;

my $DEBUG = 0;

=pod

=head1 SYNOPSIS

MonarchInstrument can be used to add comments to HTML or other output
showing the context in which the HTML was generated. This is useful
for understanding how a particular page or other output was generated.

Once the show_trace_as_<language>_comment() calls are inserted into
code as shown below, the output can be turned off at any time by
editing MonarchInstrument.pm and setting the value of $DEBUG to 0.

=over 4

 use MonarchInstrument;

 my $html;
 $html = qq{<html>
    @{[&$Instrument::show_trace_as_html_comment()]}
    <body>\n<p>some content</p>\n</body>\n</html>};

=back

-or-

=over 4

 $html  = "<html>\n";
 $html .= &$Instrument::show_trace_as_html_comment();
 $html .= "<body>\n<p>some content</p>\n</body>\n</html>\n";

=back

The above examples will emit HTML like the following:

=over 4

 <html>
 <!--generated by Go::go() called from /www/cgi-bin/file.cgi line 55-->
 <body>
 <p>some content</p>
 </body>
 </html>

=back

Full list of available subroutines:

=over 4

 show_trace_as_text();
 show_trace_as_html();
 show_trace_as_html_comment();
 show_trace_as_shell_comment();
 show_trace_as_batch_comment();
 show_trace_as_javascript_comment();

=back

=cut


our $show_trace_as_text               = create_trace_reporter($DEBUG, "text");
our $show_trace_as_html               = create_trace_reporter($DEBUG, "html");
our $show_trace_as_html_comment       = create_trace_reporter($DEBUG, "html_comment");
our $show_trace_as_shell_comment      = create_trace_reporter($DEBUG, "shell");
our $show_trace_as_batch_comment      = create_trace_reporter($DEBUG, "batch");
our $show_trace_as_javascript_comment = create_trace_reporter($DEBUG, "javascript");


sub create_trace_reporter {
	my $debug = shift;
	my $lang  = shift || 'html_comment';

	my $template = '';

	if ($lang =~ /^html$/i) {
		$template = "\n<span class=\"monarch_instrument_trace\">TRACE</span>\n";
	}
	if ($lang =~ /^html_comment$/i) {
		$template = "\n<!--TRACE-->\n";
	}
	elsif ($lang =~ /^javascript$/i) {
		$template = "\n// TRACE\n"; # do not remove the newline at the end of this template
	}
	elsif ($lang =~ /^text$/i) {
		$template = "TRACE\n";
	}
	elsif ($lang =~ /^shell$/i) {
		$template = "# TRACE\n";
	}
	elsif ($lang =~ /^batch$/i) {
		$template = "rem TRACE\n";
	}

	# return a closure so we can call this by $show_trace_<language> from inside qq() blocks
	return sub {
		return '' unless ($debug);
		my ($package,$filename,$line,$subroutine,$hasargs,$wantarray,$evaltext,$is_require,$hints,$bitmask) = caller(1);
		#$filename =~ s{.*/}{};
		my $trace = "generated by $subroutine() called from $filename line $line ";
		my $output = $template;
		$output =~ s{TRACE}{$trace};
		return $output;
	};

}


1;

