/*-
 * Copyright 2003 John-Mark Gurney.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: header.c,v 1.6 2003/09/15 23:37:19 jmg Exp $
 *
 */

#include <stdlib.h>
#include <string.h>

#include <mimemisc.h>
#include <util.h>

#include <mime_priv.h>

const char *
mime_getvalue(struct mime_header *head, const char *key)
{
	return mime_getvaluec(head, key, NULL);
}

const char *
mime_getvaluec(struct mime_header *head, const char *key, int *plc)
{
	return attrib_get((struct attrib *)head, key, plc);
}

static struct mime_header *
getinitmh()
{
	return (struct mime_header *)attrib_create();
}

static char *
foldvalue(char *value, int len, const char **endptr, const char *crlfpair)
{
	char *end;
	char *ret;
	char *crlf;
	int alc;

	end = value + len;
	ret = NULL;
	crlf = NULL;	/* XXX - bogus? */
	alc = 0;

	while (value < end) {
		crlf = memmem(value, end - value, crlfpair, strlen(crlfpair));
		if (crlf == NULL)
			goto fv_error;

		ret = realloc(ret, crlf - value + alc + 1);
		memcpy(ret + alc, value, crlf - value);
		ret[crlf - value + alc] = '\0';
		alc += crlf - value;

		if (crlf[strlen(crlfpair)] == ' '
		    || crlf[strlen(crlfpair)] == '\t')
			value = crlf + strlen(crlfpair) + 1;
		else
			break;
	}

	*endptr = crlf + strlen(crlfpair);

	return ret;

fv_error:
	free(ret);
	return NULL;
}

static void
destroymh(struct mime_header *mh)
{
	attrib_free((struct attrib *)mh);
}

struct mime_header *
mime_parseheader(const char *header, size_t len, const char **last,
    const char *crlfpair)
{
	const char *end;
	char *colon;
	char *crlf;
	char *name;
	char *value;
	struct attrib *ret;

	ret = (struct attrib *)getinitmh();
	end = header + len;

	while (header < end) {
		if (memcmp(header, crlfpair, strlen(crlfpair)) == 0) {
			header += strlen(crlfpair);
			break;
		}

		/*
		 * assume that we allways start at the begining of a
		 * header line
		 */
		colon = memchr(header, ':', end - header);
		crlf = memmem(header, end - header, crlfpair, strlen(crlfpair));

		if (colon == NULL || crlf == NULL || colon > crlf)
			goto mph_error;

		/* get the key */
		name = malloc(colon - header + 1);
		memcpy(name, header, colon - header);
		name[colon - header] = '\0';

		value = foldvalue(colon + 1, end - colon - 1, &header,
		    crlfpair);

		attrib_addnodup(ret, name, value);
	}

	if (last != NULL)
		*last = header;
		
	return (struct mime_header *)ret;

mph_error:
	destroymh((struct mime_header *)ret);
	return NULL;
}

void
mime_headerdelete(struct mime_header *mh)
{
	destroymh(mh);
}
