/*
 * Copyright (c) 2004 Peter O'Gorman <peter@pogma.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a 
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included 
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 */



#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <paths.h>
#include <IOKit/IOKitLib.h>
#include <CoreFoundation/CoreFoundation.h>
#include <IOKit/IOBSD.h>
#include <IOKit/storage/ata/ATASMARTLib.h>
#include <IOKit/storage/IOBlockStorageDevice.h>
#include <IOKit/IOCFPlugIn.h>

/* failure count, how many thresholds are we below */

static int failcount = 0;

/* Flags/command line options */
enum {
  kMaxwellQuiet = 0x0001,
  kMaxwellReport = 0x0002,
  kMaxwellSimpleTest = 0x0004,
  kMaxwellExtendedTest = 0x0008,
  kMaxwellAwaitCompletion = 0x0010,
  kMaxwellEnableSMART = 0x0020,
  kMaxwellDisableSMART = 0x0040
};

typedef struct ATASMARTAttributeStruct {
  UInt8           attributeID;
  UInt8           statusFlags[2];
  UInt8           attributeValue;
  UInt8           reserved1;
  UInt8           reserved2[6]; /* This seems to be the */
								/* actual data.			*/
  UInt8           reserved3;
} ATASMARTAttributeStruct;

typedef struct ATASMARTThresholdsStruct {
  UInt8           attributeID;
  UInt8           thresholdVal;;
  UInt8           reserved[10];
} ATASMARTThresholdsStruct;

typedef struct ATASMARTDataStruct {
  UInt16          revNumber;
  ATASMARTAttributeStruct attributes[30];
  UInt8           offLineDataCollectionStatus;
  UInt8           selfTestExecutionStatus;
  UInt8           secondsToCompleteOffLineActivity[2];
  UInt8           vendorSpecific2;
  UInt8           offLineDataCollectionCapability;
  UInt8           SMARTCapability[2];
  UInt8           errorLoggingCapability;
  UInt8           vendorSpecific3;
  UInt8           shortTestPollingInterval;	/* expressed in minutes */
  UInt8           extendedTestPollingInterval;	/* expressed in minutes */
  UInt8           reserved[12];
  UInt8           vendorSpecific4[125];
  UInt8           checksum;
} ATASMARTDataStruct;

typedef struct ATASMARTThreshStruct	// good naming dude ... NOT!
{
  UInt16          revNumber;
  ATASMARTThresholdsStruct thresholds[30];
  UInt8           reserved[18];
  UInt8           vendorSpecific[131];
  UInt8           checksum;
} ATASMARTThreshStruct;


/* print_usage()												*/
/* needs to print a little more help!							*/
void
print_usage()
{
  fprintf(stderr, "Usage: %s -hsewnbqrd:\n", getprogname());
}

/* enable_smart(IOATASMARTInterface **)							*/
/* Turn on smart on the passed device							*/
IOReturn
enable_smart(IOATASMARTInterface ** SMARTInterface)
{
  IOReturn        ioresult = 0;
  ioresult =
	(*SMARTInterface)->SMARTEnableDisableOperations(SMARTInterface, 1);
  if (!ioresult)
	ioresult =
	  (*SMARTInterface)->SMARTEnableDisableAutosave(SMARTInterface,
													1);
  return ioresult;
}

/* disable_smart(IOATASMARTInterface **)							*/
/* Turn off smart on the passed device, never worked for me...		*/
IOReturn
disable_smart(IOATASMARTInterface ** SMARTInterface)
{
  IOReturn        ioresult = 0;
  ioresult =
	(*SMARTInterface)->SMARTEnableDisableOperations(SMARTInterface, 0);
  if (!ioresult)
	ioresult =
	  (*SMARTInterface)->SMARTEnableDisableAutosave(SMARTInterface,
													0);
  return ioresult;
}

/* MyGetDeviceFilePath												*/
/* Copied verbatim from apple sample code							*/
kern_return_t
MyGetDeviceFilePath(io_registry_entry_t nextMedia,
					char *deviceFilePath, CFIndex maxPathSize)
{
  kern_return_t   kernResult = KERN_FAILURE;
  CFTypeRef       deviceFilePathAsCFString;
  *deviceFilePath = '\0';
  deviceFilePathAsCFString =
	IORegistryEntryCreateCFProperty(nextMedia, CFSTR(kIOBSDNameKey),
									kCFAllocatorDefault, 0);
  if (deviceFilePathAsCFString) {
	size_t          devPathLength = strlen(_PATH_DEV);
	strcpy(deviceFilePath, _PATH_DEV);

	if (CFStringGetCString(deviceFilePathAsCFString,
						   deviceFilePath + devPathLength,
						   maxPathSize - devPathLength,
						   kCFStringEncodingASCII)) {
	  kernResult = KERN_SUCCESS;
	}
	CFRelease(deviceFilePathAsCFString);
  }
  return kernResult;
}

/* await_test_completion												*/
/* Hang around, sleeping lots until the self test is complete.			*/
IOReturn
await_test_completion(IOATASMARTInterface ** SMARTInterface, int flags)
{
  IOReturn        ioresult = 0;
  ATASMARTDataStruct smartData;
  ioresult =
	(*SMARTInterface)->SMARTReadData(SMARTInterface,
									 (ATASMARTData *) & smartData);
  if (ioresult)
	return ioresult;
  fprintf(stdout,
			  "Awaiting self test completion, this may take a while, 30 minutes or more.\n");
  while (0xF0 == (smartData.selfTestExecutionStatus & 0xF0)) {
	UInt8           progress = 0;

	if (!(flags & kMaxwellQuiet)) {
	  progress = 10 - (smartData.selfTestExecutionStatus & 0x0F);
	  while (progress) {
		fprintf(stderr, ".");
		progress--;
	  }
	  if (!(flags & kMaxwellQuiet))
		fprintf(stderr, "\n");
	}
	sleep(60 * smartData.shortTestPollingInterval);	// sleep time
	ioresult =
	  (*SMARTInterface)->SMARTReadData(SMARTInterface,
									   (ATASMARTData *) & smartData);
	if (ioresult)
	  return ioresult;	// FIXME
  }
  return 0;
}


/* testname */
/* returns the names of various tests */
const char     *
testname(UInt8 value)
{
  char           *retVal = "Unknown";
  switch (value) {
  case 0:
	retVal = "Invaid 0";
	break;
  case 1:
	retVal = "Raw Read Error Rate";
	break;
  case 2:
	retVal = "Throughput Performance";
	break;
  case 3:
	retVal = "Spin Up Time";
	break;
  case 4:
	retVal = "Start/Stop Count";
	break;
  case 5:
	retVal = "Reallocated Sector Count";
	break;
  case 7:
	retVal = "Seek Error Rate";
	break;
  case 8:
	retVal = "Seek Time Performance";
	break;
  case 9:
	retVal = "Power-On Hours Count **";
	break;
  case 10:
	retVal = "Spin Retry Count";
	break;
  case 11:
	retVal = "Calibration Retry Count";
	break;
  case 12:
	retVal = "Device Power Cycle Count";
	break;
  case 191:
	retVal = "Gsense Error Rate";
	break;
  case 192:
	retVal = "Power Off Retract Count";
	break;
  case 193:
	retVal = "Load/Unload Cycle Count";
	break;
  case 194:
	retVal = "Device Temperature";
	break;
  case 195:
	retVal = "On The Fly Error Rate";
	break;
  case 196:
	retVal = "Reallocation Event Count";
	break;
  case 197:
	retVal = "Current Pending Sector Count";
	break;
  case 198:
	retVal = "Off-Line Scan Uncorrectable Sector Count";
	break;
  case 199:
	retVal = "Ultra DMA CRC Error Count";
	break;
  case 200:
	retVal = "Write Preamp Errors";
	break;
  case 201:
	retVal = "Off Track Errors";
	break;
  case 202:
	retVal = "DAM Error Rate";
	break;
  case 203:
	retVal = "ECC Errors";
	break;
  case 204:
	retVal = "Raw Read Error Rate";
	break;
  case 205:
	retVal = "Thermal Asperity Rate";
	break;
  case 206:
	retVal = "Unknown 206";
	break;
  case 207:
	retVal = "Spin High Current";
	break;
  case 208:
	retVal = "Spin Buzz";
	break;
  case 209:
	retVal = "Off Line Seek Performance";
	break;
  }
  return retVal;
}

/* doRealStuff */
/* quite surprisingly, this is where the real stuff happens */
void
doRealStuff(IOATASMARTInterface ** SMARTInterface, int flags)
{
  UInt16          id[256];
  ATASMARTDataStruct smartData;
  ATASMARTThreshStruct threshData;
  char            firmwareRev[9];
  char            serialNumber[21];
  char            modelNumber[41];
  IOReturn        ioresult = 0;
  Boolean         failing = 0;
  int             thirtyCount;
  int             temperature = -1;
  // get the device identity data
  ioresult =
	(*SMARTInterface)->GetATAIdentifyData(SMARTInterface, (void *) &id,
										  512, NULL);
  if (ioresult) {
	if (!(flags & kMaxwellQuiet))
	  fprintf(stderr, "GetATAIdentifyData failed\n");
	exit(1);		// should never be here
  }
  // we are only interested in ATA devices, not ATAPI
  if (0 != (id[0] & 0xF000)) {
	return;
  }

  memcpy(serialNumber, &id[10], 20);
  serialNumber[20] = 0;
  memcpy(modelNumber, &id[27], 40);
  modelNumber[40] = 0;
  memcpy(firmwareRev, &id[23], 8);
  firmwareRev[8] = 0;
  if (flags & kMaxwellReport)
	fprintf(stdout, "Serial: %s\nModel: %s\nFirmware: %s\n",
			serialNumber, modelNumber, firmwareRev);
  if (1 != (id[82] & 0x0001)) {
	if (flags & kMaxwellReport)
	  fprintf(stdout, "Device is not S.M.A.R.T.\n");
	return;
  } else {
	if (flags & kMaxwellReport)
	  fprintf(stdout, "Device supports S.M.A.R.T. operations\n");
	if (2 == (id[84] & 0x0002)) {
	  if (flags & kMaxwellReport)
		fprintf(stdout, "SMART self-test supported\n");
	}
	if (1 == (id[84] & 0x0001)) {
	  if (flags & kMaxwellReport)
		fprintf(stdout, "SMART error logging supported\n");
	}
  }
  if (1 != (id[85] & 0x0001)) {
	if (flags & kMaxwellReport)
	  fprintf(stdout, "S.M.A.R.T. operations are not enabled\n");
	return;
  } else {
	if (flags & kMaxwellReport)
	  fprintf(stdout, "S.M.A.R.T. operations are enabled\n");
  }
  if (2 == (id[87] & 0x0002)) {
	if (flags & kMaxwellReport)
	  fprintf(stdout, "SMART self-test enabled\n");
  }
  if (1 == (id[87] & 0x0001)) {
	if (flags & kMaxwellReport)
	  fprintf(stdout, "SMART error logging enabled\n");
  }

  if (flags & (kMaxwellSimpleTest | kMaxwellExtendedTest)) {
	ioresult =
	  (*SMARTInterface)->SMARTExecuteOffLineImmediate(SMARTInterface,
													  (flags &
													   kMaxwellExtendedTest)
													  ? 1 : 0);
	if (ioresult)
	  return;		// FIXME
  }
  // now we wait for completion.

  if (flags & kMaxwellAwaitCompletion) {
	ioresult = await_test_completion(SMARTInterface, flags);
	if (ioresult)
	  return;		// FIXME
  }
  ioresult =
	(*SMARTInterface)->SMARTReadData(SMARTInterface,
									 (ATASMARTData *) & smartData);
  if (ioresult)
	return;			// FIXME 
  // Whee, now we have a post-test smartData struct. Get the
  // thresholds...
  ioresult =
	(*SMARTInterface)->SMARTReadDataThresholds(SMARTInterface,
											   (ATASMARTDataThresholds
												*) & threshData);
  if (ioresult)
	return;			// FIXME
  // 

  if (flags & kMaxwellReport) {
	
	if (0xF0 == (smartData.selfTestExecutionStatus & 0xF0)) {
	  fprintf(stdout, "Test in progress %1i0%% remaining\n",smartData.selfTestExecutionStatus & 0x0F);
	}
	fprintf(stdout, "Off line collection status is %x\n",
			smartData.offLineDataCollectionStatus);
	fprintf(stdout, "Time to complete Off line Data collection: %i hours, %i minutes\n",(*(UInt16*)&smartData.secondsToCompleteOffLineActivity) / 3600, ((*(UInt16*)&smartData.secondsToCompleteOffLineActivity) % 3600)/60);
	fprintf(stdout, "Status is %s\n", failing ? "BAD" : "GOOD");
	fprintf(stdout, "      %-40s%6s %6s %6s %12s\n", "TEST", "THRSH",
			"VALUE", "STATUS","RAW");

	fprintf(stdout,
			"------------------------------------------------------------------------------\n");
  }	
  for (thirtyCount = 0; thirtyCount < 30; thirtyCount++) {
    ATASMARTAttributeStruct *attr =
		&smartData.attributes[thirtyCount];
	ATASMARTThresholdsStruct *thresh =
		&threshData.thresholds[thirtyCount];
	long long rawdata = 0;
	int x;
	int             status = attr->statusFlags[0];
	status <<= 8;
	status |= attr->statusFlags[1];
	if (!attr->attributeID)
		continue;
	if (attr->attributeID != thresh->attributeID)
		continue;
	if (194 == attr->attributeID)
		temperature = attr->attributeValue;
	x = 6;
	while (x--) {
		rawdata += attr->reserved2[x];
		rawdata <<= 8;
	}
	rawdata >>=8;
	if (attr->attributeValue <= thresh->thresholdVal) {
	  failcount++;
	}
	if (flags & kMaxwellReport) {
	  fprintf(stdout, "(%3i) %-40s", attr->attributeID,
			  testname(attr->attributeID));
	  fprintf(stdout, "%6i ", thresh->thresholdVal);
	  fprintf(stdout, "%6i ", attr->attributeValue);
	  fprintf(stdout, "0x%4.4x ", status);
	  fprintf(stdout, "%12lld\n",rawdata);
	}
  }
  if (flags & kMaxwellReport) {
    fprintf(stdout,
			"------------------------------------------------------------------------------\n");

	if (temperature > -1) {
	  char           *tempstr;
	  if (temperature == 0)
		tempstr = "less than -20";
	  else if (temperature == 255)
		tempstr = "greater than 107";
	  else
		tempstr = NULL;
	  if (tempstr)
		fprintf(stdout,
				"Device temperature is %s degrees centigrade\n",
				tempstr);
	  else
		fprintf(stdout,
				"Device temperature is %i degrees centigrade\n",
				(temperature / 2) - 20);
	  fprintf(stdout,
			  "------------------------------------------------------------------------------\n");
	  fprintf(stdout,
			  "The names listed above may not actually be correct for each test value. The\n");
	  fprintf(stdout,
			  "real values printed may make no sense whatever and the temperature may be\n");
	  fprintf(stdout,
			  "some crazy value. Different device manufacturers do things differently :(.\n");
	  fprintf(stdout,
			  "------------------------------------------------------------------------------\n");
	}
  }
  ioresult =
	(*SMARTInterface)->SMARTReturnStatus(SMARTInterface, &failing);
  if (ioresult)
	exit(1);		// FIXME
  if (failing) {
	if (!(flags & kMaxwellQuiet))
	  fprintf(stderr, "Device: %s Reported FAIL status\n",
			  modelNumber);
	exit(64);
  } else {
	if (!(flags & kMaxwellQuiet))
	  fprintf(stderr, "Device: %s Reported PASS status\n",
			  modelNumber);
  }
}

int
main(int argc, const char **argv)
{
  int             ch;
  kern_return_t   result = KERN_SUCCESS;
  mach_port_t     masterport = NULL;
  char           *devname = NULL;
  CFMutableDictionaryRef matchingDictionary = NULL;
  io_iterator_t   ata_iterator;
  io_registry_entry_t ata_drive;
  int             flags = 0;
  int             foundDev = 0;
  setprogname(argv[0]);
  // Get command line options;
  while ((ch = getopt(argc, (char *const *) argv, "hsewnbqrd:")) != -1) {
	switch (ch) {
	case 'h':
	  print_usage();	// help
	  exit(0);
	case 's':
	  flags |= kMaxwellSimpleTest;	// simple test
	  break;
	case 'e':
	  flags |= kMaxwellExtendedTest;	// extended test
	  break;
	case 'w':
	  flags |= kMaxwellAwaitCompletion;	// wait for test
	  // completion
	  break;
	case 'n':
	  flags |= kMaxwellDisableSMART;	// Disable SMART operations on this device
	  break;
	case 'b':
	  flags |= kMaxwellEnableSMART;	// Enable SMART operations on this device
	  break;
	case 'q':
	  flags |= kMaxwellQuiet;	// quiet
	  break;
	case 'r':
	  flags |= kMaxwellReport;	// print a report
	  break;
	case 'd':
	  devname = strdup(optarg);	// specify the device to use.
	  break;
	case '?':
	default:
	  print_usage();	// unknown option
	  exit(1);
	}
  }
  if ((flags & kMaxwellQuiet) && (flags & kMaxwellReport))
	flags -= kMaxwellReport;	// -q disables *all* messages.

  // get a mach_port 
  result = IOMasterPort(MACH_PORT_NULL, &masterport);
  if (KERN_SUCCESS != result) {
	if (!(flags & kMaxwellQuiet))
	  fprintf(stderr, "IOMasterPort failed...\n");
	exit(1);
  }
  // match all Disks
  matchingDictionary = IOServiceMatching(kIOBlockStorageDeviceClass);
  if (NULL == matchingDictionary) {
	if (!(flags & kMaxwellQuiet))
	  fprintf(stderr, "IOServiceMatching failed...\n");
	exit(1);
  }
  result =
	IOServiceGetMatchingServices(masterport, matchingDictionary,
								 &ata_iterator);
  if (KERN_SUCCESS != result || 0 == ata_iterator) {
	if (!(flags & kMaxwellQuiet))
	  fprintf(stderr, "IOServiceGetMatchingServices failed...\n");
	exit(1);
  }
  while ((ata_drive = IOIteratorNext(ata_iterator))) {
	IOReturn        ioresult = 0;
	SInt32          theScore = 0;
	IOCFPlugInInterface **SMARTPlugin = 0;
	IOATASMARTInterface **SMARTInterface = 0;
	io_registry_entry_t child_ata_drive;
	io_string_t     pathName;
	char            deviceFilePath[128] = { 0 };

	ioresult =
	  IOCreatePlugInInterfaceForService(ata_drive,
										kIOATASMARTUserClientTypeID,
										kIOCFPlugInInterfaceID,
										&SMARTPlugin, &theScore);
	if (ioresult) {
	  IOObjectRelease(ata_drive);
	  ata_drive = 0;
	  continue;
	}
	// we got the plugin, but we need the interface, seems a little
	// strange to me, but hey, this is mostly above my head.
	SMARTInterface = 0;
	(*SMARTPlugin)->QueryInterface(SMARTPlugin,
								   CFUUIDGetUUIDBytes
								   (kIOATASMARTInterfaceID),
								   (void **) &SMARTInterface);
	if (!SMARTInterface) {
	  IOObjectRelease(ata_drive);
	  ata_drive = 0;
	  (*SMARTPlugin)->Release(SMARTPlugin);
	  continue;
	}
	// The child of the child should be an IOMedia object whcih will
	// have the BSD name :/
	IORegistryEntryGetPath(ata_drive, kIOServicePlane, pathName);
	ioresult =
	  IORegistryEntryGetChildEntry(ata_drive, kIOServicePlane,
								   &child_ata_drive);
	ioresult =
	  IORegistryEntryGetChildEntry(child_ata_drive, kIOServicePlane,
								   &child_ata_drive);
	IORegistryEntryGetPath(child_ata_drive, kIOServicePlane, pathName);
	MyGetDeviceFilePath(child_ata_drive, deviceFilePath, 128);
	if (flags & kMaxwellReport)
	  fprintf(stdout, "BSD Path:%s\n", deviceFilePath);
	if (devname
	    && strncmp(devname, deviceFilePath, strlen(deviceFilePath))) {
	  continue;
	}
	if (flags & kMaxwellDisableSMART) {
	  ioresult = disable_smart(SMARTInterface);
	  if (ioresult) {
		if (!(flags & kMaxwellQuiet))
		  fprintf(stderr, "Disable SMART Failed\n");
		exit(1);
	  }
	  exit(0);
	}
	if (flags & kMaxwellEnableSMART) {
	  ioresult = enable_smart(SMARTInterface);
	  if (ioresult) {
		if (!(flags & kMaxwellQuiet))
		  fprintf(stderr, "Enable SMART Failed\n");
		exit(1);
	  }
	  exit(0);
	}
	// Now we have the interface we can finally do real stuff!!!
	doRealStuff(SMARTInterface, flags);
	foundDev++;
	IOObjectRelease(ata_drive);
	ata_drive = 0;
  }
  IOObjectRelease(ata_iterator);
  if (!foundDev && devname) {
	if (!(flags & kMaxwellQuiet))
	  fprintf(stderr, "Did not find an ata drive at %s\n", devname);
	exit(1);
  }
  if (!foundDev) {
	if (!(flags & kMaxwellQuiet))
	  fprintf(stderr,
			  "No devices were found" \
			  " ... do you have another application open keeping a" \
			  "reference to the ATASMART plugin open?\n");
	exit(1);
  }
  if (ata_drive)
	IOObjectRelease(ata_drive);
  if (failcount)
  {
	if (!(flags & kMaxwellQuiet)) {
	  fprintf(stderr,"Some tests failed. One or more devices may be failing.\n");
	  exit (64 + failcount);
	}
  }
  return 0;
}
