// license:BSD-3-Clause
// copyright-holders:Vas Crabb
/*************************************************************************

    Laser Battle / Lazarian - Cat and Mouse

*************************************************************************/
#ifndef MAME_INCLUDES_LASERBAT_H
#define MAME_INCLUDES_LASERBAT_H

#pragma once

#include "audio/zaccaria.h"

#include "cpu/s2650/s2650.h"

#include "machine/6821pia.h"
#include "machine/pla.h"
#include "machine/s2636.h"

#include "sound/ay8910.h"
#include "sound/sn76477.h"
#include "sound/tms3615.h"

#include "emupal.h"
#include "screen.h"


class laserbat_state_base : public driver_device
{
public:
	laserbat_state_base(const machine_config &mconfig, device_type type, const char *tag)
		: driver_device(mconfig, type, tag)
		, m_mux_ports(*this, {"ROW0", "ROW1", "SW1", "SW2"})
		, m_row1(*this, "ROW1")
		, m_row2(*this, "ROW2")
		, m_maincpu(*this, "maincpu")
		, m_screen(*this, "screen")
		, m_palette(*this, "palette")
		, m_gfxmix(*this, "gfxmix")
		, m_pvi(*this, "pvi%u", 1U)
		, m_gfxdecode(*this, "gfxdecode")
		, m_scanline_timer(nullptr)
		, m_gfx1(nullptr)
		, m_gfx2(nullptr)
		, m_input_mux(0)
		, m_mpx_p_1_2(false)
		, m_mpx_bkeff(false)
		, m_nave(false)
		, m_clr_lum(0)
		, m_shp(0)
		, m_wcoh(0)
		, m_wcov(0)
		, m_abeff1(false)
		, m_abeff2(false)
		, m_mpx_eff2_sh(false)
		, m_coleff(0)
		, m_neg1(false)
		, m_neg2(false)
		, m_rhsc(0)
		, m_whsc(0)
		, m_csound1(0)
		, m_csound2(0)
	{
	}

	void init_laserbat();

	void laserbat_base(machine_config &config);
	void laserbat_io_map(address_map &map);
	void laserbat_map(address_map &map);

protected:
	enum { TIMER_SCANLINE };

	// control ports
	void ct_io_w(uint8_t data);
	uint8_t rrowx_r();

	INTERRUPT_GEN_MEMBER(laserbat_interrupt);

	// video memory and control ports
	void videoram_w(offs_t offset, uint8_t data);
	void wcoh_w(uint8_t data);
	void wcov_w(uint8_t data);
	void cnt_eff_w(uint8_t data);
	void cnt_nav_w(uint8_t data);

	// sound control ports
	virtual uint8_t rhsc_r();
	virtual void whsc_w(uint8_t data);
	virtual void csound1_w(uint8_t data);
	virtual void csound2_w(uint8_t data);

	// running the video
	virtual void video_start() override;
	uint32_t screen_update_laserbat(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect);

	virtual void device_timer(emu_timer &timer, device_timer_id id, int param, void *ptr) override;

	// video functions
	TIMER_CALLBACK_MEMBER(video_line);

	// input lines
	required_ioport_array<4> m_mux_ports;
	required_ioport m_row1;
	required_ioport m_row2;

	// main CPU device
	required_device<s2650_device> m_maincpu;

	// video devices
	required_device<screen_device>         m_screen;
	required_device<palette_device>        m_palette;
	required_device<pla_device>            m_gfxmix;
	required_device_array<s2636_device, 3> m_pvi;
	required_device<gfxdecode_device>      m_gfxdecode;

	// stuff for rendering video
	emu_timer       *m_scanline_timer;
	bitmap_ind16    m_bitmap;
	uint8_t const   *m_gfx1;
	uint8_t const   *m_gfx2;

	// control lines
	unsigned        m_input_mux;
	bool            m_mpx_p_1_2;

	// RAM used by TTL video hardware, writable by CPU
	uint8_t         m_bg_ram[0x400];    // background tilemap
	uint8_t         m_eff_ram[0x400];   // per-scanline effects (A8 not wired meaning only half is usable)
	bool            m_mpx_bkeff;        // select between writing background and effects memory

	// signals affecting the TTL-generated 32x32 sprite
	bool            m_nave;             // 1-bit enable
	unsigned        m_clr_lum;          // 3-bit colour/luminance
	unsigned        m_shp;              // 3-bit shape
	unsigned        m_wcoh;             // 8-bit offset horizontal
	unsigned        m_wcov;             // 8-bit offset vertical

	// video effects signals
	bool            m_abeff1;           // 1-bit effect enable
	bool            m_abeff2;           // 1-bit effect enable
	bool            m_mpx_eff2_sh;      // 1-bit effect selection
	unsigned        m_coleff;           // 2-bit colour effect
	bool            m_neg1;             // 1-bit area selection
	bool            m_neg2;             // 1-bit area selection

	// sound board I/O signals
	unsigned        m_rhsc;             // 8-bit input from J7
	unsigned        m_whsc;             // 8-bit output to J7
	unsigned        m_csound1;          // bits 1-8 on J3
	unsigned        m_csound2;          // bits 9-16 on J3
};


class laserbat_state : public laserbat_state_base
{
public:
	laserbat_state(const machine_config &mconfig, device_type type, const char *tag)
		: laserbat_state_base(mconfig, type, tag)
		, m_csg(*this, "csg")
		, m_synth_low(*this, "synth_low")
		, m_synth_high(*this, "synth_high")
		, m_keys(0)
	{
	}

	void laserbat(machine_config &config);

protected:
	// initialisation/startup
	virtual void machine_start() override;

	// video initialisation
	void laserbat_palette(palette_device &palette) const;

	// sound control ports
	virtual void csound2_w(uint8_t data) override;

	// sound board devices
	required_device<sn76477_device> m_csg;
	required_device<tms3615_device> m_synth_low;
	required_device<tms3615_device> m_synth_high;

	// register state
	unsigned    m_keys;     // low octave keys 1-13 and high octave keys 2-12 (24 bits)
};


class catnmous_state : public laserbat_state_base
{
public:
	catnmous_state(const machine_config &mconfig, device_type type, const char *tag)
		: laserbat_state_base(mconfig, type, tag)
		, m_audiopcb(*this, "audiopcb")
	{
	}

	void catnmous(machine_config &config);

protected:

	// video initialisation
	void catnmous_palette(palette_device &palette) const;

	// sound control ports
	virtual void csound1_w(uint8_t data) override;
	virtual void csound2_w(uint8_t data) override;

	required_device<zac1b11107_audio_device>    m_audiopcb;
};

#endif // MAME_INCLUDES_LASERBAT_H
