// makeicns
// Converts images to Apple's icns format.
// Written by nicolasweber@gmx.de, released under MIT license.
// Belongs to docerator, http://code.google.com/p/docerator/

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>

#import <Cocoa/Cocoa.h>

#include "IconFamily.h"

#if MAC_OS_X_VERSION_MAX_ALLOWED <= MAC_OS_X_VERSION_10_4
// This is defined in 10.5 and beyond in IconStorage.h
enum {
	kIconServices512PixelDataARGB = 'ic09' /* non-premultiplied 512x512 ARGB bitmap*/
};
#endif

BOOL nastyGlobalVar_Center=FALSE;

//This will now be set in Makefile:
//#define VERSION "1.1 (20090413)"

void usage() {
	fprintf(stderr,
	        "makeicns v%s", VERSION);
	if(strlen(CHANGESET)) fprintf(stderr, " (%s)",CHANGESET);
	fprintf(stderr, "\n\n");
	fprintf(stderr,
	        "Usage: makeicns [k1=v1] [k2=v2] ...\n\n");
	fprintf(stderr,
	        "Keys and values include:\n");
	fprintf(stderr,
	        "    512: Name of input image for 512x512 variant of icon\n");
	fprintf(stderr,
	        "    256: Name of input image for 256x256 variant of icon\n");
	fprintf(stderr,
	        "    128: Name of input image for 128x128 variant of icon\n");
	fprintf(stderr,
	        "     32: Name of input image for 32x32 variant of icon\n");
	fprintf(stderr,
	        "     16: Name of input image for 16x16 variant of icon\n");
	fprintf(stderr,
	        "     in: Name of input image for all variants not having an explicit name\n");
	fprintf(stderr,
	        "    out: Name of output file, defaults to first nonempty input name,\n"
	        "         but with icns extension\n\n");
	fprintf(stderr,
	        "  align: [center, left, right, top, bottom] {First letter suffices!}\n\n");

	fprintf(stderr,
	        "Examples:\n\n"
	        "  makeicns -512 image.png -32 image.png\n"
	        "      Creates image.icns with only a 512x512 and a 32x32 variant.\n\n"
	        "  makeicns -in myfile.jpg -32 otherfile.png -out outfile.icns\n"
	        "      Creates outfile.icns with sizes 512, 256, 128, and 16 containing data\n"
	        "      from myfile.jpg and with size 32 containing data from otherfile.png.\n");
	exit(1);
}


NSBitmapImageRep* getBitmapImageRepOfSize(NSImage* img, int size, char align) {
// align can be 'c','l','r','t', or 'b'

	// Don't resample if it's not necessary
#if 0
	// IconFamily does not work correctly with
	// NSAlphaNonpremultipliedBitmapFormat images, so this has to stay disabled
	// until IconFamily is fixed (if ever).
	NSEnumerator* e = [[img representations] objectEnumerator];
	NSImageRep* ir;
	while ((ir = [e nextObject])) {
		if (![ir isKindOfClass:[NSBitmapImageRep class]]) continue;

		NSBitmapImageRep* br = (NSBitmapImageRep*)ir;
		//NSLog(@"%@", br);
		if ([br pixelsWide] == size && [br pixelsHigh] == size
		    && ([[br colorSpaceName] isEqualToString:NSDeviceRGBColorSpace]
		        || [[br colorSpaceName] isEqualToString:NSCalibratedRGBColorSpace])
		    && ([br bitsPerPixel] == 24 || [br bitsPerPixel] == 32)
		    )
			return br;
	}
#endif

	float x,y;

	NSSize imageSize = [img size];
	NSSize targetSize = NSMakeSize (size, size);
	float width  = imageSize.width;
	float height = imageSize.height;

	float targetWidth  = targetSize.width;
	float targetHeight = targetSize.height;

	// scaleFactor will be the fraction that we'll
	// use to adjust the size. For example, if we shrink
	// an image by half, scaleFactor will be 0.5. the
	// scaledWidth and scaledHeight will be the original,
	// multiplied by the scaleFactor.
	//
	// IMPORTANT: the "targetHeight" is the size of the space
	// we're drawing into. The "scaledHeight" is the height that
	// the image actually is drawn at, once we take into
	// account the ideal of maintaining proportions

	float scaleFactor  = 0.0;
	int scaledWidth  = targetWidth;
	int scaledHeight = targetHeight;

	NSPoint thumbnailPoint = NSZeroPoint;

	// since not all images are square, we want to scale
	// proportionately. To do this, we find the longest
	// edge and use that as a guide.

	if ( NSEqualSizes( imageSize, targetSize ) == NO )
	{
		// use the longeset edge as a guide. if the
		// image is wider than tall, we'll figure out
		// the scale factor by dividing it by the
		// intended width. Otherwise, we'll use the
		// height.

		float widthFactor  = targetWidth / width;
		float heightFactor = targetHeight / height;

		if ( widthFactor < heightFactor )
			scaleFactor = widthFactor;
		else
			scaleFactor = heightFactor;

		// ex: 500 * 0.5 = 250 (newWidth)

		scaledWidth  = floor((width  * scaleFactor)+0.5);
		scaledHeight = floor((height * scaleFactor)+0.5);

//     NSLog(@"size %d", size);
//     NSLog(@"width %f", width);
//     NSLog(@"height %f", height);
//     NSLog(@"targetWidth %f", targetWidth);
//     NSLog(@"targetHeight %f", targetHeight);
//     NSLog(@"widthFactor %f", widthFactor);
//     NSLog(@"heightFactor %f", heightFactor);
//     NSLog(@"scaleFactor %f", scaleFactor);
//     NSLog(@"scaledWidth %d", scaledWidth);
//     NSLog(@"scaledHeight %d", scaledHeight);

	}


	NSLog(@"Resampling for %d wide x %d high", scaledWidth, scaledHeight);
	NSBitmapImageRep* r = [[NSBitmapImageRep alloc]
	                       initWithBitmapDataPlanes:NULL
	                       pixelsWide:size
	                       pixelsHigh:size
	                       bitsPerSample:8
	                       samplesPerPixel:4
	                       hasAlpha:YES
	                       isPlanar:NO
	                       colorSpaceName:NSDeviceRGBColorSpace
	                       bitmapFormat:0
	                       bytesPerRow:0
	                       bitsPerPixel:0];

	[NSGraphicsContext saveGraphicsState];
	NSGraphicsContext* context = [NSGraphicsContext
	                              graphicsContextWithBitmapImageRep:r];
	[context setShouldAntialias:YES];
	[context setImageInterpolation:NSImageInterpolationHigh];
	[NSGraphicsContext setCurrentContext:context];

	x=y=0.0;

	if(scaledWidth<scaledHeight) {
		switch (align) {
		case 'l':
			// nothing to be done
			break;
		case 'r':
			x=scaledHeight-scaledWidth;
			break;
		case 'c':
		default:
			x=(scaledHeight-scaledWidth)/2;
			break;
		}
	}
	else {
		switch (align) {
		case 't':
			y=scaledWidth-scaledHeight;
			break;
		case 'b':
			// nothing to be done
			break;
		case 'c':
		default:
			y=(scaledWidth-scaledHeight)/2;
			break;
		}
	}


	[img drawInRect:NSMakeRect(x, y, scaledWidth, scaledHeight)
	 fromRect:NSZeroRect
	 operation:NSCompositeCopy
	 fraction:1.0];

	[NSGraphicsContext restoreGraphicsState];

	return r;
}


int main(int argc, char* argv[]) {
	int i;
	BOOL fileDoesNotExist=FALSE;
	NSString* alignOption;
	char align;

	NSAutoreleasePool * pool = [[NSAutoreleasePool alloc] init];
	NSApplicationLoad();

	struct {
		NSString* paramName;
		OSType type, mask;
		int size;
		NSString* inputName;
	} inputs[] = {
		{ @"512", kIconServices512PixelDataARGB, 0, 512, nil },
		{ @"256", kIconServices256PixelDataARGB, 0, 256, nil },
		{ @"128", kThumbnail32BitData, kThumbnail8BitMask, 128, nil },
		{ @"32", kLarge32BitData, kLarge8BitMask, 32, nil },
		{ @"16", kSmall32BitData, kSmall8BitMask, 16, nil },
	};
	const int N = sizeof(inputs)/sizeof(inputs[0]);

	// Process arguments -- Thanks Greg!
	//http://unixjunkie.blogspot.com/2006/07/command-line-processing-in-cocoa.html
	NSUserDefaults* args = [NSUserDefaults standardUserDefaults];

	BOOL foundInputParam = NO;
	NSString* outputName = [args stringForKey:@"out"];
	NSString* defaultIn = [args stringForKey:@"in"];
	for (i = 0; i < N; ++i) {
		inputs[i].inputName = [args stringForKey:inputs[i].paramName];
		if (inputs[i].inputName == nil)
			inputs[i].inputName = defaultIn;
		foundInputParam = foundInputParam || inputs[i].inputName != nil;

		// Create default output name if necessary
		if (outputName == nil && inputs[i].inputName != nil)
			outputName = [[inputs[i].inputName stringByDeletingPathExtension]
			              stringByAppendingPathExtension:@"icns"];
	}

	if (!foundInputParam)
	{
		usage();
		[pool drain];
		return 1;
	}

	for (i = 0; i < N; ++i) {
		if (inputs[i].inputName != nil && ![[NSFileManager defaultManager] fileExistsAtPath:inputs[i].inputName])
		{
			fileDoesNotExist=TRUE;
			fprintf(stderr, "File '%s' does not exist!\n", [inputs[i].inputName UTF8String]);
			break;
		}
	}

	if (fileDoesNotExist)
	{
		[pool drain];
		return 2;
	}

	align=0;
	alignOption=[[args stringForKey:@"align"] lowercaseString];
	if ( alignOption!=nil ) align=[alignOption UTF8String][0];

	// Create output
	IconFamily* output = [IconFamily iconFamily];

	for (i = 0; i < N; ++i) {
		if (inputs[i].inputName == nil) continue;
		NSImage* img = [[[NSImage alloc] initWithContentsOfFile:inputs[i].inputName]
		                autorelease];

		NSBitmapImageRep* rep = getBitmapImageRepOfSize(img, inputs[i].size, align);
		[output setIconFamilyElement:inputs[i].type fromBitmapImageRep:rep];
		if (inputs[i].mask != 0)
			[output setIconFamilyElement:inputs[i].mask fromBitmapImageRep:rep];
	}

	// Write output
	if ([output writeToFile:outputName])
		NSLog(@"Wrote output file \"%@\"", outputName);
	else
		NSLog(@"Failed to write \"%@\"", outputName);

	[pool drain];
	return 0;
}
