/*
 * MacClipboard Mac OS X copy & paste plug-in
 * Copyright (C) 2004 Brion Vibber
 *
 * Based in part on WinClipboard
 * Copyright (C) 1999 Hans Breuer
 * Hans Breuer, Hans@Breuer.org
 * 08/07/99
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Based on (at least) the following plug-ins:
 *  Header
 *  WinSnap
 *  WinClipboard
 *
 * Any suggestions, bug-reports or patches are welcome.
 */

/*
The Mac clipboard ('pasteboard' or 'scrap') can contain data in multiple
formats. The least common denominator for graphic data is 'PICT', a QuickDraw
picture.

Cocoa apps prefer TIFF, which also lets us transfer alpha channel data
relatively reliably. When we post a TIFF, Cocoa does PICT translation for us.

PICTs are like metafiles in Windows GDI, and can be vector as well as
bitmap. The way to handle these seems to be to create an offscreen graphics
world and have QuickDraw draw the picture into it, then suck the bits out...
this deals with any vector stuff and also colorspace conversion, since we
can just ask for a 32-bit space.

Rich text editors may produce RTFD which contains embedded TIFF images; we
can extract these and paste them. (They may also contain EPS images; we
don't support those yet.)

CAVEATS:
* NSBitmapImageRep wants pre-multiplied alpha. This may introduce slight
  corruption on multiple copy-paste cycles.
* For large images, this probably wastes cycles and memory with the
  intermediate offscreen rendering.
* Indexed images are not supported as paste targets.
* Indexed images are copied as RGB; the indexed palette is not attached.


TODO:
* Support pasting into indexed images
* Support more TIFF options... colorspaces etc?
* Extract EPS images from RTFD?

One day, it'd be nice if this were built in to the system
and separate external copy/paste commands wouldn't be needed.

*/

/* #include "config.h" */

#import <Cocoa/Cocoa.h>
#include <Carbon/Carbon.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>

#include <gtk/gtk.h>

#include <libgimp/gimp.h>

#include <libintl.h>
#define _(String) gettext (String)

#include "clipboard_xfer.h"


/* FIXME: I'll use -1 as IMAGE_NONE. Is it correct ???
 */
#define IMAGE_NONE -1

/* Declare some local functions.
 */
static void   query      (void);
static void   run        (const gchar      *name,
                          gint              nparams,
                          const GimpParam  *param,
                          gint             *nreturn_vals,
                          GimpParam       **return_vals);

/* Plugin function prototypes
 */
static gboolean clipboard_copy             (gboolean          interactive,
                                            gint32            image_ID,
                                            gint32            drawable_ID);

static gboolean clipboard_paste            (gboolean          interactive,
                                            gint32            image_ID,
                                            gint32            drawable_ID);

static gboolean clipboard_paste_board      (gboolean          interactive,
                                            gint32            image_ID,
                                            gint32            drawable_ID,
                                            NSPasteboard     *board);

static gboolean clipboard_paste_service    (gboolean         interactive,
                                            NSString        *service);


/* PICT bits */
static gboolean clipboard_offscreen_pict   (PicHandle         pic,
                                            guchar            fill,
                                            guint32          *outWidth,
                                            guint32          *outHeight,
                                            guint32          *outRowstride,
                                            guchar          **outBits,
                                            GWorldPtr        *outGworld);

static gboolean clipboard_buffer_has_alpha (guchar           *bits,
                                            guint32           width,
                                            guint32           height,
                                            guint32           rowstride);

static gboolean clipboard_extract_alpha    (guchar           *bits,
                                            guchar           *bits_white,
                                            guint32           width,
                                            guint32           height,
                                            guint32           rowstride);

static gboolean clipboard_paste_pict       (gboolean          interactive,
                                            gint32            image_ID,
                                            gint32            drawable_ID,
                                            NSData           *data,
                                            gint32           *image_out);

/* Other image types */
static gboolean clipboard_paste_bitmap     (gboolean          interactive,
                                            gint32            image_ID,
                                            gint32            drawable_ID,
                                            NSBitmapImageRep *bitmap,
                                            gint32           *image_out);

/* The raw paste */
static gboolean clipboard_paste_image      (gboolean          interactive,
                                            gint32            image_ID,
                                            gint32            drawable_ID,
                                            NSSize            dpi,
                                            guchar           *bits,
                                            guint32           width,
                                            guint32           height,
                                            guint32           rowstride,
                                            guint32           channels,
                                            guint32           bitsPerPixel,
                                            guint32           layertype,
                                            xfer_func_type    xfer_func,
                                            guchar           *cmap,
                                            gint32           *image_out);

/* Meta-paste */
static gboolean clipboard_paste_rtfd       (gboolean         interactive,
                                            gint32           image_ID,
                                            gint32           drawable_ID,
                                            NSData          *data);

static gint32   clipboard_load_pict        (gboolean         interactive,
                                            gchar            *filename);

GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,  /* init_proc  */
  NULL,  /* quit_proc  */
  query, /* query_proc */
  run,   /* run_proc   */
};


MAIN ()

static void
query ()
{
  static GimpParamDef copy_args[] =
  {
    { GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
    { GIMP_PDB_IMAGE, "image", "Input image" },
    { GIMP_PDB_DRAWABLE, "drawable", "Drawable to save" }
  };
  static GimpParamDef load_args[] =
  {
    { GIMP_PDB_INT32,    "run_mode",     "Interactive, non-interactive" },
    { GIMP_PDB_STRING,   "filename",     "The name of the file to load" },
    { GIMP_PDB_STRING,   "raw_filename", "The name entered"             }
  };
  static GimpParamDef load_return_vals[] =
  {
    { GIMP_PDB_IMAGE, "image", "Output image" },
  };

  gimp_install_procedure ("plug_in_clipboard_copy",
                          _("copy image to clipboard"),
                          _("Copies the active drawable to the Macintosh clipboard."),
                          "Brion Vibber",
                          "Brion Vibber",
                          "2004",
                          _("<Image>/Edit/Copy to Clipboard"),
                          "INDEXED*, GRAY*, RGB*",
                          GIMP_PLUGIN,
                          G_N_ELEMENTS (copy_args), 0,
                          copy_args, NULL);

  gimp_install_procedure ("plug_in_clipboard_paste",
                          _("paste image from clipboard"),
                          _("Paste image from Macintosh clipboard into active image."),
                          "Brion Vibber",
                          "Brion Vibber",
                          "2004",
                          _("<Image>/Edit/Paste from Clipboard"),
                          "GRAY*, RGB*",
                          GIMP_PLUGIN,
                          G_N_ELEMENTS (copy_args), 0,
                          copy_args, NULL);

  gimp_install_procedure ("plug_in_clipboard_paste_as_new",
                          _("Get image from clipboard"),
                          _("Open the image on the Macintosh clipboard as a new document."),
                          "Brion Vibber",
                          "Brion Vibber",
                          "2004",
                          _("<Toolbox>/File/Acquire/From Clipboard"),
                          "",
                          GIMP_PLUGIN,
                          1, 0,
                          copy_args, NULL);

  gimp_install_procedure ("plug_in_clipboard_import_image",
                          _("Import from Image Capture"),
                          _("Import an image from a scanner, camera, or other device."),
                          "Brion Vibber",
                          "Brion Vibber",
                          "2004",
                          _("<Toolbox>/File/Acquire/Image Capture..."),
                          "",
                          GIMP_PLUGIN,
                          1, 0,
                          copy_args, NULL);

  gimp_install_procedure ("plug_in_clipboard_grab_screen",
                          _("Grab screenshot"),
                          _("Grab a full-screen screenshot using the Grab system service. The user must select a monitor to capture from."),
                          "Brion Vibber",
                          "Brion Vibber",
                          "2004",
                          _("<Toolbox>/File/Acquire/Grab/Screen..."),
                          "",
                          GIMP_PLUGIN,
                          1, 0,
                          copy_args, NULL);

  gimp_install_procedure ("plug_in_clipboard_grab_selection",
                          _("Grab selection from screen"),
                          _("Grab a partial screenshot with the Grab application; requiring the user to select an area of the screen to capture. A new window will be opened with the captured image."),
                          "Brion Vibber",
                          "Brion Vibber",
                          "2004",
                          _("<Toolbox>/File/Acquire/Grab/Selection..."),
                          "",
                          GIMP_PLUGIN,
                          1, 0,
                          copy_args, NULL);

  gimp_install_procedure ("plug_in_clipboard_grab_timed",
                          _("Grab screenshot after delay"),
                          _("Grab a delayed screenshot using the Grab system service"),
                          "Brion Vibber",
                          "Brion Vibber",
                          "2004",
                          _("<Toolbox>/File/Acquire/Grab/Timed Screen..."),
                          "",
                          GIMP_PLUGIN,
                          1, 0,
                          copy_args, NULL);

  gimp_install_procedure ("file_pict_load",
                          "Loads files of Macintosh PICT file format",
                          "Loads files of Macintosh PICT file format",
                          "Brion Vibber <brion@pobox.com>",
                          "Brion Vibber <brion@pobox.com>",
                          "2004",
                          "<Load>/pict",
                          NULL,
                          GIMP_PLUGIN,
                          G_N_ELEMENTS (load_args),
                          G_N_ELEMENTS (load_return_vals),
                          load_args, load_return_vals);

  gimp_register_magic_load_handler ("file_pict_load",
                                    "pict",
                                    "",
                                    "");
}

static void
run (const gchar      *name,
     gint              nparams,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  static GimpParam values[2];
  GimpRunMode run_mode;
  
  run_mode = param[0].data.d_int32;
  int interactive = (GIMP_RUN_INTERACTIVE==run_mode);
  int ok = FALSE;
  
  NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
  
  *nreturn_vals = 1;
  *return_vals = values;
  values[0].type = GIMP_PDB_STATUS;
  
  if (strcmp (name, "plug_in_clipboard_copy") == 0)
    ok = clipboard_copy (interactive,
                         param[1].data.d_int32,
                         param[2].data.d_int32);
  else if (strcmp (name, "plug_in_clipboard_paste") == 0)
    ok = clipboard_paste (interactive,
                          param[1].data.d_int32,
                          param[2].data.d_int32);
  else if (strcmp (name, "plug_in_clipboard_paste_as_new") == 0)
    ok = clipboard_paste (interactive, IMAGE_NONE, IMAGE_NONE);
  else if (strcmp (name, "plug_in_clipboard_import_image") == 0)
    ok = clipboard_paste_service (interactive, @"Import Image");
  else if (strcmp (name, "plug_in_clipboard_grab_screen") == 0)
    ok = clipboard_paste_service (interactive, @"Grab/Screen");
  else if (strcmp (name, "plug_in_clipboard_grab_selection") == 0)
    ok = clipboard_paste_service (interactive, @"Grab/Selection");
  else if (strcmp (name, "plug_in_clipboard_grab_timed") == 0)
    ok = clipboard_paste_service (interactive, @"Grab/Timed Screen");
  else if (strcmp (name, "file_pict_load") == 0)
    {
      gint32 image = clipboard_load_pict (interactive, param[1].data.d_string);
      ok = (image != -1);
      if (ok)
        {
          *nreturn_vals = 2;
          values[1].type         = GIMP_PDB_IMAGE;
          values[1].data.d_image = image;
        }
    }
  else
    {
      values[0].data.d_status = GIMP_PDB_CALLING_ERROR;
      [pool release];
      return;
    }
  
  values[0].data.d_status = ok ? GIMP_PDB_SUCCESS : GIMP_PDB_EXECUTION_ERROR;
  [pool release];
}

/*
 * Put some dummy data on the GDK/X11 clipboard and clear it; if we don't,
 * Apple X11's clipboard bridging interferes if there was already something
 * on it, and we can't get our new data onto the pasteboard.
 */
static void
clipboard_clear_gdk()
{
  GtkClipboard *board;

  gtk_init (NULL, NULL);
  board = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
  gtk_clipboard_set_text (board, "", 0);
  gtk_clipboard_clear (board);
}


/*
 * Copy the current layer to the pasteboard as a TIFF
 * FIXME: Limit to selection if any
 */
static gboolean
clipboard_copy (gboolean interactive,
                gint32   image_ID,
                gint32   drawable_ID)
{
  GimpDrawable *drawable;
  GimpImageType drawable_type;
  GimpPixelRgn pixel_rgn;

  clipboard_clear_gdk();

  /* Make sure we can get something to the clipboard first */
  NSPasteboard *board = [NSPasteboard generalPasteboard];
  [board
    declareTypes: [NSArray arrayWithObject: NSTIFFPboardType]
    owner: nil];

  /* Make the image... */
  drawable = gimp_drawable_get (drawable_ID);
  drawable_type = gimp_drawable_type (drawable_ID);
  gimp_pixel_rgn_init (&pixel_rgn, drawable, 0, 0, drawable->width, drawable->height, FALSE, FALSE);

  /* following the slow part ... */
  if (interactive)
    gimp_progress_init (_("Copying..."));
  int progress = 0;
  int max_progress = drawable->width * drawable->height;
  
  unsigned char *cmap = NULL;
  int nColors;
  int channels;
  NSString *colorSpace;
  BOOL alpha;
  xfer_func_type xfer_func;

  switch (drawable_type)
    {
    case GIMP_INDEXED_IMAGE:
      cmap = gimp_image_get_cmap (image_ID, &nColors);
      xfer_func = xfer_index_rgb;
      colorSpace = NSCalibratedRGBColorSpace;
      channels = 3;
      alpha = NO;
      break;
    case GIMP_INDEXEDA_IMAGE:
      cmap = gimp_image_get_cmap (image_ID, &nColors);
      xfer_func = xfer_indexa_rgbaP;
      colorSpace = NSCalibratedRGBColorSpace;
      channels = 4;
      alpha = YES;
      break;
    case GIMP_GRAY_IMAGE:
      xfer_func = xfer_gray_gray;
      colorSpace = NSCalibratedWhiteColorSpace;
      channels = 1;
      alpha = NO;
      break;
    case GIMP_GRAYA_IMAGE:
      xfer_func = xfer_graya_grayaP;
      colorSpace = NSCalibratedWhiteColorSpace;
      channels = 2;
      alpha = YES;
      break;
    case GIMP_RGB_IMAGE:
      xfer_func = xfer_rgb_rgb;
      colorSpace = NSCalibratedRGBColorSpace;
      channels = 3;
      alpha = NO;
      break;
    case GIMP_RGBA_IMAGE:
      xfer_func = xfer_rgba_rgbaP;
      colorSpace = NSCalibratedRGBColorSpace;
      channels = 4;
      alpha = YES;
      break;
    default:
      g_message(_("Unsupported image type; can't copy."));
      return FALSE;
    }

  guint32 width = drawable->width;
  guint32 height = drawable->height;
  guint32 bpp = drawable->bpp;
  guint32 row = width * channels;

  // align
  if ((row & 0x3) != 0)
    row += 4 - (row & 0x03);
  guchar *bits = malloc(row * height);

  int y;

  GimpPixelRgn region;
  gimp_pixel_rgn_init(&region, drawable, 0, 0, width, height, FALSE, FALSE);

  gpointer pr;
  for (pr = gimp_pixel_rgns_register (1, &region);
       pr != NULL;
       pr = gimp_pixel_rgns_process (pr))
    {
      for (y = 0; y < region.h; y++)
        xfer_func (&region.data[y*region.rowstride],
                   &bits[(y+region.y)*row+region.x*channels],
                   region.w, cmap);
      progress += (region.w * region.h);
      gimp_progress_update ((double) progress / (double) max_progress);
    }

  if (cmap != NULL)
    g_free (cmap);

  NSBitmapImageRep *bitmap = [[NSBitmapImageRep alloc]
    initWithBitmapDataPlanes: &bits
    pixelsWide: width
    pixelsHigh: height
    bitsPerSample: 8
    samplesPerPixel: channels
    hasAlpha: alpha
    isPlanar: NO
    colorSpaceName: colorSpace
    bytesPerRow: row
    bitsPerPixel: channels*8];

  if (bitmap == nil)
    {
      g_message(_("Couldn't create copy of bitmap."));
      free(bits);
    }
  else
    {
      [board setData: [bitmap TIFFRepresentation] forType: NSTIFFPboardType];

      /* Force conversion to PICT before we leave */
      [board types];
      [board dataForType: NSPICTPboardType];
    }

  return TRUE;
}

/*
 * Paste from the general pasteboard into new or pre-existing image.
 */
static gboolean
clipboard_paste (gboolean interactive,
                 gint32   image_ID,
                 gint32   drawable_ID)
{
  NSPasteboard *board = [NSPasteboard generalPasteboard];
  return clipboard_paste_board (interactive, image_ID, drawable_ID, board);
}

/*
 * Call a given service that provides images, and paste from that.
 * Will always create a new image.
 */
static gboolean
clipboard_paste_service (gboolean   interactive,
                         NSString  *service)
{
  NSApplication *app = [NSApplication sharedApplication];
  [app registerServicesMenuSendTypes: nil
       returnTypes: [NSArray arrayWithObjects: NSTIFFPboardType, nil]];

  NSPasteboard *board = [NSPasteboard pasteboardWithUniqueName];

  [board declareTypes: [NSArray array] owner: nil];
  if (NSPerformService(service, board))
    return clipboard_paste_board (interactive, IMAGE_NONE, IMAGE_NONE, board);

  g_message (_("Couldn't run system service %s"), [service cString]);
  return FALSE;
}




/*
 * Find pasteable image(s) from a given pasteboard and um... paste them.
 */
static gboolean
clipboard_paste_board (gboolean   interactive,
                       gint32     image_ID,
                       gint32     drawable_ID,
                       NSPasteboard *board)
{
  gboolean retval = FALSE;

  NSString *type = [board availableTypeFromArray:
    [NSArray arrayWithObjects:
      NSTIFFPboardType,
      NSPICTPboardType,
      NSRTFDPboardType,
      nil]];
  if (type == nil)
    {
      g_message (_("Clipboard is empty or doesn't contain a picture."));
      return FALSE;
	}

  NSData *data = [board dataForType: type];
  if (data == nil)
    {
      g_message (_("Couldn't get image from clipboard."));
      return FALSE;
    }

  if ([type isEqualToString: NSTIFFPboardType])
    return clipboard_paste_bitmap (interactive, image_ID, drawable_ID,
      [NSBitmapImageRep imageRepWithData: data], NULL);
  else if ([type isEqualToString: NSPICTPboardType])
    return clipboard_paste_pict (interactive, image_ID, drawable_ID, data, NULL);
  else if ([type isEqualToString: NSRTFDPboardType])
    return clipboard_paste_rtfd (interactive, image_ID, drawable_ID, data);

  g_message (_("Unsupported type. This shouldn't happen!"));
  return FALSE;
}


/*
 * Render a given PICT into a newly created offscreen 32-bit graphics
 * world, returning some handy information about the bits.
 *
 *   fill should be either 0x00 (transparent) or 0xff (solid white) 
 *   *outGworld must be released by caller with DisposeGWorld()
 */
static gboolean
clipboard_offscreen_pict (PicHandle  pic,
                          guchar     fill,
                          guint32   *outWidth,
                          guint32   *outHeight,
                          guint32   *outRowstride,
                          guchar   **outBits,
                          GWorldPtr *outGworld)
{
  OSStatus err;
  PictInfo info;
  err = GetPictInfo (pic, &info, 0, 0, 0, 0);
  if (err)
    {
      g_warning ("GetPictInfo returned %d", (int)err);
      return FALSE;
    }
  
  guint32 width = info.sourceRect.right - info.sourceRect.left;
  guint32 height = info.sourceRect.bottom - info.sourceRect.top;
  *outWidth = width;
  *outHeight = height;
  
  GWorldPtr gworld;
  QDErr qerr = NewGWorld (&gworld, 32, &info.sourceRect, NULL, NULL, 0);
  if (qerr)
    {
      g_warning ("NewGWorld returned %d", (int)qerr);
      return FALSE;
    }
  *outGworld = gworld;
  
  SetGWorld (gworld, NULL);
  PixMapHandle pixmap = GetGWorldPixMap (gworld);
  guchar *bits = (guchar *)GetPixBaseAddr (pixmap);
  *outBits = bits;
  
  guint32 rowstride = (*pixmap)->rowBytes & 0x3fff;
  *outRowstride = rowstride;
  
  /* Sometimes the picture won't fill the whole area;
     pre-fill the background as transparent or solid */
  memset (bits, fill, rowstride * height);

  DrawPicture (pic, &info.sourceRect);
  return TRUE;
}

/*
 * Returns TRUE if non-zero alpha values are detected.
 * Doesn't detect whether the image is actually transparent or not.
 */
static gboolean
clipboard_buffer_has_alpha (guchar  *bits,
                            guint32  width,
                            guint32  height,
                            guint32  rowstride)
{
  int x, y;
  for (y = 0; y < height; y++)
    for (x = 0; x < width; x++)
      if (0 != bits[y * rowstride + x * 4])
        return TRUE;
  return FALSE;
}

/*
 * Awful hack for pasting alpha images from Photoshop; we can't get the
 * alpha channel values directly out of it.
 *
 * On input:
 *  bits contains aRGB image premultipled with black
 *  bits_white contains aRGB image premultipled with white
 *
 * On output, the alpha channel values in bits are filled in.
 */
static gboolean
clipboard_extract_alpha (guchar  *bits,
                         guchar  *bits_white,
                         guint32  width,
                         guint32  height,
                         guint32  rowstride)
{
  guint32 x, y, n, rdiff, gdiff, bdiff;
  for (y = 0; y < height; y++)
    {
      n = y * rowstride;
      for (x = 0; x < width; x++)
        {
          /*
             d' = a * s + (1 - a) * d
          where:
                        1 = d
               bits_white = d'
                     bits = a * s

             a = 1 - bits_white + bits
          */
          
          /* Rounding may put the channels off by one */
          rdiff = 0xff - bits_white[n + 1] + bits[n + 1];
          /* gdiff = 0xff - bits_white[n + 2] + bits[n + 2]; */
          /* bdiff = 0xff - bits_white[n + 3] + bits[n + 3]; */
          bits[n] = rdiff;
          n += 4;
          /* g_print("reconstructed alpha %02x %02x %02x\n", rdiff, gdiff, bdiff); */
        }
    }
  return TRUE;
}

/*
 * Paste a PICT using QuickDraw
 */
static gboolean
clipboard_paste_pict (gboolean interactive,
                      gint32   image_ID,
                      gint32   drawable_ID,
                      NSData  *data,
                      gint32  *image_out)
{
  PicHandle pic;
  pic = (PicHandle)NewHandle ([data length]);
  if (pic == NULL)
    {
      g_warning ("clipboard_paste_pict: Couldn't allocate memory for picture (NewHandle returned FALSE)");
      return FALSE;
	}

  [data getBytes: *pic];

  guint32   width, height, rowstride;
  GWorldPtr gworld;
  guchar    *bits;
  if (! clipboard_offscreen_pict (pic, 0x00,
           &width, &height, &rowstride,
           &bits, &gworld))
    {
      DisposeHandle ((Handle)pic);
      return FALSE;
    }
  
  gboolean hasalpha = clipboard_buffer_has_alpha (bits, width, height, rowstride);
  if (! hasalpha)
    {
      /* PICTs pasted from Photoshop don't touch the alpha
         channel but do store transparency. Ugly hack ahead!
         We render the image again on a white background,
         then extract alpha values from the difference.     */
      GWorldPtr gworld_white;
      guchar    *bits_white;
      if (clipboard_offscreen_pict (pic, 0xff,
                                    &width, &height, &rowstride,
                                    &bits_white, &gworld_white))
        {
          hasalpha = clipboard_extract_alpha (bits, bits_white,
                                              width, height, rowstride);
          DisposeGWorld (gworld_white);
        }
    }
  
  guint32 layertype;
  xfer_func_type xfer_func;
  if (image_ID != IMAGE_NONE)
    {
      GimpDrawable *drawable = gimp_drawable_get (drawable_ID);
      if (drawable->bpp < 3)
        {
          layertype = GIMP_GRAYA_IMAGE;
          if (hasalpha)
            xfer_func = xfer_argbP_graya;
          else
            xfer_func = xfer_0rgb_graya;
        }
      else
        {
          layertype = GIMP_RGBA_IMAGE;
          if (hasalpha)
            xfer_func = xfer_argbP_rgba;
          else
            xfer_func = xfer_0rgb_rgba;
        }
    }
  else
    {
      if (hasalpha)
        {
          layertype = GIMP_RGBA_IMAGE;
          xfer_func = xfer_argbP_rgba;
        }
      else
        {
          layertype = GIMP_RGB_IMAGE;
          xfer_func = xfer_0rgb_rgb;
        }
    }
  
  NSSize dpi = {72.0, 72.0};
  gboolean retval = clipboard_paste_image (interactive, image_ID, drawable_ID, dpi,
                                           bits, width, height, rowstride,
                                           4, 32, layertype, xfer_func, NULL, image_out);
  
  DisposeGWorld (gworld);
  DisposeHandle ((Handle)pic);
  
  return retval;
}

static gboolean
clipboard_paste_image (gboolean       interactive,
                       gint32         image_ID,
                       gint32         drawable_ID,
                       NSSize         dpi,
                       guchar        *bits,
                       guint32        width,
                       guint32        height,
                       guint32        rowstride,
                       guint32        channels,
                       guint32        bitsPerPixel,
                       guint32        layertype,
                       xfer_func_type xfer_func,
                       guchar        *cmap,
                       gint32        *image_out)
{
  gchar   *layername;
  guint32  float_on;
  gboolean is_new;
  
  if (image_ID == IMAGE_NONE)
    {
      /* Create an appropriate new image */
      guint32 imagetype;
      switch (layertype)
        {
        case GIMP_RGB_IMAGE:
        case GIMP_RGBA_IMAGE:
          imagetype = GIMP_RGB;
          break;
        case GIMP_GRAY_IMAGE:
        case GIMP_GRAYA_IMAGE:
          imagetype = GIMP_GRAY;
          break;
        default:
          g_warning ("clipboard_paste_image: Invalid layer type requested (%d)",
            (int)layertype);
          return FALSE;
        }
      image_ID = gimp_image_new (width, height, imagetype);
      gimp_image_undo_disable (image_ID);
      gimp_image_set_resolution (image_ID, dpi.width, dpi.height);
      gimp_image_set_unit (image_ID, GIMP_UNIT_INCH);
      layername = _("Background");
      float_on = 0;
      is_new = TRUE;
    }
  else
    {
      /* Add new layer as floating selection */
      layername = _("Pasted Layer");
      float_on = drawable_ID;
      is_new = FALSE;
    }
  drawable_ID = gimp_layer_new (image_ID, layername, width, height,
                                layertype, 100, GIMP_NORMAL_MODE);
  if (! float_on)
    gimp_image_add_layer (image_ID, drawable_ID, -1);
  
  GimpDrawable *drawable = gimp_drawable_get (drawable_ID);

  GimpPixelRgn region;
  gimp_pixel_rgn_init (&region, drawable, 0, 0, width, height, TRUE, FALSE);

  if (interactive)
    gimp_progress_init (_("Pasting..."));
  int progress = 0;
  int max_progress = width * height;
  
  gpointer pr;
  for (pr = gimp_pixel_rgns_register (1, &region);
       pr != NULL;
       pr = gimp_pixel_rgns_process (pr))
    {
      int y;
      for (y = 0; y < region.h; y++)
        {
          xfer_func (&bits[(y + region.y) * rowstride + region.x * bitsPerPixel / 8],
                     &region.data[y * region.rowstride], region.w, cmap);
        }
      progress += (region.w * region.h);
      gimp_progress_update ((double) progress / (double) max_progress);
    }

  if (float_on)
    gimp_floating_sel_attach(drawable_ID, float_on);
  gimp_drawable_flush (drawable);
  gimp_drawable_detach (drawable);

  /* Don't forget to display the new image! */
  /* ...except when loading a file */
  if (is_new && (image_out == NULL))
    {
      gimp_display_new (image_ID);
      gimp_image_undo_enable (image_ID);
    }
  else
    {
      gimp_drawable_set_visible (drawable_ID, TRUE);
      gimp_displays_flush ();
    }
  if (image_out != NULL)
    *image_out = image_ID;
  return TRUE;
}

/*
 * Paste types that can be read by NSBitmapImageRep (mainly TIFF)
 * This will always use premultiplied alpha, so we can't do a clean
 * round-trip of a partially transparent iamge.
 */
static gboolean
clipboard_paste_bitmap (gboolean          interactive,
                        gint32            image_ID,
                        gint32            drawable_ID,
                        NSBitmapImageRep *bitmap,
                        gint32           *image_out)
{
  if (bitmap == nil)
    {
      g_message(_("Couldn't get image data."));
      return FALSE;
    }

  guint32 alpha = [bitmap hasAlpha] ? 1 : 0;
  guint32 width = [bitmap pixelsWide];
  guint32 height = [bitmap pixelsHigh];

  guint32 bps = [bitmap bitsPerPixel];
  guint32 rowstride = [bitmap bytesPerRow];
  guint32 channels = [bitmap samplesPerPixel];
  
  NSSize size = [bitmap size];
  NSSize dpi;
  dpi.width = (double)width * 72.0 / size.width;
  dpi.height = (double)height * 72.0 / size.height;
  NSLog(@"Size is %f by %f, pixels is %d by %d: %f x %f dpi",
    size.width, size.height, width, height, dpi.width, dpi.height);
  
  guchar  *cmap = NULL;
  gboolean grayPositive;

  NSString *colorSpace = [bitmap colorSpaceName];
  if ([colorSpace isEqualToString: NSCalibratedRGBColorSpace] ||
      [colorSpace isEqualToString: NSDeviceRGBColorSpace])
    {
      if (channels != (3 + alpha))
        {
          g_message (_("Don't understand %d-channel image in RGB image (%s; alpha: %s)"),
                     channels, [colorSpace cString], alpha ? "yes" : "no");
          return FALSE;
        }
    }
  else if ([colorSpace isEqualToString: NSCalibratedWhiteColorSpace] ||
           [colorSpace isEqualToString: NSDeviceWhiteColorSpace])
    {
      grayPositive = TRUE;
      if (channels != (1 + alpha))
        {
          g_message (_("Don't understand %d-channel image in grayscale image (%s; alpha: %s)"),
                     channels, [colorSpace cString], alpha ? "yes" : "no");
          return FALSE;
        }
    }
  else if ([colorSpace isEqualToString: NSCalibratedBlackColorSpace] ||
           [colorSpace isEqualToString: NSDeviceBlackColorSpace])
    {
      grayPositive = FALSE;
      if (channels != (1 + alpha))
        {
          g_message (_("Don't understand %d-channel image in grayscale image (%s; alpha: %s)"),
                     channels, [colorSpace cString], alpha ? "yes" : "no");
          return FALSE;
        }
    }
  else
    {
      g_message (_("Unknown colorspace! %s"), [colorSpace cString]);
      return FALSE;
    }

  guint32 sampleBits = [bitmap bitsPerSample];
  if (sampleBits != 8 && (sampleBits != 1 || channels != 1 || image_ID != IMAGE_NONE))
    {
      /* 1-bit supported for newly pasted gray images only! */
      g_message(_("Unsupported bits per sample %d"), sampleBits);
      return FALSE;
    }

  if ([bitmap isPlanar])
    {
      g_message(_("Planar images not supported."));
      return FALSE;
    }

  guchar *bits = [bitmap bitmapData];

  gboolean gray;
  xfer_func_type xfer_func;
  guint32 imagetype, layertype;
  
  /* 1-bit color maps */
  guchar whitemap[] = {0, 0, 0,  255, 255, 255};
  guchar blackmap[] = {255, 255, 255,  0, 0, 0};
  
  if (IMAGE_NONE == image_ID)
    {
      /* create new image */
      switch (channels)
        {
        case 1:
          imagetype = GIMP_GRAY;
          layertype = GIMP_GRAY_IMAGE;
          gray = TRUE;
          if (sampleBits == 1)
            {
              xfer_func = xfer_mono_gray;
              cmap = grayPositive ? whitemap : blackmap;
            }
          else
            xfer_func = xfer_gray_gray;
          break;
        case 2:
          imagetype = GIMP_GRAY;
          layertype = GIMP_GRAYA_IMAGE;
          xfer_func = xfer_grayaP_graya;
          gray = TRUE;
          break;
        case 3:
          imagetype = GIMP_RGB;
          layertype = GIMP_RGB_IMAGE;
          xfer_func = xfer_rgb_rgb;
          gray = FALSE;
          break;
        case 4:
        default:
          imagetype = GIMP_RGB;
          layertype = GIMP_RGBA_IMAGE;
          xfer_func = xfer_rgbaP_rgba;
          gray = FALSE;
        }
    }
  else
    {
      GimpDrawable *drawable = gimp_drawable_get (drawable_ID);
      gray = (drawable->bpp < 3); // no indexed we hope...
      if (gray)
        {
          layertype = GIMP_GRAYA_IMAGE;
          switch (channels)
            {
            case 1: xfer_func = xfer_gray_graya; break;
            case 2: xfer_func = xfer_grayaP_graya; break;
            case 3: xfer_func = xfer_rgb_graya; break;
            case 4: xfer_func = xfer_rgbaP_graya; break;
            default: g_message("urk"); return FALSE;
            }
        }
      else
        {
          layertype = GIMP_RGBA_IMAGE;
          switch(channels)
            {
            case 1: xfer_func = xfer_gray_rgba; break;
            case 2: xfer_func = xfer_grayaP_rgba; break;
            case 3: xfer_func = xfer_rgb_rgba; break;
            case 4: xfer_func = xfer_rgbaP_rgba; break;
            default: g_message("urk"); return FALSE;
            }
        }
    }

  return clipboard_paste_image (interactive, image_ID, drawable_ID, dpi,
                                bits, width, height, rowstride,
                                channels, bps, layertype, xfer_func, cmap, image_out);
}

static gboolean clipboard_paste_rtfd (gboolean   interactive,
			  gint32     image_ID,
			  gint32     drawable_ID,
			  NSData*    data)
{
  /* Look for tasty TIFFs inside ... */
  NSFileWrapper *wrap = [[NSFileWrapper alloc]
    initWithSerializedRepresentation: data];
  if (wrap != nil)
    {
      if ([wrap isDirectory])
        {
          NSDictionary *files = [wrap fileWrappers];
          if (files != nil)
            {
              NSEnumerator *e = [files keyEnumerator];
              NSString *key;
              int numCopied = 0;
              while ((key = [e nextObject]))
                {
                  NSFileWrapper *item = [files objectForKey: key];
                  if ([[item filename] hasSuffix: @".tiff"])
                    {
                      if (clipboard_paste_bitmap (interactive,
                              image_ID, drawable_ID,
                              [NSBitmapImageRep imageRepWithData:
                                [item regularFileContents]], NULL))
                        numCopied++;
                      else
                        NSLog (@"Not a TIFF: %s", [item filename]);
                    }
                }
              if (numCopied == 0)
                {
                  g_message (_("Couldn't extract any images from rich text on clipboard."));
                  return FALSE;
                }
              return TRUE;
            }
          else
            NSLog (@"RTFD fileWrappers returned nil.");
        }
      else
        NSLog (@"RTFD didn't contain a directory.");
    }
  else
    NSLog (@"Couldn't read rich text data.");
  g_message (_("Rich text on clipboard is corrupt, can't read."));
  return FALSE;
}

static gint32
clipboard_load_pict (gboolean  interactive,
                     gchar    *filename)
{
  struct stat filestat;
  if (0 == stat (filename, &filestat))
    {
      long fsize = filestat.st_size - 512;
      guchar *buf = malloc (fsize);
      FILE *file = fopen (filename, "rb");
      fseek (file, 512, SEEK_SET);
      fread (buf, 1, fsize, file);
      fclose (file);
      NSData *data = [NSData dataWithBytesNoCopy: buf length: fsize];
      if (data != nil)
        {
          gint32 image_out;
          if (clipboard_paste_pict (interactive, -1, -1, data, &image_out))
            {
              gimp_image_set_filename (image_out, filename);
              return image_out;
            }
        }
    }
  g_message ("Could not open file %s", filename);
  return -1;
}
