/*  luash, a Lua Shell for your console
    Copyright (C) 2009  Roland Brochard

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as
    published by the Free Software Foundation, either version 3 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

#include <iostream>
#include <sstream>
#include <stdlib.h>
#include <stdio.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <lua.hpp>
#include "lua_code.h"

#define __version_string__	"1.0.0"

using namespace std;

lua_State *L;

/* **************************************************************** */
/*                                                                  */
/*                  Interface to Readline Completion                */
/*                                                                  */
/* **************************************************************** */

char **luashell_completion PARAMS((const char *, int, int));

/* Tell the GNU Readline library how to complete.  We want to try to complete
   on command names if this is the first word in the line, or on filenames
   if not. */
void initialize_readline ()
{
  /* Allow conditional parsing of the ~/.inputrc file. */
  rl_readline_name = "FileMan";

  /* Tell the completer that we want a crack first. */
  rl_attempted_completion_function = luashell_completion;
}

/* Attempt to complete on the contents of TEXT.  START and END bound the
   region of rl_line_buffer that contains the word to complete.  TEXT is
   the word to complete.  We can use the entire contents of rl_line_buffer
   in case we want to do some simple parsing.  Return the array of matches,
   or NULL if there aren't any. */
char **luashell_completion (const char *text, int start, int end)
{
  char **matches;

  matches = (char **)NULL;

  /* If this word is at the start of the line, then it is a command
     to complete.  Otherwise it is the name of a file in the current
     directory. */
  if (start == 0)
  {
    lua_getglobal(L, "_auto_complete");
    lua_pushstring(L, text);
    if (lua_pcall(L, 1, 2, 0))
    {
      cout << lua_tostring(L, -1) << endl;
      return (matches);
    }
    int l = lua_tointeger(L, -2);
    if (l == 0)
    {
      lua_pop(L, 2);
      return (matches);
    }
    matches = (char**)malloc(sizeof(char*) * (l + 1));

    for(int i = 0 ; i < l ; ++i)
    {
      lua_rawgeti(L, -1, 1 + i);
      matches[i] = strdup(lua_tostring(L, -1));
      lua_pop(L, 1);
    }
    lua_pop(L, 2);
    matches[l] = NULL;
  }

  return (matches);
}

char *command = NULL;
bool nextTimeExit = false;

int lua_readline(lua_State *L)
{
  if (command)
  {
    if (nextTimeExit)
      exit(0);
    lua_pushstring(L, command);
    nextTimeExit = true;
    return 1;
  }
  
  lua_getglobal(L, "_PROMPT");
  char *l = readline(lua_tostring(L, -1));
  lua_pop(L, 1);
  if (l)
  {
    lua_pushstring(L, l);
    add_history(l);
    stifle_history(100);
    free(l);
  }
  else
    lua_pushnil(L);
  return 1;
}

int lua_getpid(lua_State *L)
{
  lua_pushinteger(L, getpid());
  return 1;
}

int lua_getppid(lua_State *L)
{
  lua_pushinteger(L, getppid());
  return 1;
}

int main(int argc, char **argv)
{
  setenv("SHELL", argv[0], 1);
  if (argc > 1)
  {
    int _argc = argc - 1;
    char **_argv = argv + 1;
    for(; _argc > 0 ; --_argc, ++_argv)
    {
      if (strcmp(*_argv, "--version") == 0)
      {
	cout << "Lua Shell, version " << __version_string__ << endl;
	cout << "Copyright (C) 2009 Roland Brochard" << endl;
	cout << "License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>" << endl << endl;
	cout << "This is free software; you are free to change and redistribute it." << endl;
	cout << "There is NO WARRANTY, to the extent permitted by law." << endl;
	return 0;
      }
      if (strcmp(*_argv, "--help") == 0)
      {
	cout << "Lua Shell, version " << __version_string__ << endl;
	cout << "Usage:\tluash [option]" << endl;
	cout << "options:" << endl;
	cout << "\t\t--help" << endl;
	cout << "\t\t--version" << endl;
	cout << "\t\t-c command" << endl;
	return 0;
      }
      if (strcmp(*_argv, "-c") == 0)
      {
	--_argc;
	++_argv;
	stringstream cmd;
	bool first = true;
	bool special = false;
	bool commandName = true;
	for(;_argc > 0;--_argc,++_argv)
	{
	  special |= (strcmp(*_argv, "/") == 0);
	  if (!first)
	    cmd << ' ';
	  if (!special && !commandName)
	    cmd << '"';
	  for(char *cur = *_argv ; *cur ; ++cur)
	  {
	    if (*cur == '"' && !special && !commandName)
	      cmd << '\\';
	    cmd << *cur;
	  }
	  if (!first && !special && !commandName)
	    cmd << '"';
	  first = false;
	  commandName = special;
	  special = false;
	}
	command = strdup(cmd.str().c_str());
      }
    }
  }
  initialize_readline ();

  time_t timer = time(NULL);
  int n = 0;
  
  while(true)
  {
    L = lua_open();
    luaL_openlibs(L);
    
    lua_register(L, "readline", lua_readline);
    lua_register(L, "getpid", lua_getpid);
    lua_register(L, "getppid", lua_getppid);
    
    if (n >= 2)		// If the shell keeps crashing, then load a safe environment
    {
      if (command)	// If we try to run a command, then we failled
	exit(0);
      lua_pushboolean(L, true);
      lua_setglobal(L, "luashellsafemode");
    }
    else if (command)
    {
      lua_pushboolean(L, true);
      lua_setglobal(L, "luashellhideintro");
    }
    
    if (luaL_loadstring(L, lua_code))
    {
      cout << lua_tostring(L, -1) << endl;
    }
    lua_pcall(L, 0, 0, 0);
    cout << lua_tostring(L, -1) << endl;
    cout << "shell crashed! reseting shell" << endl;
    
    lua_close(L);
    if (time(NULL) - timer > 60)
    {
      timer = time(NULL);
      n = 0;
    }
    ++n;
  }
  
  return 0;
}