---
-- Filter a complex-valued signal with a complex-valued FIR band-stop filter
-- generated by the window design method. This filter is asymmetric in the
-- frequency domain.
--
-- $$ y[n] = (x * h_{bsf})[n] $$
--
-- @category Filtering
-- @block ComplexBandstopFilterBlock
-- @tparam int num_taps Number of FIR taps, must be odd
-- @tparam {number,number} cutoffs Cutoff frequencies in Hz
-- @tparam[opt=nil] number nyquist Nyquist frequency, if specifying
--                                 normalized cutoff frequencies
-- @tparam[opt='hamming'] string window Window type
--
-- @signature in:ComplexFloat32 > out:ComplexFloat32
--
-- @usage
-- -- Complex bandstop filter, 128 taps, 18 kHz to 20 kHz
-- local bpf1 = radio.BandstopFilterBlock(128, {18e3, 20e3})
--
-- -- Complex bandstop filter, 128 taps, -18 kHz to -20 kHz
-- local bpf2 = radio.BandstopFilterBlock(128, {-18e3, -20e3})

local ffi = require('ffi')

local block = require('radio.core.block')
local types = require('radio.types')
local filter_utils = require('radio.utilities.filter_utils')

local FIRFilterBlock = require('radio.blocks.signal.firfilter')

local ComplexBandstopFilterBlock = block.factory("ComplexBandstopFilterBlock", FIRFilterBlock)

function ComplexBandstopFilterBlock:instantiate(num_taps, cutoffs, nyquist, window)
    assert(num_taps, "Missing argument #1 (num_taps)")
    self.cutoffs = assert(cutoffs, "Missing argument #2 (cutoffs)")
    self.window = window or "hamming"
    self.nyquist = nyquist

    FIRFilterBlock.instantiate(self, types.ComplexFloat32.vector(num_taps))
end

function ComplexBandstopFilterBlock:initialize()
    -- Compute Nyquist frequency
    local nyquist = self.nyquist or (self:get_rate()/2)

    -- Generate taps
    local cutoffs = {self.cutoffs[1]/nyquist, self.cutoffs[2]/nyquist}
    local taps = filter_utils.firwin_complex_bandstop(self.taps.length, cutoffs, self.window)
    self.taps = types.ComplexFloat32.vector_from_array(taps)

    FIRFilterBlock.initialize(self)
end

return ComplexBandstopFilterBlock
