# Comparison to GNU Radio

Both GNU Radio and LuaRadio are frameworks for flow graph signal processing.
Many of the concepts regarding flow graphs and blocks in LuaRadio imitate or
are inspired by GNU Radio.

However, LuaRadio differs from GNU Radio in many respects:

* [Purpose](#purpose)
* [Footprint](#footprint)
* [License](#license)
* [Extension](#extension)
* [Features](#features)
    * [Type Signatures](#type-signatures)
    * [Sample Rate Propagation](#sample-rate-propagation)
    * [Embeddability](#embeddability)
* [Performance](#performance)
* [Maturity](#maturity)

## Purpose

LuaRadio is not a GNU Radio replacement, but rather an alternative inclined
more towards scripting and prototyping, emphasizing fast, compilation-free
block development.

## Footprint

GNU Radio tends to have a large size footprint. A typical binary GNU Radio
installation weighs in at over 100 MB. When the installation includes
development support, the [boost](https://www.boost.org/) development headers
and numerous other libraries and tools (gsl, blas, swig, orc, etc.) can add
another 100 MB or more.

GNU Radio also has a large compilation footprint, in terms of time and
dependencies. Compilation of the GNU Radio framework and its modules is
generally slow, and can be temperamental due to its dependency on boost.
Compiled versions of GNU Radio modules and applications can break in complex
ways when the underlying GNU Radio or boost runtimes are updated. These modules
and applications may require routine re-compilation — sometimes in a particular
order — to be linked correctly with the updated libraries and each other.

LuaRadio's total footprint is on the order of 1 MB and its blocks require no
compilation. LuaRadio has zero dependencies for non-real-time applications,
and may otherwise use [liquid-dsp](https://github.com/jgaeddert/liquid-dsp) or
[VOLK](http://libvolk.org/) for acceleration in real-time applications.
[liquid-dsp](https://github.com/jgaeddert/liquid-dsp) shares LuaRadio's
affinity for minimal dependencies and making software-defined radio more
accessible to embedded Linux platforms.

## License

GNU Radio is licensed GPL v3, a
[copyleft](https://en.wikipedia.org/wiki/Copyleft) license.

LuaRadio is licensed MIT, a more permissive license.

## Extension

GNU Radio blocks are typically written in C++, and tend to have a lot of
boilerplate for building, Python binding, graphical editor binding, and
testing. This is mitigated with tooling, i.e. the `gr_modtool`, but to some,
this may contribute to the opaqueness of the framework. See the [Working with
GNU Radio in C++
tutorial](https://wiki.gnuradio.org/index.php/Guided_Tutorial_GNU_Radio_in_C%2B%2B)
for examples of the block development flow and block skeletons.

GNU Radio C++ blocks naturally require compilation before they can be used in a
flow graph, which affects the ease of their iteration. C++ blocks sometimes use
a custom form of string templating (e.g.
[`add_XX_impl.cc.t`](https://github.com/gnuradio/gnuradio/blob/v3.7.9.2/gr-blocks/lib/add_XX_impl.cc.t#L53)),
despite being written in C++, to code generate support for multiple data types.
C++ blocks are usually bound to Python with [SWIG](http://www.swig.org/), which
generates usable, but unnatural bindings. Blocks can also be written in pure
Python (see the [Working with GNU Radio in Python
tutorial](https://wiki.gnuradio.org/index.php/Guided_Tutorial_GNU_Radio_in_Python)),
and though they are limited in performance compared to C++ blocks, do provide
an avenue for more rapid prototyping. Altogether, the uninitiated may still
find GNU Radio block development shrouded in boilerplate and esoterica. For
examples of typical GNU Radio blocks, see the source code of one of the
third-party modules on [CGRAN](http://www.cgran.org/).

GNU Radio does not support creating custom first-class data types for
serialization between blocks, but blocks may exchange Protocol Data Units
(PDUs) with [Tagged
Streams](https://www.gnuradio.org/doc/doxygen/page_tagged_stream_blocks.html),
which demarcate the boundaries of the structured data, or use asynchronous
[Message Passing](https://www.gnuradio.org/doc/doxygen/page_msg_passing.html),
which pass data out-of-band of flow graph streams.

LuaRadio blocks are written in pure Lua, have substantially less boilerplate,
and require no compilation or bindings. With LuaRadio, the convenience of
scripting applies to both defining flow graphs and developing blocks. For
examples of typical LuaRadio blocks, see the [Creating
Blocks](3.creating-blocks.md) guide.

LuaRadio supports creating custom first-class data types based on fixed-size C
structures or variable-sized Lua objects for serialization between blocks. This
makes it easy and natural to create blocks that produce structured data, e.g.
decoded packets. See the custom types section of [Creating
Blocks](3.creating-blocks.md#custom-types) for examples.

Compare and contrast GNU Radio and LuaRadio blocks in their source trees: [GNU
Radio](https://github.com/gnuradio/gnuradio/tree/master/gr-blocks/lib),
[LuaRadio](../radio/blocks/signal/).

## Features

### Type Signatures

GNU Radio's I/O signature system is limited to matching sizes of types (e.g.
`sizeof(gr_complex)`) in block connections, whereas LuaRadio's type signature
system is safer and more flexible. Types in block connections are matched
explicitly in LuaRadio, and input types can also be specified by a function
predicate. For example, the [`JSONSink`](0.reference-manual.md#jsonsink) block
can accept any data type that implements `to_json()`.

GNU Radio blocks can only support one type signature each, which pushes the
problem of block type differentiation to the users. GNU Radio has developed a
rich nomenclature for annotating their block names with their type signature,
using obscure suffixes like `_cc` and `_ff`, e.g. `multiply_cc`, `multiply_ff`,
`fir_filter_ccc`, `fir_filter_ccf`, `fir_filter_fff` — scroll through the [GNU
Radio Sphinx documentation](https://www.gnuradio.org/doc/sphinx-v3.7.9.2/) for
more examples.

On the other hand, LuaRadio blocks can support multiple type signatures. For
example, the [`MultiplyBlock`](0.reference-manual.md#multiplyblock) and
[`FIRFilterBlock`](0.reference-manual.md#firfilterblock) can accept both
complex-valued and real-valued input/output types, and the framework will
automatically differentiate the correct type signature based on the block's
connections in a flow graph. This helps with code reuse and consolidation, as
well as ease of use in defining flow graphs.

### Sample Rate Propagation

GNU Radio requires users to manually specify the sample rate of every block
that needs it. This can be alleviated by defining common sample rates in
variables that are shared among blocks, but ultimately is still error prone.
Some blocks require normalized parameters (e.g. angular frequency), which also
obligates users to compute the correct normalization.

LuaRadio automatically propagates sample rates between all blocks. This
reduces the amount of configuration needed by blocks, and any frequency
parameters that are required can be specified in units of hertz. It also allows
for a suite of convenience blocks that automatically perform sample rate
dependent initialization, e.g. the
[`LowpassFilterBlock`](0.reference-manual.md#lowpassfilterblock), whose
parameters are just number of taps and a cutoff frequency in hertz.

### Embeddability

LuaRadio's small footprint allows it to be wholly embeddable in applications.
See the [Embedding LuaRadio](4.embedding-luaradio.md) guide for more details
and examples.

## Performance

Generally speaking, LuaRadio performance is on the same order as GNU Radio
performance. In computationally expensive blocks, like filters, LuaRadio has
matching or slightly better performance to GNU Radio. In other cases, LuaRadio
may range from 30% to 80% the performance of GNU Radio, but this is typically
for blocks that are already in the very high throughput territory, e.g.
hundreds to thousands of megasamples per second on an Intel i5 CPU. In a few
cases, LuaRadio performance exceeds GNU Radio performance.

While LuaRadio performance is more than adequate for real-time SDR projects,
GNU Radio still performs better than LuaRadio overall. If performance is your
priority, then you should use GNU Radio.

See the [Benchmarks](https://luaradio.io/benchmarks.html) page for empirical
benchmarks on both LuaRadio and GNU Radio across several platforms.

Future infrastructure improvements and additional library acceleration will
help LuaRadio close the performance gap.

## Maturity

GNU Radio has a larger community, more blocks, more hardware support, and a
more mature codebase.

LuaRadio does not have a graphical flow graph editing tool, like the GNU Radio
Companion (GRC).

GNU Radio currently has some more efficient signal processing blocks, like
polyphase filters, or filters with integrated translation and decimation, but
these can be added to LuaRadio as needed.
