#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <assert.h>

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include "config.h"
#define LOCALIZER_INTERNAL

/**
 * dataset for the database
 */

typedef struct {
	unsigned long netmask; /**< netmask */
	unsigned long hostmask; /**< hostmask */
	int city; /**< index of city */
	int provider; /**< index of the provider */
	int country; /**< index of the country */
	int province; /**< index of the province */
	double distance; /**< estimated distance to the nearest access-point */
	double latitude; /**< latitude */
	double longitude; /**< longitude */
} ldata;


typedef struct {
	char *data;
	int data_size;
	int *offset;
	int off_size;
	int off_pos;
} ldata_strings;

/**
 * internal datastructure for the localizer
 * 
 * is used the get the string based on a index
 * 
 * ex.:
 *   fprintf(stderr, "city: %%s\\n", STRING(l, l->ndat[0]->city));
 * 
 * returns the string for the first entry in the database
 * 
 * use it to get the string for
 * - city, provider, country and province
 */

#define STRING(l,n)   (l->strings.data + l->strings.offset[n])

typedef struct {
	ldata **ndat;  /**< unsorted array of datasets */
	int ndat_size; /**< physical size of the database */
	int ndat_pos;  /**< current position of the last dataset */
	int *ndat_vector; /**< sorted vector to the datasets ndata */
	
	/* cites */
	ldata_strings strings;
	
	/* shm */
	int shm_size; /**< calculated size of the share-memory segment */
	time_t file_mtime; /**< last modification time for the database (.csv) */
	off_t file_size; /**< original size of the database (.cvs) */
	void *shm; /**< set if we it is taken from shm */
} localizer;

#include "localizer.h"

#include "mio.h"


/**
 * ID of the Shared Memory Segment
 */
#define LOC_SHM_ID (key_t)((0x1234) + getuid())

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif

/**
 * convert a IP in dot-notation into the internal representation (long int)
 * 
 * localizer_search() requires a 
 * 
 * @param ip IP in dot-notation (e.g. 192.168.23.3)
 * @return IP as long it, 0 on error
 * @see localizer_search
 */

long localizer_ip2int(const char *ip ) {
	char *p2, *p3, *p4;
	long ret = 0;
	
	char p1[16];
	
	strncpy(p1, ip, sizeof(p1) - 1);
	p1[sizeof(p1) - 1] = '\0';
	
	if ((p2 = strchr(p1, '.'))) {
		*(p2++) = '\0';
		if ((p3 = strchr(p2, '.'))) {
			*(p3++) = '\0';
			if ((p4 = strchr(p3, '.'))) {
				*(p4++) = '\0';
	
				ret =	strtoul(p1, NULL, 10) << 24 |
					strtoul(p2, NULL, 10) << 16 |
					strtoul(p3, NULL, 10) << 8 |
					strtoul(p4, NULL, 10) ;
			}
		}
	}
	
	return ret;
}


int localizer_insert_string(localizer *l, const char *str) {
	int ndx = -1;
	int i, pos = 0;
	static int *sort = NULL;
	static int next_max = 1;
	
	/* get country index */
	for (i = pos = next_max / 2; ; i /= 2) {
		int cmp;
		if (pos < 0) {
			pos += i;
		} else if (pos >= l->strings.off_pos) {
			pos -= i;
		} else {
			cmp = strcmp(str, l->strings.data + l->strings.offset[sort[pos]]);
			
			if (cmp == 0) {
				/* found */
				ndx = sort[pos];
				break;
			} else if (cmp < 0) {
				pos -= i;
			} else {
				pos += i;
			}
		}
		if (i == 0) break;
	}

	/* not found */
	if (ndx == -1) {
		int sl = strlen(str);
		int off;
		
		/* init array */
		if (l->strings.off_size == 0) {
			l->strings.off_size = 256;
			l->strings.off_pos = 0;
			l->strings.data_size = 256;
			l->strings.data = malloc(l->strings.data_size * sizeof(char));
			*(l->strings.data) = '\0';
			
			l->strings.offset = malloc(l->strings.off_size * sizeof(int));
			for (i = 0; i < l->strings.off_size; i++) {
				l->strings.offset[i] = 0;
			}
			
			sort = malloc(l->strings.off_size * sizeof(int));
		}
		
		/* resize index */
		if (l->strings.off_size == l->strings.off_pos + 1) {
			l->strings.off_size += 256;
			l->strings.offset = realloc(l->strings.offset, l->strings.off_size * sizeof(int));
			
			for (i = l->strings.off_pos + 1; i < l->strings.off_size; i++) {
				l->strings.offset[i] = 0;
			}
			
			sort = realloc(sort, l->strings.off_size * sizeof(int));
		}
		
		/* resize datastorage */
		off = l->strings.offset[l->strings.off_pos];
		
		if (off + sl + 1 > l->strings.data_size) {
			l->strings.data_size += sl > 256 ? sl : 256;
			l->strings.data = realloc(l->strings.data, l->strings.data_size * sizeof(char));
		}
		
		/* add the string to the storage */
		strcpy(l->strings.data + off, str);
		
		ndx = l->strings.off_pos++;
		
		/* set beginning of next string */
		l->strings.offset[l->strings.off_pos] = off + sl + 1;
		
		/**
		 * insert the ndx into the sorted vector array
		 * 
		 * - search new position
		 */
		
		if (pos != ndx &&
		    ((pos < 0) || 
		     strcmp(str, l->strings.data + l->strings.offset[sort[pos]]) > 0)) {
			pos++;
		} 
			
		/* move everything on step to the right */
		for (i = l->strings.off_pos - 1; i > pos; i--) {
			sort[i] = sort[i - 1];
		}
		
		/* insert */
		sort[pos] = l->strings.off_pos - 1;
			
		if (next_max == ndx) {
			next_max *= 2;
		}
	}
	return ndx;
}

/**
 * insert a record into in the database
 * 
 * used by localizer_read()
 * 
 * @param l localizer hanle
 * @param net netmask 
 * @param host hostmask
 * @param city corresponding city
 * @param provider name of the provider
 * @return 0 on success
 * @see localizer_read
 */

int localizer_insert(localizer *l, unsigned long net, unsigned long host, 
		     const char *city, 
		     const char *province, 
		     const char *country, 
		     const char *provider,
		     double distance,
		     double latitude,
		     double longitude) {
	int i;
	int country_ndx = -1, provider_ndx = -1, province_ndx = -1, city_ndx = -1;
	
	if (!l) return -1;
	
	/* allocate the first chunk of memory */
	if (l->ndat == NULL) {
		l->ndat_size = 256;
		l->ndat = malloc(sizeof(ldata *) * l->ndat_size);
		
		for (i = 0; i < l->ndat_size; i++) {
			l->ndat[i] = malloc(sizeof(ldata));
			l->ndat[i]->city = -1;
			l->ndat[i]->provider = -1;
			l->ndat[i]->province = -1;
			l->ndat[i]->country = -1;
			l->ndat[i]->hostmask = 0;
			l->ndat[i]->netmask = 0;
			l->ndat[i]->distance = 0;
			l->ndat[i]->latitude = 0;
			l->ndat[i]->longitude = 0;
			
		}
	}
	
	
	/* resize if neccessary */
	if (l->ndat_pos == l->ndat_size) {
		l->ndat_size += 256;
		l->ndat = realloc(l->ndat, sizeof(ldata *) * l->ndat_size);
		
		for (i = l->ndat_pos; i < l->ndat_size; i++) {
			l->ndat[i] = malloc(sizeof(ldata));
			l->ndat[i]->city = -1;
			l->ndat[i]->provider = -1;
			l->ndat[i]->province = -1;
			l->ndat[i]->country = -1;
			l->ndat[i]->hostmask = 0;
			l->ndat[i]->netmask = 0;
			l->ndat[i]->distance = 0;
			l->ndat[i]->latitude = 0;
			l->ndat[i]->longitude = 0;
		}
	}
	
	country_ndx  = localizer_insert_string(l, country);
	provider_ndx = localizer_insert_string(l, provider);
	province_ndx = localizer_insert_string(l, province);
	city_ndx     = localizer_insert_string(l, city);
	
	/* copy the data */
	l->ndat[l->ndat_pos]->netmask = net;
	l->ndat[l->ndat_pos]->hostmask = host;
	l->ndat[l->ndat_pos]->city = city_ndx;
	l->ndat[l->ndat_pos]->provider = provider_ndx;
	l->ndat[l->ndat_pos]->province = province_ndx;
	l->ndat[l->ndat_pos]->country = country_ndx;
	l->ndat[l->ndat_pos]->latitude = latitude;
	l->ndat[l->ndat_pos]->longitude = longitude;
	l->ndat[l->ndat_pos]->distance = distance;
	
	l->ndat_pos++;
	
	return 0;
}

/**
 * sorts the internal datastructures for faster access
 * 
 * After reading in the dataset the dataset is unsorted. localizer_msort 
 * uses a indirect merge-sort to sort the internal structure
 * 
 * @param loc localizer handle
 * @param a pointer to the datastructure
 * @param b pointer to the work-space
 * @param l left position in the internal structure
 * @param r right position in the internal structure
 */
void localizer_msort(localizer *loc, int *a, int *b, int l, int r) {
	int i, j, k, m;
	
	if (r > l) {
		m = (r + l) / 2;
		localizer_msort(loc, a, b, l, m);
		localizer_msort(loc, a, b, m+1, r);
		for (i = m + 1; i > l; i--) b[i-1] = a[i-1];
		for (j = m; j < r; j++) b[r+m-j] = a[j+1];
		for (k = l; k <= r; k++) 
			a[k] = (loc->ndat[b[i]]->netmask < loc->ndat[b[j]]->netmask) ? b[i++] : b[j--];
	}
}

/**
 * generate the internal vectors to the datastructures
 * 
 * @param l localizer handle
 * @return 0 on success
 */

int localizer_genvector(localizer *l) {
	int i;
	int *ndat_vector_b;
	
	/* prepare the buffers */
	l->ndat_vector = malloc(sizeof(int) * l->ndat_pos);
	ndat_vector_b = malloc(sizeof(int) * l->ndat_pos);

	for (i = 0; i < l->ndat_pos; i++) 
		l->ndat_vector[i] = i;
	
	localizer_msort(l, l->ndat_vector, ndat_vector_b, 0, l->ndat_pos - 1);
	
	free(ndat_vector_b);
	
	return 0;
}

/**
 * lookup a ip in the database
 * 
 * @param loc localizer handle
 * @param ip IP in long int represenation
 * @see localizer_ip2int
 */

int localizer_search (const localizer *loc, unsigned long ip, l_data_export *data) {
	int l, r;

	if (!loc) return -1;
	if (!loc->ndat) return -1;
	
	r = loc->ndat_pos - 1;
	l = 0;
	
	while (l != r) {
		int m = l + ((r - l) / 2);
		unsigned long hm, nm, im;
		int ndx = loc->ndat_vector[m];
		
		hm = loc->ndat[ndx]->hostmask;
		nm = loc->ndat[ndx]->netmask & hm;
		im = ip & hm;
		
		if (im == nm) {
			l = m;
			break;
		} else if (im < nm) {
			r = m;
		} else {
			l = m + 1;
		}
		/* printf("l: %d, r: %d\n", l, r); */
	}
	
	/*printf(" %lx ?= %lx \n", 
	       (ip & loc->ndat[loc->ndat_vector[l]]->hostmask), 
	       loc->ndat[loc->ndat_vector[l]]->netmask);
	*/
	
	if ((loc->ndat[loc->ndat_vector[l]]->netmask & loc->ndat[loc->ndat_vector[l]]->hostmask) == 
	    (ip & loc->ndat[loc->ndat_vector[l]]->hostmask)) {
		int ndx = loc->ndat_vector[l];
		
#define COPY(x) \
	data->x = loc->ndat[ndx]->x;
		COPY(distance);
		COPY(latitude);
		COPY(longitude);
		COPY(netmask);
		COPY(hostmask);
#undef COPY
#define COPY(x) \
	data->x = STRING(loc, loc->ndat[ndx]->x);
		COPY(city);
		COPY(province);
		COPY(provider);
		COPY(country);
#undef COPY
		
		return 0;
	} else {
		return -1;
	}
}

int localizer_status (const localizer *loc, l_status_export *data) {
	if (!loc) return -1;
	if (!loc->ndat) return -1;
	
	data->num_elements = loc->ndat_pos;
	
	data->shm_enabled  = loc->shm != NULL ? 1 : 0;
	data->shm_size     = loc->shm_size;
	data->shm_last_mod = loc->file_mtime;
	data->shm_db_size  = loc->file_size;
	
	return 0;
}

void localizer_dump_mem(void *void_p, long size) {
	long i;
	unsigned char *p = void_p;
	const int SW = 16;
	
	fprintf(stderr, "mem-dump  ");
	for (i = 0; i < SW; i++) {
		fprintf(stderr, " %02lx", i);
	}
	fprintf(stderr, "\n");
	for (i = 0; i < size; i++) {
		/* start of line */
		if (i % SW == 0) {
			fprintf(stderr, "%10p", p + i);
		}
		fprintf(stderr, " %02x", *(p + i));
		/* end of line */
		if (i % SW == SW - 1) {
			int j;
			fprintf(stderr, " "); 
			for (j = 0; j < SW; j++) {
				fprintf(stderr, 
					"%c",
					*(p + i + 1 - SW + j) < 32 ||
					*(p + i + 1 - SW + j) > 127 
					? '.'
					: *(p + i + 1 - SW + j)
					);
			}
			fprintf(stderr, "\n");
		}
	}
	
	fprintf(stderr, "\n");
}

void localizer_dump_core(const localizer *l) {
	fprintf(stderr, 
		"l = %p\n"
		"  ndat        = %p\n"
		"  ndat_size   = 0x%08x (%d)\n"
		"  ndat_pos    = 0x%08x (%d)\n"
		"  ndat_vector = %p\n"
		"  shm_size    = 0x%08x (%d)\n",
		l,
		l->ndat,
		l->ndat_size, l->ndat_size,
		l->ndat_pos, l->ndat_pos,
		l->ndat_vector,
		l->shm_size, l->shm_size);
}

int localizer_read_shm(localizer *l, const char *filename) {
	int shm_id;
	localizer *nl = NULL;
	int size, i;
	struct stat st;
	
	if (l == NULL) return -1;
	
	
	if (filename && stat(filename, &st)) {
		fprintf(stderr, "LOC: %s.%d: %s\n", __FILE__, __LINE__, strerror(errno));
		return -1;
	}
	
	/* get the data from the shared memory segment */
	if ((shm_id = shmget(LOC_SHM_ID, 0, 0400)) != -1) {
		struct shmid_ds buf;
		
		nl = (localizer *) shmat(shm_id, 0, SHM_RDONLY);
		if (shmctl(shm_id, IPC_STAT, &buf)) {
			fprintf(stderr, "SHM: %s.%d: %s\n", __FILE__, __LINE__, strerror(errno));
		}
		
		
		/* Check if everything is ok */
		if (nl->shm != (void *)1) {
			/* "houston, we've got a problem"
			 * 
			 * The LOC_SHM_ID is used by another process 
			 */
			
			fprintf(stderr, "SHM: in_shm != 1\n");
			
			if (shmdt(nl)) {
				fprintf(stderr, "SHM: %s.%d: %s\n", __FILE__, __LINE__, strerror(errno));
			}
			
			nl = NULL;
		} else if (nl->shm_size != buf.shm_segsz) {
			/* registered size != segment size */
			fprintf(stderr, "SHM: wrong size\n");
			
			/* detach and destroy the segment */
			if (shmdt(nl)) {
				fprintf(stderr, "SHM: %s.%d: %s\n", __FILE__, __LINE__, strerror(errno));
			}
			
			if (shmctl(shm_id, IPC_RMID, NULL)) {
				fprintf(stderr, "SHM: %s.%d: %s\n", __FILE__, __LINE__, strerror(errno));
			}
			
			l = NULL;
		} else if (filename &&
			   (st.st_mtime != nl->file_mtime ||
			    st.st_size != nl->file_size)) {
			/* the shm is not in sync with DB */
			
			/* detach and destroy the segment */
			if (shmdt(nl)) {
				fprintf(stderr, "SHM: %s.%d: %s\n", __FILE__, __LINE__, strerror(errno));
			}
			
			if (shmctl(shm_id, IPC_RMID, NULL)) {
				fprintf(stderr, "SHM: %s.%d: %s\n", __FILE__, __LINE__, strerror(errno));
			}
			
			nl = NULL;
		}
	}
	
	/* ok, let's fall back to something usefull */

	if (nl) {
		/* pointer arithmetics */
		unsigned char *void_nl = (void *)nl;
		
		/* copy ints */
		l->file_mtime = nl->file_mtime;
		l->file_size = nl->file_size;
		l->shm_size = nl->shm_size;
		l->ndat_size = nl->ndat_size;
		l->ndat_pos = nl->ndat_pos;
		
		l->strings.data_size = nl->strings.data_size;
		l->strings.off_size = nl->strings.off_size;
		l->strings.off_pos = nl->strings.off_pos;
		
		/* recalculated the offsets */
		l->ndat_vector = (int *)(void_nl + (int)nl->ndat_vector);
		l->ndat = malloc(l->ndat_size * sizeof(ldata));
		for (i = 0; i < l->ndat_size; i++) {
			ldata **ld = (void *)(void_nl + (int)nl->ndat);
			l->ndat[i] = (void *)(void_nl + (int)ld[i]);
		}
		l->strings.data = (void *)(void_nl + (int)nl->strings.data);
		l->strings.offset = (void *)(void_nl + (int)nl->strings.offset);
		
		/* put shm-pointer into the localizer handle*/
		l->shm = nl;
	} else if (filename) {
		mfile mf;
		buffer *buf = buffer_init();
		
		if (-1 == mopen(&mf, filename)) {
			fprintf(stderr, "LOC: %s.%d: %s\n", __FILE__, __LINE__, strerror(errno));
			return -1;
		}

		l->file_size = st.st_size;
		l->file_mtime = st.st_mtime;
		
		while (mgets(&mf, buf) != NULL) {
			/* find the , */
			char *netmask, *hostmask, *city, *provider,
				*country, *province;
			
			netmask = buf->ptr;
			
			if (buf->ptr[buf->used-2] == '\n')
				buf->ptr[buf->used-2] = '\0';
			
			if ((hostmask = strchr(netmask, ',')) && 
			    (city = strchr(hostmask + 1, ',')) &&
			    (province = strchr(city + 1, ',')) &&
			    (country = strchr(province + 1, ',')) &&
			    (provider = strchr(country + 1, ',')) 
			    ) {
				long net, host;
				char *distance, *latitude, *longitude;
				
				*hostmask++ = '\0';
				*city++ = '\0';
				*provider++ = '\0';
				*country++ = '\0';
				*province++ = '\0';
				
				net = localizer_ip2int(netmask);
				host = localizer_ip2int(hostmask);
				
				if ((distance = strchr(provider, ',')) && 
				    (latitude = strchr(distance + 1, ',')) &&
				    (longitude = strchr(latitude + 1, ',')) 
				    ) {
					double dist, lat, lng;
					
					*distance++ = '\0';
					*latitude++ = '\0';
					*longitude++ = '\0';
					
					dist = strtod(distance, NULL);
					lat = strtod(latitude, NULL);
					lng = strtod(longitude, NULL);
					
					localizer_insert(l, net, host, 
							 city, 
							 province,
							 country,
							 provider,
							 dist,
							 lat,
							 lng);
					
				} else {
					localizer_insert(l, net, host, 
							 city, 
							 province,
							 country,
							 provider,
							 0,
							 0,
							 0);
				}
			} else {
				return -1;
			}
		}
		mclose(&mf);
		buffer_free(buf);
	
		/* generate the vectors */
		localizer_genvector(l);
		
		/* calculated the size of the shm-segment */
		size = sizeof(localizer) + /* the localizer struct */
			(l->ndat_size * sizeof(ldata *) ) + /* size of l->ndat */
			(l->ndat_size * sizeof(ldata) ) + /* size of the elements of l->ndat */
			(l->ndat_pos * sizeof(int) ) + /* size of l->ndat_vector */
			l->strings.off_size * sizeof(int) +
			l->strings.data_size
			;
		l->shm_size = size;
		
		
#define ARGS() \
	if (shm_p > size) fprintf(stderr, "SHM: %s.%d: smp_p > size !!\n", __FILE__, __LINE__);
		
		if ((shm_id = shmget(LOC_SHM_ID, l->shm_size, 0644 | IPC_CREAT | IPC_EXCL)) != -1) {
			int shm_p = 0;
			nl = (localizer *) shmat(shm_id, 0, 0);
		
			if (nl) {
				int ndat_offset = 0;
				/* pointer arithmetics */
				unsigned char *void_nl = (void *)nl;
				
				/* setup the shm */
				nl->file_mtime = l->file_mtime;
				nl->file_size = l->file_size;
				nl->shm_size = l->shm_size;
				nl->ndat_size = l->ndat_size;
				nl->ndat_pos = l->ndat_pos;
				
				/* the first bytes are for the localizer struct */
				shm_p += sizeof(localizer);
				
				/* copy the vectors */
				nl->ndat_vector = (int *)(void_nl + shm_p);
				memcpy(nl->ndat_vector, l->ndat_vector, sizeof(int) * l->ndat_pos);
				/* set offset */
				nl->ndat_vector = (void *)shm_p;
				shm_p += sizeof(int) * l->ndat_pos;
			
				/* copy the data itself */
				nl->ndat = (void *)(void_nl + shm_p);
				ndat_offset = shm_p;
				shm_p += (l->ndat_size * sizeof(ldata *));
				
				ARGS();
				
				for (i = 0; i < l->ndat_size; i++) {
					nl->ndat[i] = (void *)(void_nl + shm_p);
					memcpy(nl->ndat[i], l->ndat[i], sizeof(ldata));
					nl->ndat[i] = (void *)shm_p;
					shm_p += sizeof(ldata);
					ARGS();
				}
				nl->ndat = (void *)ndat_offset;
				
				nl->strings.data_size = l->strings.data_size;
				nl->strings.off_size = l->strings.off_size;
				nl->strings.off_pos = l->strings.off_pos;
				
				/* copy */
				nl->strings.data = (void *)(void_nl + shm_p);
				memcpy(nl->strings.data, l->strings.data, l->strings.data_size);
				nl->strings.data = (void *)shm_p;
				shm_p += l->strings.data_size;
				
				nl->strings.offset = (void *)(void_nl + shm_p);
				memcpy(nl->strings.offset, l->strings.offset, l->strings.off_size * sizeof(int));
				nl->strings.offset = (void *)shm_p;
				shm_p += l->strings.off_size * sizeof(int);
				
				/* everything is set up*/
				ARGS();
				
				/* check if the estimated size of the shared mem-segment is really big-enough */
				if (shm_p != size) {
					/* report the error */
					fprintf(stderr, "SHM: shm size != est. size: %d != %d\n", shm_p, size);
					/* detach */
					shmdt(nl);
					/* remove the segment */
					shmctl(shm_id, IPC_RMID, NULL);
				} else {
					nl->shm = (void *)1;
					
					shmdt(nl);
				}
			}
		} else {
			fprintf(stderr, "SHM: can't get shm: %s\n", strerror(errno));
		}
	} else {
		return -1;
	}
#undef ARGS
#if 0	
	localizer_dump_mem(l, sizeof(localizer));
	localizer_dump_core(l);
#endif		 
	return 0;
}

/**
 * read the CSV database from the file named <filename>
 * 
 * if filename is NULL stdin is used
 * 
 * @param l localizer handle
 * @param filename name of the database
 * 
 */

int localizer_read(localizer *l, const char *filename) {
	mfile mf;
	buffer *buf = buffer_init();
	
	if (!l) return -1;
	
	if (l->shm) {
		return 0;
	}
	
	if (-1 == mopen(&mf, filename)) {
		return -1;
	}
	
	while (mgets(&mf, buf) != NULL) {
		/* find the , */
		char *netmask, *hostmask, *city, *provider,
			*country, *province;
		
		netmask = buf->ptr;
		
		if (buf->ptr[buf->used-2] == '\n')
			buf->ptr[buf->used-2] = '\0';
			
		if ((hostmask = strchr(netmask, ',')) && 
		    (city = strchr(hostmask + 1, ',')) &&
		    (province = strchr(city + 1, ',')) &&
		    (country = strchr(province + 1, ',')) &&
		    (provider = strchr(country + 1, ',')) 
		    ) {
			long net, host;
			char *distance, *latitude, *longitude;
			
			*hostmask++ = '\0';
			*city++ = '\0';
			*provider++ = '\0';
			*country++ = '\0';
			*province++ = '\0';
			
			net = localizer_ip2int(netmask);
			host = localizer_ip2int(hostmask);
			
			if ((distance = strchr(provider, ',')) && 
			    (latitude = strchr(distance + 1, ',')) &&
			    (longitude = strchr(latitude + 1, ',')) 
			    ) {
				double dist, lat, lng;
				
				*distance++ = '\0';
				*latitude++ = '\0';
				*longitude++ = '\0';
				
				dist = strtod(distance, NULL);
				lat = strtod(latitude, NULL);
				lng = strtod(longitude, NULL);
				
				localizer_insert(l, net, host, 
						 city, 
						 province,
						 country,
						 provider,
						 dist,
						 lat,
						 lng);
				
			} else {
				localizer_insert(l, net, host, 
						 city, 
						 province,
						 country,
						 provider,
						 0,
						 0,
						 0);
			}
		} else {
			return -1;
		}
	}
	mclose(&mf);
	buffer_free(buf);
	
	/* generate the vectors */
	localizer_genvector(l);
#if 0	
	localizer_dump_mem(l, sizeof(localizer));
	localizer_dump_core(l);
#endif		 
	return 0;
}

/**
 * create a localizer handle
 * 
 * @return localizer handle
 */
localizer * localizer_init() {
	localizer *l = NULL;
	
	/* ok, let's fall back to something usefull */
	if (l == NULL) {
		l = malloc(sizeof(localizer));
	
		l->ndat = NULL;
		l->ndat_pos = 0;
		l->ndat_size = 0;
		l->ndat_vector = NULL;
		
		l->strings.off_size = 0;
		l->strings.off_pos = 0;
		l->strings.offset = NULL;
		l->strings.data = NULL;
		l->strings.data_size = 0;
		
		
		l->shm = NULL;
		l->shm_size = 0;
		l->file_mtime = 0;
		l->file_size = 0;
	}
	
	return l;
}

/**
 * destroy a localizer handle
 * 
 * @param l localizer handle
 */
void localizer_free(localizer * l) {
	int i;
	
	if (!l) return;
	
	/* don't remove the content of the shared memory */
	if (l->shm) {
		shmdt(l->shm);
		free(l->ndat);
		free(l);
	} else {
	
		for (i = 0; i < l->ndat_size; i++) {
			if (l->ndat[i]) free(l->ndat[i]);
		}
		
		if (l->ndat) free(l->ndat);
		if (l->ndat_vector) free(l->ndat_vector);
		if (l->strings.data) free(l->strings.data);
		if (l->strings.offset) free(l->strings.offset);
		free(l);
	}
}
