//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
// UNSUPPORTED: c++03, c++11, c++14, c++17

// <chrono>
// class year_month;

// constexpr bool operator==(const year_month& x, const year_month& y) noexcept;
//   Returns: x.year() == y.year() && x.month() == y.month().
//
// constexpr bool operator< (const year_month& x, const year_month& y) noexcept;
//   Returns:
//      If x.year() < y.year() returns true.
//      Otherwise, if x.year() > y.year() returns false.
//      Otherwise, returns x.month() < y.month().

#include <chrono>
#include <type_traits>
#include <cassert>

#include "test_macros.h"
#include "test_comparisons.h"

int main(int, char**)
{
    using year       = std::chrono::year;
    using month      = std::chrono::month;
    using year_month = std::chrono::year_month;

    AssertComparisonsAreNoexcept<year_month>();
    AssertComparisonsReturnBool<year_month>();

    static_assert( testComparisons(
        year_month{year{1234}, std::chrono::January},
        year_month{year{1234}, std::chrono::January},
        true, false), "");

    static_assert( testComparisons(
        year_month{year{1234}, std::chrono::January},
        year_month{year{1234}, std::chrono::February},
        false, true), "");

    static_assert( testComparisons(
        year_month{year{1234}, std::chrono::January},
        year_month{year{1235}, std::chrono::January},
        false, true), "");

    //  same year, different months
    for (unsigned i = 1; i < 12; ++i)
        for (unsigned j = 1; j < 12; ++j)
            assert((testComparisons(
                year_month{year{1234}, month{i}},
                year_month{year{1234}, month{j}},
                i == j, i < j )));

    //  same month, different years
    for (int i = 1000; i < 2000; ++i)
        for (int j = 1000; j < 2000; ++j)
        assert((testComparisons(
            year_month{year{i}, std::chrono::January},
            year_month{year{j}, std::chrono::January},
            i == j, i < j )));

    return 0;
}
