# Copyright (C) 2004-2012 Bastian Kleineidam
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
"""
Test cgi form routines.
"""
import unittest
import urllib.parse
from io import BytesIO
from wsgiref.util import setup_testing_defaults
from linkcheck.lc_cgi import checkform, checklink, LCFormError, application
from linkcheck.strformat import limit


class TestWsgi(unittest.TestCase):
    """Test wsgi application."""

    def test_form_valid_url(self):
        # Check url validity.
        env = dict()
        form = dict(url="http://www.example.com/", level="1")
        checkform(form, env)

    def test_form_empty_url(self):
        # Check with empty url.
        env = dict()
        form = dict(url="", level="0")
        self.assertRaises(LCFormError, checkform, form, env)

    def test_form_default_url(self):
        # Check with default url.
        env = dict()
        form = dict(url="http://", level="0")
        self.assertRaises(LCFormError, checkform, form, env)

    def test_form_invalid_url(self):
        # Check url (in)validity.
        env = dict()
        form = dict(url="http://www.foo bar/", level="0")
        self.assertRaises(LCFormError, checkform, form, env)

    def test_checklink(self):
        form = dict(url="http://www.example.com/", level="0")
        checklink(form, {})

    def test_application(self):
        form = dict(url="http://www.example.com/", level="0")
        formdata = urllib.parse.urlencode(form)
        formdata = formdata.encode("ascii")
        environ = {"wsgi.input": BytesIO(formdata)}
        setup_testing_defaults(environ)
        test_response = b""
        test_headers = [None]
        test_status = [None]

        def start_response(status, headers):
            test_status[0] = status
            test_headers[0] = headers

        for str_data in application(environ, start_response):
            if not isinstance(str_data, bytes):
                err = "answer is not a byte string: %r" % limit(str_data, 30)
                self.assertTrue(False, err)
            test_response += str_data
        self.assertEqual(test_status[0], "200 OK")
        self.assertTrue(b"Generated by LinkChecker" in test_response)
