#include <xdkwrl/tools/transformator.h>

#include <stdio.h>
#include <math.h>
#include <malloc.h>

#include <iomanip>
#include <stdexcept>

using namespace std;
using namespace wrl;

// INTERNALS DETAILS: the matrix are stored in this way.
//
//       j=0 j=1 j=2 j=3
//      
// i=0 |  0   1   2   3  
// i=1 |  4   5   6   7  
// i=2 |  8   9  10  11
// i=3 | 12  13  14  15
//
// m_[i*4+j] = m[i][j]
//
// the multiplication by a vector is done *on left* that is you have T=M*V
// as represented below:
//
//         |     | 
//         |  M  | 
//         |     |
//
// [  V  ] [  T  ]
//
// The multiplication of two matrices is done on left so R=A*B refers to the
// multiplication represented below.
//         |     | 
//         |  A  | 
//         |     |
//
// |     | |     |
// |  B  | |  R  |
// |     | |     |
//
// Here is the correponding code to multiply a vector
//
//    X = x*m[0] + y*m[4] + z*m[ 8] + w*m[12];
//    Y = x*m[1] + y*m[5] + z*m[ 9] + w*m[13];
//    Z = x*m[2] + y*m[6] + z*m[10] + w*m[14];
//    W = x*m[3] + y*m[7] + z*m[11] + w*m[15];

// 

namespace
{
  /*
   * Multiply matrix a and put results in out. In other words out =
   * a*b. The same matrix than a or b can be used for out.  
   */
  void
  multiply(float* out,const float* a,const float* b)
  {
    float r[16];
    for (unsigned short i=0; i<4; ++i)
    {
      for (unsigned short j=0; j<4; ++j)
      {
	float sum = 0.0f;
	for (unsigned short k=0; k<4; ++k)
	{
	  sum += b[i*4+k] * a[k*4+j];
	}
	r[i*4+j] = sum;
      }
    }
    for (unsigned short i=0;i<16;++i)
    {
      out[i] = r[i];
    }
  }
  /*
   * Load identity atrix in m.
   */
  inline void
  loadIdentity(float* m)
  {
    memset(m,0,16*sizeof(float));
    m[0] = m[5] = m[10] = m[15] = 1.0f;
  }
  inline void
  accumulate(const float temp,float& pos,float& neg)
  {
    if (temp >= 0.0f)
    {
      pos += temp;
    }
    else
    {
      neg += temp;
    }
  }
  /*!
   * Gets the inverse of an affine matrix
   */
  void
  invertAffineMatrix(const float *in,float *out)
  {
    float det_l,pos=0,neg=0,temp;
    temp = in[0] * in[5] * in[10];
    accumulate(temp,pos,neg);
    temp = in[1] * in[6] * in[8];
    accumulate(temp,pos,neg);
    temp = in[2] * in[4] * in[9];
    accumulate(temp,pos,neg);
    temp = -in[2] * in[5] * in[8];
    accumulate(temp,pos,neg);
    temp = -in[1] * in[4] * in[10];
    accumulate(temp,pos,neg);
    temp = -in[0] * in[6] * in[9];
    accumulate(temp,pos,neg);
    det_l = (pos + neg);
    if (fabs(det_l/(pos-neg)) < 1e-15)
    {
      throw runtime_error("matrix is not invertable");
    }
    det_l = 1.0f/det_l;
    out[0] = (in[5]*in[10] - in[6]*in[9]) * det_l;
    out[4] = (in[6]*in[8] - in[4]*in[10]) * det_l;
    out[8] = (in[4]*in[9] - in[5]*in[8]) * det_l;
    out[1] = (in[2]*in[9] - in[1]*in[10]) * det_l;
    out[5] = (in[0]*in[10] - in[2]*in[8]) * det_l;
    out[9] = (in[1]*in[8] - in[0]*in[9]) * det_l;
    out[2] = (in[1]*in[6] - in[2]*in[5]) * det_l;
    out[6] = (in[2]*in[4] - in[0]*in[6]) * det_l;
    out[10] = (in[0]*in[5] - in[1]*in[4]) * det_l;
    out[12] = -(in[12]*out[0] + in[13]*out[4] + in[14]*out[8]);
    out[13] = -(in[12]*out[1] + in[13]*out[5] + in[14]*out[9]);
    out[14] = -(in[12]*out[2] + in[13]*out[6] + in[14]*out[10]);
    out[3] = out[7] = out[11] = 0.0;
    out[15] = 1.0;
  }
  void
  transpose(const float *in,float *out)
  {
    out[0] = in[0];
    out[1] = in[4];
    out[2] = in[8];
    out[3] = in[12];
    out[4] = in[1];
    out[5] = in[5];
    out[6] = in[9];
    out[7] = in[13];
    out[8] = in[2];
    out[9] = in[6];
    out[10] = in[10];
    out[11] = in[14];
    out[12] = in[3];
    out[13] = in[7];
    out[14] = in[11];
    out[15] = in[15];
  }
}
//***************************************************************
// Implementation of Transformator
//***************************************************************
/*! \class wrl::Transformator
 * \ingroup tools
 *
 * Represent a transformation thru a 4x4 matrix.  The matrix can be
 * incrementally built using members function to add
 * translation/scaling/rotations.  The order of construction is the same
 * than for openGL.  So last call is applicated first to point when
 * transformed. In the example below, M et N are transformed in the same
 * way (but notice that t2 reverses its calls). 
 * \code
  float M[3] = { 0.5,1.2,0.7 };
  float N[3] = { 0.5,1.2,0.7 };
  
  Transformator t0;
  Transformator t1;
  t0.rotate(0.3f,1.0f,1.0f,1.0f);
  t1.scale(2.0f,-1.0f,3.0f);
  
  t0.transform(M);
  t1.transform(M);

  Transformator t2;
  t2.scale(2.0f,-1.0f,3.0f);
  t2.rotate(0.3f,1.0f,1.0f,1.0f);
  t2.transform(N);
 * \endcode
 */


/*! 
 * Make an indentity transformator
 */
Transformator::Transformator()
{
  loadIdentity();
}
/*! 
 * Reset the matrix to identity transformation.
 */
void
Transformator::loadIdentity()
{
  ::loadIdentity(m_);
}
/*! 
 * Loads the \p m matrix as transformation. In m, the first brackets
 * indicates the lines, the second one the column of the matrix.
 */
void
Transformator::loadMatrix(const float* m)
{
  memcpy(m_,m,16*sizeof(float));
}
/*! 
 * Copy to \p m matrix the transformation matrix. In m, the first brackets
 * indicates the lines, the second one the column of the matrix.
 */
void
Transformator::getMatrix(float* m) const
{
  memcpy(m,m_,16*sizeof(float));
}
/*! 
 * Post multiply the matrix. So M <- M*m. In other words, when
 * transforming a vertex, m will be performed before the current
 * transformation takes place.
 *
 * You can use a Transformator as argument (see operator const float*()).
 */
void
Transformator::postMultMatrix(const float* m)
{
  multiply(m_,m_,m);
}
/*!
 * Pre multiply the matrix. So M <- m*M. In other words, when
 * transforming a vertex, m will be performed after the current
 * transformation takes place.
 *
 * You can use a Transformator as argument (see operator const float*()).
 */
void
Transformator::preMultMatrix(const float* m)
{
  multiply(m_,m,m_);
}
/*!
 * Add a translation of (\p x,\p y,\p z) to the current transformation/
 */
void
Transformator::translate(float x,float y,float z)
{
  float m[16];
  ::loadIdentity(m);
  m[12] = x;
  m[13] = y;
  m[14] = z;
  postMultMatrix(m);
}
/*!
 * Add a translation around axis (\p x,\p y,\p z) to the current
 * transformation/. Angle \p angle is in radians. Axis needs not to be
 * normalized.
 */
void
Transformator::rotate(float rad,float x,float y,float z)
{
  const float len = sqrtf(x*x + y*y + z*z);
  if (len > 0.0000001)
  {
    x /= len;
    y /= len;
    z /= len;
  }
  const float sinth = sinf(rad);
  const float costh = cosf(rad);
  const float costhcomp = 1.0f - costh;
  float m[16];
  float* pm = m;
  *pm++ = costhcomp*x*x + costh;
  *pm++ = costhcomp*x*y + sinth*z;
  *pm++ = costhcomp*x*z - sinth*y;
  *pm++ = 0.0f;
  *pm++ = costhcomp*x*y - sinth*z;
  *pm++ = costhcomp*y*y + costh;
  *pm++ = costhcomp*y*z + sinth*x;
  *pm++ = 0.0f;
  *pm++ = costhcomp*x*z + sinth*y;
  *pm++ = costhcomp*y*z - sinth*x;
  *pm++ = costhcomp*z*z + costh;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm   = 1.0f;
  postMultMatrix(m);
}
/*!
 * Add a scaling.
 */
void
Transformator::scale(float x,float y,float z)
{
  float m[16];
  float* pm = m;
  *pm++ = x;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = y;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = z;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm++ = 0.0f;
  *pm   = 1.0f;
  postMultMatrix(m);
}
/*!
 * Provided for convenience. Equivalent to <tt>scale(s,s,s)</tt>
 */
void
Transformator::scale(float s)
{
  scale(s,s,s);
}
/*!
 * Apply the transformation to 3 components point \p src and put the result
 * to \p dest. This two vectors can be the same (temporary values are used).
 */
void
Transformator::transform(const float* src,float* dst) const
{
  const float x = src[0]*m_[0] + src[1]*m_[4] + src[2]*m_[ 8] + m_[12];
  const float y = src[0]*m_[1] + src[1]*m_[5] + src[2]*m_[ 9] + m_[13];
  const float z = src[0]*m_[2] + src[1]*m_[6] + src[2]*m_[10] + m_[14];
  const float w = src[0]*m_[3] + src[1]*m_[7] + src[2]*m_[11] + m_[15];
  dst[0] = x/w;
  dst[1] = y/w;
  dst[2] = z/w;
}
/*!
 * Write the internal matrix on stream;
 */
ostream&
wrl::operator<<(ostream& s,const Transformator& t)
{
  for (unsigned short i=0;i<4;++i)
  {
    for (unsigned short j=0;j<4;++j)
    {
      s<<setw(8)<<setprecision(3)<<t.element(i,j)<<" ";
    }
    s<<"\n";
  }
  return s;
}
/*!
 * Returns a Transformator whose matrix is the inverse transpose of this
 * one.
 */
Transformator
Transformator::inverseTranspose() const
{
  Transformator t;
  float tmp[16];
  transpose(m_,tmp);
  invertAffineMatrix(tmp,t.m_);
  return t;  
}
/*! 
 * Return true iff the 2 transformations are the same.
 */
bool
wrl::operator==(const wrl::Transformator& t0,
		const wrl::Transformator& t1)
{
  return (t0.m_[ 0] == t1.m_[ 0] &&
	  t0.m_[ 1] == t1.m_[ 1] &&
	  t0.m_[ 2] == t1.m_[ 2] &&
	  t0.m_[ 3] == t1.m_[ 3] &&
	  t0.m_[ 4] == t1.m_[ 4] &&
	  t0.m_[ 5] == t1.m_[ 5] &&
	  t0.m_[ 6] == t1.m_[ 6] &&
	  t0.m_[ 7] == t1.m_[ 7] &&
	  t0.m_[ 8] == t1.m_[ 8] &&
	  t0.m_[ 9] == t1.m_[ 9] &&
	  t0.m_[10] == t1.m_[10] &&
	  t0.m_[11] == t1.m_[11] &&
	  t0.m_[12] == t1.m_[12] &&
	  t0.m_[13] == t1.m_[13] &&
	  t0.m_[14] == t1.m_[14] &&
	  t0.m_[15] == t1.m_[15]);
}
/*! 
 * Return true iff the 2 transformations are different.
 */
bool
wrl::operator!=(const wrl::Transformator& t0,
		const wrl::Transformator& t1)
{
  return (t0.m_[ 0] != t1.m_[ 0] ||
	  t0.m_[ 1] != t1.m_[ 1] ||
	  t0.m_[ 2] != t1.m_[ 2] ||
	  t0.m_[ 3] != t1.m_[ 3] ||
	  t0.m_[ 4] != t1.m_[ 4] ||
	  t0.m_[ 5] != t1.m_[ 5] ||
	  t0.m_[ 6] != t1.m_[ 6] ||
	  t0.m_[ 7] != t1.m_[ 7] ||
	  t0.m_[ 8] != t1.m_[ 8] ||
	  t0.m_[ 9] != t1.m_[ 9] ||
	  t0.m_[10] != t1.m_[10] ||
	  t0.m_[11] != t1.m_[11] ||
	  t0.m_[12] != t1.m_[12] ||
	  t0.m_[13] != t1.m_[13] ||
	  t0.m_[14] != t1.m_[14] ||
	  t0.m_[15] != t1.m_[15]);
}
/*! 
 * Provide strict ordering.
 */
bool
wrl::operator<(const Transformator& t0,
	       const Transformator& t1)
{
  const float* pt0 = t0.m_;
  const float* pt1 = t1.m_;
  for (unsigned int i=0;i<16;++i,++pt0,++pt1)
  {
    if (*pt0 < *pt1) return true;
    if (*pt0 > *pt1) return false;
  }
  return false;
}
//************************************************************
// Implementation of TransformatorHierarchy
//************************************************************
/*! \class wrl::TransformatorHierarchy
 * \ingroup tools
 *
 * A class to represent a hierarchy of tranform. It is designed for
 * efficiency, that is when using transform(), the full hierarchy is not
 * re-evaluated . Instead the resulting transformation is maintained by
 * push() and pop(), using a stack to guarantee exactness when
 * pushing/popping.
 */
/*!
 * Create an empty hierarchy equivalent to identity transform.
 */
TransformatorHierarchy::TransformatorHierarchy()
{
}
void
TransformatorHierarchy::pop()
{
  t_.pop_back();
  // Reevaluate the resulting matrix
  result_.loadIdentity();
  for (deque<Transformator>::const_iterator iter = t_.begin();
       iter != t_.end();++iter)
  {
    result_.postMultMatrix(*iter);
  }
  invTransResult_ = result_.inverseTranspose();
}
/*!
 * Write the resulting internal matrix on stream;
 */
ostream&
wrl::operator<<(ostream& s,const TransformatorHierarchy& t)
{
  return s<<t.result_;
}
/*! 
 * Return true iff the resulting transformations of the 2 hierarchies are
 * the same.
 */
bool
wrl::operator==(const wrl::TransformatorHierarchy& th0,
	   const wrl::TransformatorHierarchy& th1)
{
  return th0.result_ == th1.result_;
}
/*! 
 * Return true iff the resulting transformations of the 2 hierarchies are
 * different.
 */
bool
wrl::operator!=(const wrl::TransformatorHierarchy& th0,
	   const wrl::TransformatorHierarchy& th1)
{
  return th0.result_ != th1.result_;
}
/*! 
 * Provide strict ordering.
 */
bool
wrl::operator<(const TransformatorHierarchy& th0,
	       const TransformatorHierarchy& th1)
{
  return th0.result_ < th1.result_;
}
// Local variables section.
// This is only used by emacs!
// Local Variables:
// ff-search-directories: ("." "../../../include/xdkwrl/tools/")
// End:
