#include <xdkwrl/tools/prettyprinter.h>

#include <xdkwrl/fieldtypes/sfbool.h>
#include <xdkwrl/fieldtypes/sfcolor.h>
#include <xdkwrl/fieldtypes/sffloat.h>
#include <xdkwrl/fieldtypes/sfimage.h>
#include <xdkwrl/fieldtypes/sfint32.h>
#include <xdkwrl/fieldtypes/sfnode.h>
#include <xdkwrl/fieldtypes/sfrotation.h>
#include <xdkwrl/fieldtypes/sfstring.h>
#include <xdkwrl/fieldtypes/sftime.h>
#include <xdkwrl/fieldtypes/sfvec2f.h>
#include <xdkwrl/fieldtypes/sfvec3f.h>
#include <xdkwrl/fieldtypes/mfcolor.h>
#include <xdkwrl/fieldtypes/mffloat.h>
#include <xdkwrl/fieldtypes/mfint32.h>
#include <xdkwrl/fieldtypes/mfnode.h>
#include <xdkwrl/fieldtypes/mfrotation.h>
#include <xdkwrl/fieldtypes/mfstring.h>
#include <xdkwrl/fieldtypes/mftime.h>
#include <xdkwrl/fieldtypes/mfvec2f.h>
#include <xdkwrl/fieldtypes/mfvec3f.h>
#include <xdkwrl/node.h>
#include <xdkwrl/script.h>
#include <xdkwrl/scene.h>
#include <xdkwrl/proto.h>

#include <iomanip>

using namespace std;
using namespace wrl;

//************************************************************
// Implementation of PrettyPrinter
//************************************************************
/*! \class wrl::PrettyPrinter
 * \ingroup tools
 *
 */
PrettyPrinter::PrettyPrinter(std::ostream& s)
  : ostm_(&s),
    indent_(0),
    bol_(true),
    forcep_(0)
{
}
void
PrettyPrinter::inc()
{
  indent_ += 2;
}
void
PrettyPrinter::dec()
{
  indent_ -= 2;
}
void
PrettyPrinter::newline()
{
  if (!bol_)
  {
    (*ostm_)<<'\n'<<string(indent_,' ');
    bol_ = true;
  }
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFBool&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFColor&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFFloat&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFImage&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFInt32&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const Node::FieldHandle& h)
{
  switch (h.fieldTypeId())
  {
  case sfBool:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfBoolValue());
    p.bol_ = false;
    break;
  case sfColor:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfColorValue());
    p.bol_ = false;
    break;
  case sfFloat:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfFloatValue());
    p.bol_ = false;
    break;
  case sfImage:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfImageValue());
    p.bol_ = false;
    break;
  case sfInt32:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfInt32Value());
    p.bol_ = false;
    break;
  case sfNode:
    if (*h.sfNodeValue() != NULL || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.sfNodeValue());
      p.bol_ = false;
    }
    break;
  case sfRotation:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfRotationValue());
    p.bol_ = false;
    break;
  case sfString:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfStringValue());
    p.bol_ = false;
    break;
  case sfTime:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfTimeValue());
    p.bol_ = false;
    break;
  case sfVec2f:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfVec2fValue());
    p.bol_ = false;
    break;
  case sfVec3f:
    (*p.ostm_)<<h.fieldName()<<' ';
    p<<(*h.sfVec3fValue());
    p.bol_ = false;
    break;
  case mfColor:
    if (!h.mfColorValue()->empty() || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.mfColorValue());
      p.bol_ = false;
    }
    break;
  case mfFloat:
    if (!h.mfFloatValue()->empty() || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.mfFloatValue());
      p.bol_ = false;
    }
    break;
  case mfInt32:
    if (!h.mfInt32Value()->empty() || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.mfInt32Value());
      p.bol_ = false;
    }
    break;
  case mfNode:
    if (!h.mfNodeValue()->empty() || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.mfNodeValue());
      p.bol_ = false;
    }
    break;
  case mfRotation:
    if (!h.mfRotationValue()->empty() || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.mfRotationValue());
      p.bol_ = false;
    }
    break;
  case mfString:
    if (!h.mfStringValue()->empty() || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.mfStringValue());
      p.bol_ = false;
    }
    break;
  case mfTime:
    if (!h.mfTimeValue()->empty() || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.mfTimeValue());
      p.bol_ = false;
    }
    break;
  case mfVec2f:
    if (!h.mfVec2fValue()->empty() || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.mfVec2fValue());
      p.bol_ = false;
    }
    break;
  case mfVec3f:
    if (!h.mfVec3fValue()->empty() || p.forcep_ > 0)
    {
      (*p.ostm_)<<h.fieldName()<<' ';
      p<<(*h.mfVec3fValue());
      p.bol_ = false;
    }
    break;
  case unknown:
    break;
  }
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const Node::EventInHandle& h)
{
  (*p.ostm_)<<h.fieldName();
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const Node::EventOutHandle& h)
{
  (*p.ostm_)<<h.fieldName();
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFNode& v)
{
  if (v == NULL)
  {
    (*p.ostm_)<<"NULL";
    p.bol_ = false;
    return p;
  }
  Script* script = dynamic_cast<Script*>(v.node());
  // First check if the node has been DEF before
  map<Node*,string>::const_iterator fter = p.definedNodes_.find(v.node());
  if (fter != p.definedNodes_.end())
  {
    (*p.ostm_)<<"USE "<<fter->second;
    p.bol_ = false;
    return p;
  }
  // Otherwise print the node
  if (v->hasName())
  {
    (*p.ostm_)<<"DEF "<<v->name()<<' ';
    // Add the node to the defined list.
    p.definedNodes_[v.node()] = string(v->name());
  }
  (*p.ostm_)<<v->typeName()<<" {";
  p.bol_ = false;
  p.inc();
  for (unsigned int i=0;i<v->nbEventsIn();++i)
  {
    Node::EventInHandle h = v->eventIn(i);
    // We check if this node is not bound (if we are in a proto declaration
    // printing), in which case we use the IS syntax.
    if (!p.fieldBindingsStack_.empty())
    {
      ProtoDeclaration::BindingMap::const_iterator
	fter = p.eventInBindingsStack_.top().find(v);
      if (fter != p.eventInBindingsStack_.top().end())
      {
	list<ProtoDeclaration::Binding>::const_iterator
	  iter = fter->second.begin();
	list<ProtoDeclaration::Binding>::const_iterator
	  istop= fter->second.end();
	while (iter != istop && iter->first != h.fieldName())
	{
	  ++iter;
	}
	if (iter != istop)
	{
	  p.newline();
	  (*p.ostm_)<<h.fieldName()<<" IS "<<iter->second;
	  p.bol_ = false;
	  continue;
	}
      }
    }
    // Otherwise, there is nothing to print (standard nodes have their
    // events known statically)
  }
  for (unsigned int i=0;i<v->nbEventsOut();++i)
  {
    Node::EventOutHandle h = v->eventOut(i);
    // We check if this node is not bound (if we are in a proto declaration
    // printing), in which case we use the IS syntax.
    if (!p.fieldBindingsStack_.empty())
    {
      ProtoDeclaration::BindingMap::const_iterator
	fter = p.eventOutBindingsStack_.top().find(v);
      if (fter != p.eventOutBindingsStack_.top().end())
      {
	list<ProtoDeclaration::Binding>::const_iterator
	  iter = fter->second.begin();
	list<ProtoDeclaration::Binding>::const_iterator
	  istop= fter->second.end();
	while (iter != istop && iter->first != h.fieldName())
	{
	  ++iter;
	}
	if (iter != istop)
	{
	  p.newline();
	  (*p.ostm_)<<h.fieldName()<<" IS "<<iter->second;
	  p.bol_ = false;
	  continue;
	}
      }
    }
    // Otherwise, there is nothing to print (standard nodes have their
    // events known statically)
  }  
  for (unsigned int i=0;i<v->nbFields();++i)
  {
    Node::FieldHandle h = v->field(i);
    // We check if this node is not bound (if we are in a proto declaration
    // printing), in which case we use the IS syntax.
    if (!p.fieldBindingsStack_.empty())
    {
      ProtoDeclaration::BindingMap::const_iterator
	fter = p.fieldBindingsStack_.top().find(v);
      if (fter != p.fieldBindingsStack_.top().end())
      {
	list<ProtoDeclaration::Binding>::const_iterator
	  iter = fter->second.begin();
	list<ProtoDeclaration::Binding>::const_iterator
	  istop= fter->second.end();
	while (iter != istop && iter->first != h.fieldName())
	{
	  ++iter;
	}
	if (iter != istop)
	{
	  p.newline();
	  (*p.ostm_)<<h.fieldName()<<" IS "<<iter->second;
	  p.bol_ = false;
	  continue;
	}
      }
    }
    // Otherwise we check if the field is worth printing
    if (v->isSetToDefaultValue(i))
    {
      continue;
    }
    p.newline();
    // Final step is to check if we are in a Script node in which case we
    // must print extra infos on the script.
    if (script != NULL && script->isDeclaredField(i))
    {
      (*p.ostm_)<<"field "<<h.fieldTypeName()		
		<<string(11-strlen(h.fieldTypeName()),' ')
		<<" ";
      p.bol_ = false;
    }
    p<<h;
  }
  p.dec();
  p.newline();
  (*p.ostm_)<<'}';
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFRotation& v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFString&   v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFTime&     v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFVec2f&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const SFVec3f&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const MFColor&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const MFFloat&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const MFInt32&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const MFNode&     v)
{
  if (v.empty())
  {
    (*p.ostm_)<<"[]";
    p.bol_ = false;
    return p;
  }
  if (v.size() == 1)
  {
    p<<v[0];
    return p;
  }
  (*p.ostm_)<<'[';
  p.bol_ = false;
  p.inc();
  for (MFNode::const_iterator iter = v.begin();
       iter != v.end();++iter)
  {
    p.newline();
    p<<(*iter);
  }
  p.dec();
  p.newline();
  (*p.ostm_)<<']';
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const MFRotation& v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const MFString&   v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const MFTime&     v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const MFVec2f&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const MFVec3f&    v)
{
  (*p.ostm_)<<v;
  p.bol_ = false;
  return p;
}
PrettyPrinter&
wrl::operator<<(PrettyPrinter& p,const Scene& s)
{
  (*p.ostm_)<<"#VRML V2.0 utf8 generated by XdkWrl\n\n";
  // Print the protos
  for (deque<ProtoDeclaration*>::const_iterator iter = s.protos.begin();
       iter != s.protos.end();++iter)
  {
    ProtoDeclaration* proto = *iter;
    p.newline();
    (*p.ostm_)<<"PROTO "<<proto->typeName()<<" [ ";
    p.bol_ = false;
    unsigned int saveIndent = p.indent_;
    bool hasIndented = false;
    // The event in
    for (unsigned int i=0;i<proto->nbEventsIn();++i)
    {
      if (i >= 1)
      {
	p.newline();
      }
      Node::EventInHandle h = proto->eventIn(i);
      (*p.ostm_)<<("eventIn  ")
		<<h.fieldTypeName()
		<<string(11-strlen(h.fieldTypeName()),' ');
      p.bol_ = false;
      ++p.forcep_;
      p<<h;
      --p.forcep_;
      if (i == 0)
      {
	p.indent_ += 9+strlen(proto->typeName());
	hasIndented = true;
      }
    }
    // The event out
    for (unsigned int i=0;i<proto->nbEventsOut();++i)
    {
      if (i == 0 && proto->nbEventsIn() != 0)
      {
	p.newline();
      }
      if (i >= 1 || hasIndented)
      {
	p.newline();
      }
      Node::EventOutHandle h = proto->eventOut(i);
      (*p.ostm_)<<("eventOut ")
		<<h.fieldTypeName()
		<<string(11-strlen(h.fieldTypeName()),' ');
      p.bol_ = false;
      ++p.forcep_;
      p<<h;
      --p.forcep_;
      if (i == 0 && !hasIndented)
      {
	p.indent_ += 9+strlen(proto->typeName());
	hasIndented = true;
      }
    }
    // The fields
    for (unsigned int i=0;i<proto->nbFields();++i)
    {
      if (i == 1 && !hasIndented)
      {
	p.indent_ += 9+strlen(proto->typeName());
      }
      if (i >= 1 || hasIndented)
      {
	p.newline();
      }
      Node::FieldHandle h = proto->field(i);
      (*p.ostm_)<<(h.isExposed()?"exposedField ":"field        ")
		<<h.fieldTypeName()
		<<string(11-strlen(h.fieldTypeName()),' ');
      p.bol_ = false;
      ++p.forcep_;
      p<<h;
      --p.forcep_;
    }
    (*p.ostm_)<<" ]";
    p.bol_ = false;
    p.indent_ = saveIndent;
    p.newline();    
    (*p.ostm_)<<'{';
    p.bol_ = false;
    p.inc();
    // Print the nodes with bindings activated
    p.fieldBindingsStack_.push(proto->fieldBindings());
    p.eventInBindingsStack_.push(proto->eventInBindings());
    p.eventOutBindingsStack_.push(proto->eventOutBindings());
    for (MFNode::const_iterator iter = proto->nodes().begin();
	 iter != proto->nodes().end();++iter)
    {
      p.newline();    
      p<<*iter;
    }
    p.fieldBindingsStack_.pop();
    p.eventInBindingsStack_.pop();
    p.eventOutBindingsStack_.pop();
    p.dec();
    p.newline();
    (*p.ostm_)<<'}';
    p.bol_ = false;
  }
  // Here we have to treat the case of extern proto. They could be inlined
  // if they are "loaded" (TODO: add to pretty printer a flag to do this
  // behaviour.
  for (deque<ExternProtoDeclaration*>::const_iterator
	 iter = s.externProtos.begin();
       iter != s.externProtos.end();++iter)
  {
    ExternProtoDeclaration* proto = *iter;
    p.newline();
    (*p.ostm_)<<"EXTERNPROTO "<<proto->typeName()<<" [ ";
    p.bol_ = false;
    unsigned int saveIndent = p.indent_;
    for (unsigned int i=0;i<proto->nbFields();++i)
    {
      if (i == 1)
      {
	p.indent_ += 15+strlen(proto->typeName());
      }
      if (i >= 1)
      {
	p.newline();
      }
      Node::FieldHandle h = proto->field(i);
      (*p.ostm_)<<"field "<<h.fieldTypeName()
		<<string(11-strlen(h.fieldTypeName()),' ')
		<<h.fieldName();
      p.bol_ = false;
    }
    (*p.ostm_)<<" ]";
    p.bol_ = false;
    p.indent_ -= 2;
    p.newline();
    p<<proto->urls;
    p.indent_ = saveIndent;    
  }
  // Print the nodes
  for (MFNode::const_iterator iter = s.nodes.begin();
       iter != s.nodes.end();++iter)
  {
    p.newline();
    p<<*iter;
  }
  p.newline();  
  return p;
}

// Local variables section.
// This is only used by emacs!
// Local Variables:
// ff-search-directories: ("." "../../../include/xdkwrl/tools/")
// End:
