/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
* tmr.c                                                                        *
* Copyright (c) 2003-2005 Troy Hanson                                          *
*******************************************************************************/
static const char id[]="$Id: tmr.c,v 1.14 2005/10/24 01:40:32 thanson Exp $";

#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include "libut/ut_internal.h"


extern UT_loop_global_type UT_loop_global;
time_t UT_app_start_time;

/******************************************************************************
 * UT_msec_to_walltime()                                                      *
 * Convert a delay time (in millisec) to 'wall clock' time.                   *
 *****************************************************************************/
struct timeval UT_msec_to_walltime( unsigned msec ) {
    struct timeval tv;
    long rel_sec, rel_usec, abs_usec, sec = 1000000;  /* a sec is 10^6 usec */

    /* Convert millisec to seconds and microseconds */
    rel_sec =  msec / 1000;             
    rel_usec = msec % 1000 * 1000;
    abs_usec = rel_usec + UT_loop_global.TOD.tv_usec;

    /* Convert relative sec/usec to wall sec/usec */
    tv.tv_sec =  UT_loop_global.TOD.tv_sec + rel_sec + (abs_usec >= sec ? 1 : 0);
    tv.tv_usec = (abs_usec >= sec) ? (abs_usec - sec) : abs_usec;

    return tv;
}

/******************************************************************************
 * UT_tmr_set()                                                               *
 * Create a timer that expires in the specified number of milliseconds        *
 *****************************************************************************/
UT_API void UT_tmr_set(char *name, unsigned msec, UT_tmr_cb *cb, void *data) {
    UT_tmr *tmr, *tmp;

    /* first delete previous timer of the same name if present */
    LL_FIND(UT_loop_global.tmrs, tmr, name);
    if (tmr) UT_tmr_clear( name );

    tmr = (UT_tmr *)UT_mem_alloc( TIMERS, 1 );
    UT_strncpy( tmr->name, name, TMR_NAME_MAXLEN );

    tmr->time = UT_msec_to_walltime( msec );
    tmr->msec = msec;
    tmr->cb = cb;
    tmr->data = data;
    tmr->next = NULL;

    /* If this is the only timer, or the soonest one, mark it 'first' */
    if (!UT_loop_global.tmr_first) UT_loop_global.tmr_first = &tmr->time;  
    else if (timercmp(&tmr->time, UT_loop_global.tmr_first, <)) {
        UT_loop_global.tmr_first = &tmr->time;
    }

    LL_ADD( UT_loop_global.tmrs, tmp, tmr );
}

/******************************************************************************
 * UT_tmr_update()                                                            *
 * Scan the pending timers. If any exist and have expired, invoke their cb's. *
 * A timer cb's return value determine whether it will be auto-rescheduled.   *
 * If the return value rc <0 it is not rescheduled; if 0 its rescheduled for  *
 * the same interval it had last time; and if rc>0 its rescheduled for rc msec*
 *****************************************************************************/
int UT_tmr_update() {
    UT_tmr *tmr, *tmr_nxt, *tmp, *expd_tmrs = NULL;
    struct timeval *soonest = NULL; 
    int rc;

    /* Anything to do? Check for (a) no timers; or (b) no expired timers */
    if (!UT_loop_global.tmr_first) return;              
    if (timercmp(&UT_loop_global.TOD, UT_loop_global.tmr_first, <)) return; 

    /* One or more timers have expired. Unlink their respective structs
     * from the timers list and link them into an expired list. Then we 
     * can invoke the timer callbacks. We don't invoke the cb's while
     * traversing the timers list since the cb's could modify the list. */
    tmr = UT_loop_global.tmrs;
    while (tmr) {
        tmr_nxt = tmr->next;

        if (timercmp(&tmr->time, &UT_loop_global.TOD, <)) {
            /* Timer expired; add to expd list */
            LL_DEL( UT_loop_global.tmrs, tmp, tmr); 
            tmr->next = NULL;
            LL_ADD( expd_tmrs, tmp, tmr);
        } else {
            /* Not yet expired, is it soonest? */
            if ( !soonest ) soonest = &tmr->time;
            else if (timercmp(&tmr->time, soonest, <)) {
                soonest = &tmr->time;
            }
        }
        tmr = tmr_nxt; 
    }

    /* Update the global 'first timer' pointer */
    UT_loop_global.tmr_first = soonest;

    /* Invoke cb's of all the expired timers. */
    UT_prf_bgn( "libut", "timer_callbacks");

    tmr=expd_tmrs; 
    while (tmr) {
        tmr_nxt = tmr->next;
        UT_LOG(Debugk, "Invoking timer cb %s", tmr->name);
        rc = (tmr->cb)( tmr->name, tmr->msec, tmr->data);
        if (rc < 0) UT_mem_free( TIMERS, tmr, 1 );
        else {
            /* Auto-reschedule this timer, changing msec if cb returned > 0 */
            if (rc > 0) tmr->msec = rc;
            tmr->time = UT_msec_to_walltime(tmr->msec);

            /* If this is the only timer, or the soonest one, mark it 'first' */
            if (!UT_loop_global.tmr_first) UT_loop_global.tmr_first =&tmr->time;
            else if (timercmp(&tmr->time, UT_loop_global.tmr_first, <)) {
                UT_loop_global.tmr_first = &tmr->time;
            }

            tmr->next = NULL;
            LL_ADD( UT_loop_global.tmrs, tmp, tmr );
        }
        tmr = tmr_nxt;
    }
    UT_prf_end( "libut", "timer_callbacks");
}

UT_API int UT_tmr_clear(char *name) {
    UT_tmr *tmr, *tmp;

    LL_FIND(UT_loop_global.tmrs, tmr, name);
    if (!tmr) {
        UT_LOG(Error, "Can't delete undefined timer %s", name);
        return -1;
    }

    UT_LOG(Info, "Deleting timer %s", name);
    LL_DEL( UT_loop_global.tmrs, tmp, tmr );
    /* recalculate the 'soonest' timer */
    UT_loop_global.tmr_first = 0;
    for(tmr=UT_loop_global.tmrs; tmr; tmr=tmr->next) {
        if (!UT_loop_global.tmr_first) UT_loop_global.tmr_first = &tmr->time;
        else if (timercmp(&tmr->time, UT_loop_global.tmr_first, <)) {
                UT_loop_global.tmr_first = &tmr->time;
        }
    }
    return 0;
}

/******************************************************************************
 * UT_tmr_shlcmd()                                                            *
 * Dump a list of pending timers to the shl port.                             *
 *****************************************************************************/
int UT_tmr_shlcmd( int argc, char* argv[] ) {
    UT_tmr *tmr = UT_loop_global.tmrs;
    struct timeval timeleft;
    char *cols[] = {"name               ", "hrs", "min", "sec", NULL };
    
    UT_shl_hdr( cols );
    while (tmr) {
        timersub(&tmr->time, &UT_loop_global.TOD, &timeleft);
        UT_shlf("%-19s %-3d %-3d %-3d\n", tmr->name, 
                        (timeleft.tv_sec / 3600),
                        (timeleft.tv_sec % 3600 / 60),
                        (timeleft.tv_sec % 60) );

        tmr = tmr->next;
    }

    return SHL_OK;
}


/*******************************************************************************
* UT_uptime_shlcmd()                                                           *
* A shl command to show the process uptime. This does not necessarily belong   *
* with the timer code. Its here because it involves the concept of time. :-)   *
*******************************************************************************/
int UT_uptime_shlcmd(int argc, char *argv[] ) {
   int days=0,hours=0,min=0,sec;
   time_t now;

   time(&now);
   sec = difftime(now,UT_app_start_time);
   while (sec >= (60*60*24)) { days++;  sec -= (60*60*24); }
   while (sec >= (60*60))    { hours++; sec -= (60*60);    }
   while (sec >= (60))       { min++;   sec -= (60);       }

   UT_shlf("program started: %s", ctime(&UT_app_start_time));
   UT_shlf("current time:    %s\n", ctime(&now));
   UT_shlf("up %d days, %d hours, %d min, %d sec\n\n", days,hours,min,sec);

   return SHL_OK;
}


int UT_tmr_init() {
    time(&UT_app_start_time);
    UT_mem_pool_create( TIMERS, sizeof(UT_tmr), 10 );
    UT_shl_cmd_create( "tmr", "show pending timers", (UT_shlcmd *)UT_tmr_shlcmd,
            NULL);
    UT_shl_cmd_create( "uptime", "show uptime", (UT_shlcmd *)UT_uptime_shlcmd,
            NULL);
}
