/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include "libut/ut_internal.h"

extern UT_log_global_type UT_log_global;
extern char *UT_loglevel_strs[];

/******************************************************************************
 * UT_log_msg_shl()                                                           *
 * Write out a log message, from shl.                                         *
 *****************************************************************************/
int UT_log_msg_shl(int argc, char *argv[] ) {
    int i;
    char *p;

    if (argc != 2) {
        UT_shlf("usage: log %s log_message", argv[0]);
        return;
    }
    
    /* Convert the log level string to numeric value and write log message */
    for (i=0; (p = UT_loglevel_strs[i]) != NULL; i++) {
        if (!strcmp( p, argv[0] )) {
            UT_LOG( i, argv[1] );
            return;
        }
    }

    UT_shlf("Unsupported log level '%s'", argv[0]);
    UT_LOG( Error, "Invalid log level '%s'; %s", argv[0], argv[1]);
}

/******************************************************************************
 * UT_log_bump_level()                                                        *
 * This is used to nudge the log level one notch higher or lower.             *
 *****************************************************************************/
int UT_log_bump_level (int argc, char*argv[]) {
    UT_loglevel new_level;

    /* Handle log level <newlevel> */
    if (!strcmp(argv[0], "level")) {
        if (argc != 2) {
            UT_shlf("usage: log level <newlevel>");
            return;
        }

        UT_var_set( LOGLEVEL_VARNAME, argv[1]);
        return;
    }

    /* Handle "log more" or "log less" */
    if (argc != 1) {
        UT_shlf("usage:\nlog more\nlog less\n");
        return;
    }

    if (!strcmp("more",argv[0]) && (UT_log_global.loglevel < Debugk)) 
            new_level = UT_log_global.loglevel + 1;
    else if (!strcmp("less",argv[0]) && (UT_log_global.loglevel > Fatal)) 
            new_level = UT_log_global.loglevel - 1;
    else {
        UT_shlf("Log level already at %s verbosity", 
                UT_log_global.loglevel ? "maximum" : "minimum");
        return;
    }

    /* Set it to its new text value, e.g. Info */
    UT_var_set( LOGLEVEL_VARNAME, UT_loglevel_strs[ new_level ] );
    UT_shlf( "Now logging at %s level", UT_loglevel_strs[UT_log_global.loglevel]);
}

/******************************************************************************
 * UT_log_reopen()                                                            *
 * Invoked if the variable holding the log file name is updated               *
 *****************************************************************************/
int UT_log_reopen(int argc, char *argv[]) {
    char abspath[PATH_MAX];
    char *file;

    /* Just show the current log file location if no arg */
    if (argc == 1) {
        UT_var_get( LOGFILE_VARNAME, &file);
        UT_shlf("%s\n", file);
        return;
    }

    if (argc > 2) {
        UT_shlf("Usage: log file [newfile]");
        return;
    }

    /* No feedback to shl if user points log file at unopenable path */
    if (argc == 2) UT_var_set( LOGFILE_VARNAME , argv[1]);
}

/******************************************************************************
 * UT_log_shlcmd()                                                            *
 * Dispatches to the appropriate subcommand                                   *
 *****************************************************************************/
int UT_log_shlcmd(  int argc, char *argv[] ) {
    char *file;
    UT_clproc *p;
    UT_cmdtable cmds[] = { 
            { "file",       UT_log_reopen },
            { "more",       UT_log_bump_level },
            { "less",       UT_log_bump_level },
            { "level",      UT_log_bump_level },
            { "Debug",      UT_log_msg_shl },
            { "Info",       UT_log_msg_shl },
            { "Warning",    UT_log_msg_shl },
            { "Error",      UT_log_msg_shl },
            { NULL,         NULL } 
    };

    if (argc == 1) {
        UT_shlf("Log level:  %s\n", UT_loglevel_strs[UT_log_global.loglevel]);
        UT_var_get( LOGFILE_VARNAME, &file);
        UT_shlf("Log file:   %s\n", file);
        return SHL_OK;
    }

    if ((p = UT_dispatch_subcmd( argv[1], cmds )) != NULL ) {
        (p)(--argc, ++argv);
    } else {
        UT_shlf("Unknown option '%s'. Try 'help %s'.", argv[1], argv[0]);
    }
    return SHL_OK;
}

int UT_log_usage(int argc,char*argv[]) {
        UT_shlf("Usage:\n");
        UT_shlf("log                   -- Show log level and log file\n");
        UT_shlf("log <level> <message> -- Write a message to the log\n");
        UT_shlf("log file [filename]   -- Show or change log file location\n");
        UT_shlf("log level <newlevel>  -- Change to specified log level\n");
        UT_shlf("log more              -- Increase logging verbosity\n");
        UT_shlf("log less              -- Decrease logging verbosity\n");
}

int UT_log_init_shl() {
        UT_shl_cmd_create( "log", "Change log file or verbosity",
                (UT_shlcmd*)UT_log_shlcmd, 
                (UT_helpcb*)UT_log_usage );
        return 0;
}
