/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
* log.c                                                                        *
* Copyright (c) 2003-2005 Troy Hanson                                          *
*******************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdarg.h>
#include <unistd.h>
#include <limits.h>
#include <stdlib.h>
#include <sys/time.h>
#include <time.h>
#include <errno.h>
#include "libut/ut_internal.h"

/* Define strings for each of the log levels */
#define EL(x) #x,
char *UT_loglevel_strs[] = { LOGLEVELS NULL };
#undef EL

/* A global structure to keep track of logging oriented info. */
UT_log_global_type UT_log_global = {
    .fd = STDERR_FILENO, 
    .loglevel = LOGLEVEL_DEFAULT,
    .datetime_fmt = LOGFILE_DATETIME_FORMAT,
    .log_line_fmt = LOGFILE_LINE_FORMAT 
};

/*******************************************************************************
* UT_setlglvl()                                                                *
* When the log level variable is updated, this cb updates the numeric global.  *
*******************************************************************************/
int UT_setlglvl(char *name, void *data) {
    int l;
    char *s;

    UT_var_get( LOGLEVEL_VARNAME, &s);
    if ( (l = UT_stridx(s,UT_loglevel_strs)) >= 0) {
        UT_log_global.loglevel = l;
        UT_LOG(Info, "Setting loglevel to %s [%d] (0=fatal,5=debugk)", s, l );
        return 0;
    }
    
    UT_LOG(Warning, "Invalid loglevel %s", s);
    return -1;
}

/*******************************************************************************
* UT_setlgfile()                                                               *
* This cb is invoked when the log file variable is changed. If the specified   *
* file is a relative path, it is first prefixed with the "base directory" var. *
*******************************************************************************/
int UT_setlgfile(char *name,void *data) {
    char abspath[PATH_MAX],*path,*basedir;
    int fd,rc;

    UT_var_get( LOGFILE_VARNAME, &path);

    if (path[0] == '/') {
        if (strlen(path) >= PATH_MAX) {
            UT_LOG(Error, "reverting; logfile path too long");
            return -1;
        } else {
            UT_strncpy( abspath, path, PATH_MAX);
        }
    } else {
        UT_var_get(SHL_BASEDIR_VARNAME, &basedir);
        rc = snprintf(abspath, PATH_MAX, "%s/%s", basedir, path);
        if (rc >= PATH_MAX || rc == -1) {
            UT_LOG(Error, "reverting; logfile absolute path too long");
            return -1;
        }
    }

    if ( (fd = open(abspath, O_WRONLY|O_APPEND|O_CREAT, LOG_CREATE_MODE)) == -1) {
        UT_LOG(Error, "Failed to open logfile %s: %s", abspath, strerror(errno));
        return -1;
    }

    /* Write a last entry to the old logfile */
    UT_LOG( Info, "Log closing. Switching to %s", abspath);
    close( UT_log_global.fd );

    /* start writing to new logfile */
    UT_log_global.fd = fd;
    UT_LOG( Info, "Opened logfile %s", abspath);
    return 0;
}

/*******************************************************************************
* UT_log_init()                                                                *
*******************************************************************************/
void UT_log_init(UT_loglevel level, char *logfile ) {
    int fd;

    if ( (fd = open(logfile, O_WRONLY|O_APPEND|O_CREAT, LOG_CREATE_MODE)) == -1) {
        /* write error to the existing log output descriptor e.g. /dev/stdout */
        UT_LOG(Error,"Failed to open logfile %s: %s", logfile, strerror(errno));
    } else UT_log_global.fd = fd;

    /* Make a var that has the log level in it */
    UT_log_global.loglevel = level;
    UT_LOG(Info, "logging opened to file %s", (fd ? logfile : LOGFILE_DEFAULT));
    UT_var_create( LOGLEVEL_VARNAME, "log level" , UT_var_string, UT_loglevel_strs[ level ] );
    UT_var_restrict( LOGLEVEL_VARNAME, strenum, UT_loglevel_strs );
    UT_var_reg_cb( LOGLEVEL_VARNAME, (UT_var_upd_cb*)UT_setlglvl, NULL ); 

    /* Make a var that has the logfile name */
    UT_var_create( LOGFILE_VARNAME, "log file", UT_var_string, ( fd ? logfile :
                LOGFILE_DEFAULT)); UT_var_reg_cb( LOGFILE_VARNAME,
                (UT_var_upd_cb*)UT_setlgfile, NULL );

    UT_log_init_shl();

}


/*******************************************************************************
* UT_log_core()                                                                *
* Write a log message at the given level. Usually called via UT_log.           *
*******************************************************************************/
void UT_log_core( UT_loglevel level, const char *func, char *file, unsigned 
        line_num, char *msg, va_list ap) {
    const char *l;
    char c, *d = "", datetime[DATETIME_MAXLEN], line[LOGLINE_MAXLEN + 1], 
        *linefmt, line_str[LINESTR_MAXLEN];
    int i = 0, expect_conversion_char = 0, spc_left,len;
    struct timeval now_tv;
    time_t now_time;
    struct tm *tm;

    
    if (level <= UT_log_global.loglevel) {

        /* Format the datetime part of the log message */
        gettimeofday( &now_tv, NULL );  
        now_time = (time_t)now_tv.tv_sec;
        tm = localtime( &now_time );  
        if (strftime( datetime, DATETIME_MAXLEN, UT_log_global.datetime_fmt, tm))
            d = datetime;


        /* Format the log line according to the specified format conversion. */
        linefmt = UT_log_global.log_line_fmt;
        expect_conversion_char = 0;
        while ((c = *(linefmt++)) && (i < LOGLINE_MAXLEN )) {
            if (expect_conversion_char) {
                expect_conversion_char = 0;
                spc_left = LOGLINE_MAXLEN - i;
                switch (c) {
                    case 't':   /* datetime string */
                        l = d;
                        break;
                    case 'l':   /* loglevel string */
                        l = ENUM_STRVAL( UT_loglevel, level);
                        break;
                    case 'f':   /* function string */
                        l = func;
                        break;
                    case 'F':   /* file name string */
                        l = file;
                        break;
                    case 'L':   /* line number string */
                        snprintf( line_str, LINESTR_MAXLEN, "%u", line_num );
                        l = line_str;
                        break;
                    case 'm':   /* msg string */
                        len = vsnprintf(&line[i], spc_left, msg, ap);   
                        i = len > spc_left ? LOGLINE_MAXLEN : i + len;
                        continue; /* skip to the next while{} iteration */
                    default:
                        /* Unsupported conversion specifier.  */
                        l = "?";
                        break;
                }
                /* append the field, advancing the index; 'm' skips this. */
                strncpy( &line[i], l, spc_left);
                len = strlen (l);
                i =  len > spc_left ? LOGLINE_MAXLEN : i + len;

            } else {
                if (c == '%') expect_conversion_char = 1;
                else line[i++] = c;
            }
        }

        /* We're done, just newline-terminate the line and print it. */
        line[i++] = '\n';   

        /* As a side note, we don't use UT_fd_write here because 
         * it contains its own UT_LOG calls. we don't want to recurse. */
        write(UT_log_global.fd, line, i);

        /* last, we terminate the whole program on fatal errors */
        if (level == Fatal) exit(EXIT_FAILURE);
    }
}

UT_API void UT_log(UT_loglevel level, const char *func, char *file, unsigned
line_num, char *msg, ...) {
  va_list ap;

  va_start(ap,msg);
  UT_log_core(level,func,file,line_num,msg,ap);
  va_end(ap);
}



