/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*******************************************************************************
* listen.c                                                                     *
* Copyright (c) 2003-2005 Troy Hanson                                          *
*******************************************************************************/
static const char id[]="$Id: listen.c,v 1.17 2005/10/23 05:31:46 thanson Exp $";

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <netinet/in.h>
#include <sys/uio.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <string.h>
#include "libut/ut_internal.h"

extern UT_loop_global_type UT_loop_global;

/*******************************************************************************
* UT_ipport_parse()                                                            *
* Convenience function to parse a strings like the following to IP and port:   *
* 127.0.0.1:4445                                                               *
* *:4445             (* implies INADDR_ANY; all IP's belonging to a host)      *
*                                            Returns -1 on error, or >= 0 if ok*
*******************************************************************************/
int UT_ipport_parse(char *ipport, in_addr_t *ip_out, int *port_out) {
    char *colon,*port,*ip,ipstr[INET_ADDRSTRLEN];
    int i,rc;

    for(i=0, colon=NULL; i<strlen(ipport) && !colon; i++) 
        if (ipport[i] == ':') colon = &ipport[i];

    if (!colon) {
        UT_LOG(Error, "Couldn't parse IP:port in %s", ipport);
        return -1;
    }

    if (colon-ipport >= INET_ADDRSTRLEN) {
        UT_LOG(Error, "IP part of IP:port %s exceeds %d chars", ipport, 
                INET_ADDRSTRLEN-1);
        return -1;
    }

    /* Parse the IP address, first extracting the substring preceding colon. */
    UT_strncpy(ipstr,ipport,colon-ipport+1);  
    if (!strcmp(ipstr,"*")) *ip_out = htonl(INADDR_ANY);
    else if ( (rc = inet_pton(AF_INET, ipstr, ip_out)) != 1) {
        UT_LOG(Error, "format error in IP %s", ipstr);
        return -1;
    }

    /* now parse the port */
    if ( sscanf(colon+1, "%d", port_out) != 1 ) {
        UT_LOG(Error, "invalid port in IP:port %s", ipport);
        return -1;
    }

    return 0;
}

/*******************************************************************************
* UT_accept_cb()                                                               *
* When a listening socket receives an incoming connection request, this        *
* function is invoked to accept it. The newly-accepted connection (connected   *
* socket) is set up so that I/O will be passed to the application callback.    *
*******************************************************************************/
int UT_accept_cb( int fd, char *name, int flags, UT_callback* cbdata) {
    char remote_ip[INET_ADDRSTRLEN], remote_ipport[UT_IPPORT_MAXLEN];
    int remote_port, fd2, flags2, flags3, flags4; 
    struct sockaddr_in sockaddr;
    UT_fd_aux_type aux;
    socklen_t len;

    len = sizeof(sockaddr);
    memset(&sockaddr,0,len);
    if ( (fd2 = accept(fd, (struct sockaddr*)&sockaddr, &len)) == -1) {
        UT_LOG(Error, "Socket accept failed: %s", strerror(errno));
        return;  
    }

    /* Use non-blocking I/O. */
    if ( (flags2 = fcntl( fd2, F_GETFL, 0)) >= 0) {
        fcntl( fd2, F_SETFL, flags2 | O_NONBLOCK);
    }

    /* Get remote IP and port */
    inet_ntop(AF_INET, &sockaddr.sin_addr, remote_ip, INET_ADDRSTRLEN);
    remote_port = ntohs(sockaddr.sin_port);
    sprintf(remote_ipport, "%s:%d", remote_ip, remote_port);
    UT_LOG(Info,"Accepted %s connection from %s", name, remote_ipport); 

    /* Register the accepted socket for I/O. Get local IP from listener aux. */
    UT_fd_cntl(fd, UTFD_GET_AUX, &aux);
    flags3 = UTFD_R | UTFD_SOCKET | UTFD_SOCKET_ACCEPTED | 
             UTFD_SOCKET_LOCALADDR | UTFD_SOCKET_REMOTEADDR;
    UT_fd_reg( fd2, name, cbdata->cb.fdcb, cbdata->data, flags3, 
               aux.socket.local_ipport, remote_ipport);

    /* Notify application of the new connection by invoking the callback. */
    flags4 = flags3 | UTFD_IS_NEWACCEPT;
    (cbdata->cb.fdcb)(fd2, name, flags4, cbdata->data);
}

/*******************************************************************************
* UT_net_listen_close()                                                        *
* Close a listener previously created with UT_net_listen.                      *
*******************************************************************************/
UT_API int UT_net_listen_close(int fd) {
    UT_fd *fdx;
    char *ip;

    HASH_FIND_INT(UT_loop_global.fds, fdx, fd, &fd);
    if (fdx) {
        if (fdx->flags & UTFD_SOCKET_LISTENING) {
            ip = fdx->aux.socket.local_ipport;
            UT_LOG(Debug, "Closing listener %s (%s)", fdx->name, ip);
            close(fdx->fd);
            UT_mem_free(CBDATA, fdx->data, 1); 
            UT_fd_unreg(fdx->fd);
            return 0;
        }
    }

    UT_LOG(Error, "Can't close listener (fd %d not listening)", fd);
    return -1;
}

/*******************************************************************************
* UT_net_listen()                                                              *
* Establish a listening socket on the specified IP address and port. The IP    *
* may be NULL implying all IP's for this host (INADDR_ANY) otherwise its a     *
* string e.g. "127.0.0.1". Incoming connection requests will be accepted and   *
* connection notification as well as I/O will be passed to the given cb.       *
*                                        Returns fd on sucess, -1 on error.    *
*******************************************************************************/
UT_API int UT_net_listen(char *name, char *ipport, UT_fd_cb *cb, void *data ) {
    struct sockaddr_in servaddr;
    int fd, flags, on=1, rc, port;
    in_addr_t ip;
    UT_callback *cbdata;

    /* Create a socket and specify reuse-addr. See Stevens UNPv1 p194. */
    fd = socket( AF_INET, SOCK_STREAM, 0 );
    if ( setsockopt( fd, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(int)) )  {
        UT_LOG( Error, "setsockopt failed: %s", strerror( errno )); 
        close(fd);
        return -1;
    }

    /* Bind the local address (IP/port specification) to the new socket. */
    memset( &servaddr, 0, sizeof(servaddr) );
    servaddr.sin_family = AF_INET;
    if (UT_ipport_parse(ipport,&ip,&port) < 0) {
        UT_LOG(Error, "can't listen: parse error in IP/port %s", ipport);
        close(fd);
        return -1;
    }
    servaddr.sin_addr.s_addr = ip;
    servaddr.sin_port =  htons( port  );
    if (bind( fd, (struct sockaddr *)&servaddr, sizeof(servaddr) ) == -1) {
        UT_LOG(Error, "bind socket to %s failed: %s", ipport, strerror(errno));
        close(fd);
        return -1;
    }

    /* Use non-blocking listener so accept won't block. (Stevens UNPv1 p422).*/
    if ( (flags = fcntl( fd, F_GETFL, 0)) >= 0) {
        flags |= O_NONBLOCK;
        if (fcntl(fd,F_SETFL,flags) == -1) UT_LOG(Error,"fcntl setfl failed");
    } else UT_LOG(Error, "fcntl getfl failed");

    /* set close-on-exec flag */
    if ( (flags = fcntl( fd, F_GETFD, 0)) >= 0) {
        flags |= FD_CLOEXEC;
        if (fcntl(fd,F_SETFD,flags) == -1) UT_LOG(Error,"fcntl setfd failed");
    } else UT_LOG(Error, "fcntl getfd failed");

    /* Listen, and setup callback to accept connections made to the listener. */
    /* The "backlog" value of 32 is probably adequate for most apps but there *
     * is no reason it can't be increased for a busy app. Stevens UNPv1 p95. */
    if ( listen( fd, 32 ) == -1 ) {
        UT_LOG(Error, "listen on %s failed: %s", ipport, strerror(errno));
        close(fd);
        return -1;
    }

    cbdata = (UT_callback*)UT_mem_alloc( CBDATA, 1 );
    cbdata->cb.fdcb = cb;
    cbdata->data = data;
    flags = UTFD_R | UTFD_SOCKET | UTFD_SOCKET_LISTENING |UTFD_SOCKET_LOCALADDR;
    UT_fd_reg(fd,name,(UT_fd_cb*)UT_accept_cb, cbdata, flags, ipport);
    UT_LOG(Info, "%s listening on %s", name, ipport);
    return fd;
}

int UT_net_listen_init() {
    return 0;
}
