/*
Copyright (c) 2003-2005, Troy Hanson
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of the copyright holder nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*****************************************************************************
 *  ut_iob.c                                                                 *
 *  Functions for input-output buffers. These are used when gathering input  *
 *  in various-sized chunks during a network read, for example, but can also *
 *  be used to construct an output buffer one chunk at a time.               *
 *  An iob is simply a linked list of malloc'd buffers.                      *
 *  Copyright (c) 2003-2005 Troy Hanson                                      *
 ****************************************************************************/
static const char id[]="$Id: iob.c,v 1.14 2005/11/07 04:08:34 thanson Exp $";

#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <string.h>
#include "libut/ut_internal.h"

/* memory pool name */
#define UTIOB "ut_iob"


/****************************************************************************
*  UT_iob_create                                                            *
*  Creates a new iob (input output buffer) with an  initial string or NULL  *
****************************************************************************/
UT_iob *UT_iob_create() {
    UT_iob *iob;

    iob = (UT_iob*)UT_mem_alloc(UTIOB, 1);
    iob->len = 0;
    iob->buf  = NULL;
    iob->next = NULL;

    return iob;
}

/****************************************************************************
*  UT_iob_free                                                              *
*  Frees up a list of iob's whose head is the argument.                     *
****************************************************************************/
int UT_iob_free(UT_iob *iob) {

    while (iob) {
        if (iob->buf) free(iob->buf);
        UT_mem_free(UTIOB, iob, 1);
        iob = iob->next;
    }
    return 0;
}

/****************************************************************************
*  UT_iob_writefd                                                           *
*  Writes the io buffer list (iob) to a file descriptor                     *
****************************************************************************/
int UT_iob_writefd(UT_iob *iob, int fd) {
    int rc,n=0;

    while (iob) {
        if (iob->len > 0) {
            if ( (rc = UT_fd_write(fd, iob->buf, iob->len)) == -1 ) {
                UT_LOG(Error, "Failed to write iob to fd %d (%d bytes written)",
                        fd, n);
                return -1;
            }
            else n += rc;
        }
        iob = iob->next;
    }
    return n;
}

/****************************************************************************
*  UT_iob_printf                                                            *
*  Append the given printf-style variable-argument string to the iob.       *
****************************************************************************/
int UT_iob_printf(UT_iob *iob, char *fmt, ...) {
    va_list ap;
    char *s;
    UT_iob *new_iob,*tmp_iob;

    va_start(ap, fmt);
    s = UT_dsprintf(fmt, ap);
    va_end(ap);

    if (s) {
        /* Use the current iob if its unused */
        if (iob->len == 0 && iob->next == NULL) {
            new_iob = iob;
        }
        else {
            new_iob = UT_iob_create();
            LL_ADD(iob, tmp_iob, new_iob);
        }
        new_iob->buf = s;
        new_iob->len = strlen(s);  /* null terminator not counted */
        return 0;
    }

    UT_LOG(Fatal,"Out of memory. Failed to allocate string.");
    return -1;  /* not reached */
}


/******************************************************************************
 * UT_iob_append()                                                            *
 * This appends a binary buffer of the given length to the iob                *
 *****************************************************************************/
int UT_iob_append(UT_iob *iob, void *b, size_t n ) {
    UT_iob *new_iob,*tmp_iob;
    void *buf;

    /* Copy the input buffer into alloc'd mem */
    buf = malloc(n);
    if (!buf) {
        UT_LOG(Fatal,"Out of memory");
        return -1; /* not reached */
    }

    /* Use the current iob if its unused */
    if (iob->len == 0 && iob->next == NULL) new_iob = iob;
    else {
        new_iob = UT_iob_create();
        LL_ADD(iob, tmp_iob, new_iob);
    }

    new_iob->buf = memcpy(buf,b,n);
    new_iob->len = n;
    return 0;
}

/***************************************************************************
 * UT_iob_bufcmp()                                                         *
 * It takes special comparison logic to compare an iob's content to that   *
 * of a conventional (linear, contiguious) memory buffer. In particular,   *
 * it must be done piecewise, since the iob is in fragments.  Only the     *
 * first "buflen" bytes of the input buffer are compared against the iob.  *
 *                                                                         *
 * Like strncmp() this function returns 0 if there is an exact match and   *
 * non-zero otherwise.                                                     *
 *                                                                         *
 **************************************************************************/
int UT_iob_bufcmp( UT_iob *iob, char *buf, size_t buflen) {
    int i,j;

    for( i = 0, j = 0; i < buflen && iob; i++ ) {
        if (j < iob->len) {
            if (buf[i] != iob->buf[j]) {
                return -1;  /* bufs differ */
            } else {
                j++;        /* byte matched, proceed */
            }
        } else {
            iob = iob->next;  /* advance to next iob */
            j = 0;        /* reset index w/in iob */
            i--;          /* re-do this comparison */
        }
    }

    /* Here if we exhausted the iob (but buffer not exhausted), or
     * we exhausted the buffer (but not the iob), or
     * we exhausted the buffer and the iob and they matched */

    if (i == buflen) return 0;  /* buf cmp found equality */
    else return -2;         /* iob shorter than buflen */
}

/******************************************************************************
 * UT_iob_len()                                                               *
 * utility function to get the total length of an IOB                         *
 *****************************************************************************/
size_t UT_iob_len( UT_iob *iob ) {
    size_t len = 0;

    while (iob) {
        len += iob->len;
        iob=iob->next;
    }

    return len;
}

/******************************************************************************
 * UT_iob_flatten()                                                           *
 * Writes an iob to contiguous alloc'd memory. Caller must free() buffer.     *
 * The trailing NULL that this function adds is not counted in buffer length. *
 * The out_len parameter may be NULL if the caller does not want the length.  *
 *****************************************************************************/
char *UT_iob_flatten( UT_iob *iob, size_t *out_len) {
    char *data, *pos;
    size_t len;

    len = UT_iob_len( iob );
    if (out_len) *out_len = len;
    if ( (data = malloc(len + 1)) == NULL ) UT_LOG( Fatal, "malloc failure");

    pos = data;
    while (iob) {
        if (iob->len > 0) {
            memcpy(pos,iob->buf,iob->len);
            pos += iob->len;
        }
        iob=iob->next;
    }
    *pos = '\0';  /* null terminate- a convenience for string iobs */

    return data;
}

int UT_iob_init() {
    UT_mem_pool_create( UTIOB,  sizeof(UT_iob), 10 );
}
