/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: print.c,v 1.34 2002/10/29 09:02:54 kingofgib Exp $
 */


/*----------------------------------------------------------------------*
 * filename:		print.c
 * created on:		Fri May 10 12:08:31 CEST 2002
 * created by:		teddykgb
 * project: 		Portable Network Library
 *----------------------------------------------------------------------*/

/*
 * Printing: debugging, errors, and the like
 */
# include <stdio.h>
# include <stdarg.h>

# include "local.h"
# if defined HAVE_SYSLOG_H
#   include <syslog.h>
# endif

# ifndef LOG_INFO
#   define LOG_INFO	0
# endif

# ifndef LOG_NOTICE
#   define LOG_NOTICE	0
# endif

# ifndef LOG_ERR
#   define LOG_ERR	0
# endif

# include <sys/types.h>
# include <sys/stat.h>
# include <fcntl.h>

static int	debugLevel = -1;
static unsigned writeFlags = 0;
static int	dbgEnabled = 0;
static int	errEnabled = 0;
static FILE *		dbgStream = NULL;
static FILE *		errStream = NULL;
static PNET_THREAD_LOCK	dbgLock = NULL;
static PNET_THREAD_LOCK	errLock = NULL;

# ifdef HAVE_SYSLOG_H
static int	toSyslog   = 0;
# endif

static int
_write_stuff( FILE * stream, const char *prefix, int priority,
	      const char *fmt, va_list ap )
{
    int 	n;
    char 	buf[256];

# if defined HAVE_VSYSLOG	/* Cygwin ain't got it .... */
    if ( toSyslog )
	{ vsyslog( priority, fmt, ap ); return 0; }
# endif

    if (prefix)
	fprintf( stream,prefix);

    if ( writeFlags & PNET_LOG_TIME )
    {
	fprintf( stream, sys_get_current_time( buf, sizeof( buf ) ) );
	fprintf( stream, " " );
    }

    if ( writeFlags & (PNET_LOG_PID | PNET_LOG_TID) )
	fprintf(stream, "[");

    if ( writeFlags & PNET_LOG_PID )
	fprintf( stream, "%u", (pnet_uint)sys_getpid());
    if ( writeFlags & PNET_LOG_TID )
    {
	if ( writeFlags & PNET_LOG_PID )
	    fprintf(stream, "/");
	fprintf( stream, "%u", (pnet_uint)pnetThreadId());
    }
    if ( writeFlags & (PNET_LOG_PID | PNET_LOG_TID) )
	fprintf(stream, "] ");

    n = vfprintf(stream,fmt,ap);

    fflush( stream );

    return n;
}
# ifdef STDWin32
# define S_IRUSR	_S_IREAD
# define S_IWUSR	_S_IWRITE
# endif

static FILE*
open_file( const char * name )
{
    FILE * fp = NULL;
    int	   fd;

    fd = open( name, O_WRONLY | O_CREAT | O_TRUNC | O_EXCL
# ifdef O_EXLOCK
	| O_EXLOCK
# endif
# ifdef O_NOFOLLOW
	| O_NOFOLLOW
# endif
	,
	S_IRUSR | S_IWUSR

	);
    if ( fd < 0 )
	fprintf(stderr,"Cannot open '%s': %s\n", name, SYSERR());
    else
	fp = fdopen( fd, "w" );

    return fp;
}
static int
create_write_locks( void )
{
    if ( ! dbgLock )
	dbgLock = pnetThreadLockCreate();

    if ( ! errLock )
	errLock = pnetThreadLockCreate();

    return - ! ( dbgLock && errLock );
}
static int
dbg_write(const char *prefix,const char *fmt,va_list ap)
{
    int n;

    if ( ! dbgEnabled )
	return 0;

    create_write_locks();

    pnetThreadLockAcquire( dbgLock );
    n = _write_stuff( dbgStream, prefix, LOG_INFO, fmt, ap );
    pnetThreadLockRelease( dbgLock );

    return n;
}

static int
err_write(int e_type,const char *prefix,const char *fmt,va_list ap)
{
    FILE * stream = errStream;
    int	   n;

    DBG( printf(" errStream = %lX\n", XX(errStream)) );

    if ( e_type != E_FATAL && ! errEnabled )
	return 0;

    if ( ! stream )
	stream = stderr;

    DBG( printf(" stream = %lX\n", XX(stream)) );

    create_write_locks();

    /* Make provisions for cases where errStream == dbgStream: we
     * just use the dbgLock for this
     */
    if ( errStream == dbgStream )
	pnetThreadLockAcquire( dbgLock );
    else
	pnetThreadLockAcquire( errLock );

    n = _write_stuff( stream, prefix, LOG_ERR, fmt, ap );

    if ( errStream == dbgStream )
	pnetThreadLockRelease( dbgLock );
    else
	pnetThreadLockRelease( errLock );

    return n;
}
/*----------------------------------------------------------------------*/
/* Exported								*/
/*----------------------------------------------------------------------*/

void
dbg_set_params(FILE *fp,unsigned flags)
{
    dbgStream = fp;
    writeFlags= flags;
}

int
dbg(const char *fmt,...)
{
    int         n;
    va_list     ap;

    va_start(ap,fmt);
        n = dbg_write(NULL,fmt,ap);
    va_end(ap);

    return n;
}

int
err(const char *fmt,...)
{
    int         n;
    va_list     ap;

    va_start(ap,fmt);
        n = err_write(0,NULL,fmt,ap);
    va_end(ap);

    return n;
}

static const char *err_pfx[] =
{
    "    - ",
    "pnet fatal - ",		/* PNETdbgFatal */
    "pnet warn  - ",		/* PNETdbgWarn */
    "pnet       - ",		/* PNETdbgMsg */
    "pnet info  - ",		/* PNETdbgInfo */
    "pnet dbg 1 - ",		/* PNETdbgDbg1 */
    "pnet dbg 2 - ",		/* PNETdbgDbg2 */
    "pnet dbg 3 - ",		/* PNETdbgDbg3 */
    "pnet dbg 4 - ",		/* PNETdbgDbg4 */
};

int
perr(unsigned int type,const char *fmt,...)
{
    int         n;
    va_list     ap;

    if (type != E_FATAL && (int)type > debugLevel)
	return 0;

    if (type >= sizeof(err_pfx)/sizeof(char*))
        type = 0;

    va_start(ap,fmt);
        n = err_write(type,err_pfx[type],fmt,ap);
    va_end(ap);

    return n;
}
int
pdbg(unsigned type,const char *fmt,...)
{
    int         n;
    va_list     ap;

    if ((int)type > debugLevel)
	return 0;

    if (type >= sizeof(err_pfx)/sizeof(char*))
        type = 0;

    va_start(ap,fmt);
        n = dbg_write(err_pfx[type],fmt,ap);
    va_end(ap);

    return n;
}
int
pnetinternalerror(const char *fname,int line,const char *fmt,...)
{
    int         n;
    va_list     ap;

    fprintf(stderr,"*** PNET internal error in %s,%d: ",fname,line);

    va_start(ap,fmt);
        n = vfprintf(stderr,fmt,ap);
    va_end(ap);

    return n;
}

void
print_to_syslog( int on, const char * prog_id )
{
# ifdef HAVE_SYSLOG_H
    toSyslog = on != 0;
    openlog( prog_id, LOG_PID, 0 );
    return;
# endif
    on = on; prog_id = prog_id;
}
/*
 * Initilize the logging environment.
 */
int
print_init( void )
{
    const char * log_file = sys_getenv( PNET_ENV_LOG_FILE );
    const char * err_file = sys_getenv( PNET_ENV_ERR_FILE );
    const char * log_flags= sys_getenv( PNET_ENV_LOG_FLAGS );

    DBG( printf("print_init()\n") );
    DBG( printf("log_file = %s\n",log_file) );
    DBG( printf("err_file = %s\n",err_file) );

    /*
     * If the log file environment variable is set, then we need to do
     * logging. We try to open the log file, and revert to stdout if we
     * cannot open the file or if the user specified stdout.
     */
    if ( ! dbgStream && log_file )
    {
	dbgEnabled = 1;

	if ( strcmp( log_file, "stdout" ) == 0 )
	    dbgStream = stdout;
	else if ( !(dbgStream = open_file( log_file ) ) )
	{
	    fprintf(stderr,"Cannot open log %s.\n", log_file );
	    dbgStream = stdout;
	}

	DBG( printf( "Opened log stream %s\n",
		     dbgStream == stdout ? "stdout" : log_file ) );
    }

    /*
     * We do the same for the error log file.
     */
    if ( ! errStream )
    {
	if ( ! err_file || ! *err_file )
	    errStream = dbgStream;
	else
	{
	    errEnabled = 1;

	    if ( strcmp( err_file, "stderr" ) == 0 )
		errStream = stderr;
	    else if ( !(errStream = open_file( err_file ) ) )
	    {
		fprintf(stderr,"Cannot open error log %s.\n", err_file );
		errStream = stderr;
	    }

	    DBG( printf( "Opened error log stream %s\n",
		    errStream == stderr ? "stderr" : err_file ) );
	}
    }

    DBG( printf(" dbgStream = %lX\n", XX(dbgStream)) );
    DBG( printf(" errStream = %lX\n", XX(errStream)) );

    if ( dbgStream || errStream )
    {
	/*
	 * Parse any log flags
	 */
	const char * p,*e;
	int    len;

	if ( ! log_flags )
	    return 0;

	writeFlags = 0;
	debugLevel = 0;

	p = log_flags;

	do {
	    if ( *p == ',' )
		p++;

	    if ( (e = strchr( p, ',' )) )
		len = e - p;
	    else
		len = strlen( p );

	    if ( len > 0 )
	    {
		if ( ! debugLevel )
		    debugLevel = atoi( p );

		if ( ! strncmp( p, "time", len ) )
		    writeFlags |= PNET_LOG_TIME;
		else if ( ! strncmp( p, "pid", len ) )
		    writeFlags |= PNET_LOG_PID;
		else if ( ! strncmp( p, "tid", len ) )
		    writeFlags |= PNET_LOG_TID;
	    }

	    p += len;
	} while ( *p );

	DBG( printf("writeFlags = %X\n", writeFlags ) );
	DBG( printf("debugLevel = %d\n", debugLevel ) );
    }

    return 0;
}
int
pnetBindOutputStreams( FILE * outstream, FILE * errstream )
{
    if ( dbgStream )
	fflush( dbgStream );
    dbgStream = outstream;

    if ( errStream )
	fflush( errStream );
    errStream = errstream;
    return 0;
}

/*----------------------------------------------------------------------*/
/* hexdump								*/
/* We dump bpl bytes per line, in tuples of 4 x 4. 			*/
/* base can be 10, 8 or 16, which means output in decimal, octal, or hex.*/
/*----------------------------------------------------------------------*/
int
pnetHexdumpX( FILE *stream, byte * buf, int len, int bpl, int base )
{
    int i = 0;
    int n = 0;
    int line = 1;
    const char *fmt;

    if ( base == 10 )
	fmt = "%.3d ";
    else if ( base == 8)
	fmt = "%.3o ";
    else
	fmt = "%.2X ";
    
    for ( ; i < len; i++)
    {
	if ( n == 0 )
	    fprintf(stream,"% .2d: ",line++);

	fprintf(stream,fmt,buf[i]);

	n++;

	if ( !(n % 4) )
	    fputc(' ',stream);

	if ( n == bpl )
	    { fputc('\n',stream); n = 0; }
    }
    if ( n > 0 )
	fputc('\n',stream);

    return 0;
}
int
pnetHexdump( FILE * stream, byte *buf, int len, int base )
{
    return pnetHexdumpX( stream, buf, len, 16, base );
}
void
pnetLog( const char * msg, ... )
{
    va_list     ap;

    SLOG( msg );
    va_start(ap,msg);
        dbg_write( err_pfx[0], msg, ap );
    va_end(ap);
 
    return;
}

void
pnetErr( const char * msg, ... )
{
    va_list     ap;

    va_start(ap,msg);
        err_write( E_FATAL, err_pfx[0], msg, ap );
    va_end(ap);
 
    return;
}
