/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
  *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: pnetthrd.c,v 1.20 2002/12/22 15:07:51 kingofgib Exp $
 */


/*----------------------------------------------------------------------*
 * filename:		pnetthrd.c
 * created on:		Tue Jul 23 20:47:22 CEST 2002
 * created by:		teddykgb
 * project: 		Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

/*----------------------------------------------------------------------*/
/* Simple thread support.						*/
/* This only works if the OS has PTHREAD support.			*/
/*----------------------------------------------------------------------*/

# ifdef HAVE_PTHREAD_H				/* {{ */

# ifdef PNET_SUN 
# define _REENTRANT
# endif

# include <pthread.h>

# ifdef PNET_SUN 
# include <thread.h>
# endif

typedef pthread_mutex_t	pnet_mutex;
typedef pthread_cond_t 	pnet_cond ;
typedef pthread_key_t   pnet_key  ;

# else 						/* }{ */

typedef void * 		pnet_mutex ;
typedef void * 		pnet_cond  ;
typedef void * 		pnet_key   ;

# endif						/* }} */


struct pnet_thread_lock
{
    pnet_mutex		mutex;
};
struct pnet_thread_cond
{
    pnet_cond		cond;
};
struct pnet_thread_key 
{
    pnet_key		key ;
};

int
pnetThreadCreate( pnet_thread_id *thrd_id,
		  PNET_THREADFUNC fun,
		  void * fun_arg )
{
    int error = 0;

    DBG( dbg("pnetThreadCreate(thrd_id = %X,fun = %X)\n", thrd_id, fun ) );

# ifdef HAVE_PTHREAD_H
    if ( ( error = pthread_create( (pthread_t*)thrd_id, NULL, fun, fun_arg ) ) )
	{ THRDERR( "pnetThreadCreate()", error ); return -1; } 

    if ( ( error = pthread_detach( (pthread_t) *thrd_id ) ) )
    	{ THRDERR( "pnetThreadCreate()", error ); return -1; }
# elif defined HAVE_WINDOWS_THREADS

    if ( ! CreateThread( NULL, 0, (LPTHREAD_START_ROUTINE) fun, fun_arg,
			 0, thrd_id ) )
	{ perr(E_FATAL,"pnetThreadCreate()"); return -1; }
# else
    if ( ! strcmp( pnetSysOSName, "none" ) )
	pnetInit();

    perr( E_FATAL,"Multithreading not implemented on %s %s\n",
    		   pnetSysOSName, pnetSysRelease );
    return -1;
# endif
    
    DBG( dbg("Thread %u created\n", *thrd_id ) );

    return 0;
}

int
pnetThreadDetach( pnet_thread_id tid )
{
    int e = 0;

# ifdef HAVE_PTHREAD_H
    e = pthread_detach( (pthread_t)tid );
# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif

    return - !! e;
}
void
pnetThreadExit( void * pExitVal )
{
    DBG( dbg("pnetThreadExit( thrd_id=%X )\n", pnetThreadId() ) );
# ifdef HAVE_PTHREAD_H
    pthread_exit( pExitVal );
# elif defined HAVE_WINDOWS_THREADS
    ExitThread( 0 );
# endif
}

pnet_thread_id
pnetThreadId( void )
{
# ifdef HAVE_PTHREAD_H    
    return (pnet_thread_id) pthread_self();
# elif defined HAVE_WINDOWS_THREADS
    return (pnet_thread_id) GetCurrentThreadId();
# endif
}

int
pnetThreadJoin( pnet_thread_id tid, void ** pValue )
{
    int e = 0;
# ifdef HAVE_PTHREAD_H
    e = pthread_join( tid, pValue );

    if ( e )
	return -1;
# ifndef PTHREAD_CANCELED
# define PTHREAD_CANCELED	( (void*) 1 )
# endif
    if ( pValue )
    {
/*
 * For some #$!@# reason CYGWIN defines PTHREAD_CANCELED
 * to be empty, so this code can't compile.
 */

# ifndef PNET_CYGWIN
	if ( *pValue == PTHREAD_CANCELED )
# endif
	    *pValue = PNET_THREAD_CANCELED;
    }

# elif defined HAVE_WINDOWS_THREADS

    e = -1;

# endif
    return - !! e;
}

/*----------------------------------------------------------------------*/
/* MUTEX Locks 								*/
/*----------------------------------------------------------------------*/
PNET_THREAD_LOCK
pnetThreadLockCreate( void )
{
    int 		e = 0;
    struct pnet_thread_lock *	lock;

    STDMALLOC( lock, sizeof( *lock ), NULL );

# ifdef HAVE_PTHREAD_H    
    e = pthread_mutex_init( &lock->mutex, NULL );
# elif defined HAVE_WINDOWS_THREADS
    e = -1; 
# endif
    if ( e )
    {
	STDFREE( lock );
	return NULL;
    }

    return lock;
}
int 
pnetThreadLockClose( PNET_THREAD_LOCK lock )
{
    int e = 0;

    DBG( dbg("pnetThreadLockClose(lock=%X)\n") );

# ifdef HAVE_PTHREAD_H    
    e = pthread_mutex_destroy( &lock->mutex );
# elif defined HAVE_WINDOWS_THREADS
    e = -1 ; 
# endif
    if ( e )
	LOGTHRDERR( "pnetThreadLockClose()", e );

    STDFREE( lock );

    return - !!e;
}

int
pnetThreadLockAcquire( PNET_THREAD_LOCK lock )
{
    int e = 0;

# ifdef HAVE_PTHREAD_H    
    e= pthread_mutex_lock( &lock->mutex );
# elif defined HAVE_WINDOWS_THREADS
    e= -1; 
# endif

    return - !!e;
}

int
pnetThreadLockTryAcquire( PNET_THREAD_LOCK lock )
{
    int e = 0;

# ifdef HAVE_PTHREAD_H    
    e= pthread_mutex_trylock( &lock->mutex );
# elif defined HAVE_WINDOWS_THREADS
    e= -1; 
# endif

    return - !!e;
}
int
pnetThreadLockRelease( PNET_THREAD_LOCK lock )
{
    int e = 0;

    DBG( dbg("pnetThreadLockRelease(lock=%X)\n") );

# ifdef HAVE_PTHREAD_H    
    e = pthread_mutex_unlock( &lock->mutex );
# elif defined HAVE_WINDOWS_THREADS
    e = -1; 
# endif
    return - !!e;
}

/*----------------------------------------------------------------------*/
/* Thread Conditions 							*/
/*----------------------------------------------------------------------*/

/*
 * Create a new cond, using the specified attributes.
 */
PNET_THREAD_COND
pnetThreadCondCreate( void * attr )
{
    struct pnet_thread_cond * 	cond;
    int	e = 0;

    STDMALLOC( cond, sizeof( *cond ), NULL );

# ifdef HAVE_PTHREAD_H
    e =  pthread_cond_init( &cond->cond, (const pthread_condattr_t *)attr );
# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif 

    if ( e )
	{ STDFREE( cond ); return NULL; }

    return cond;
}

int
pnetThreadCondFree( PNET_THREAD_COND pcond )
{
    int e = 0;

# ifdef HAVE_PTHREAD_H    
    e = pthread_cond_destroy( &pcond->cond );
# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif

    STDFREE( pcond );

    return - !! e;
}
int
pnetThreadCondSignal( PNET_THREAD_COND pcond )
{
    int e;

# ifdef HAVE_PTHREAD_H
    e = pthread_cond_signal( &pcond->cond );
# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif

    return - !! e;
}
int
pnetThreadCondBroadcast( PNET_THREAD_COND pcond )
{
    int e;

# ifdef HAVE_PTHREAD_H
    e = pthread_cond_broadcast( &pcond->cond );
# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif

    return - !! e;
}

int
pnetThreadCondWaitTimed( PNET_THREAD_COND pcond,
                         PNET_THREAD_LOCK plock,
                         int sec, long nsec )
{
    int e;

    DBG( printf( "pnetThreadCondWaitTimed( pcond=%lX, plock=%lX, "
	    "sec=%d, nsec = %d )\n", XX(pcond), XX(plock), sec, nsec ) );

# ifdef HAVE_PTHREAD_H						/* {{ */

    if ( sec < 0 )              /* No timeout, wait forever */
        e = pthread_cond_wait( &pcond->cond, &plock->mutex );
    else
    {
        struct timespec tspec;
# if defined STDLinux || ! defined HAVE_CLOCK_GETTIME	/* {{ */
	struct timeval  tv;

	gettimeofday( &tv, NULL );

	tspec.tv_sec 	= tv.tv_sec;
	tspec.tv_nsec	= tv.tv_usec * 1000;
# else
        if ( clock_gettime( CLOCK_REALTIME, &tspec ) )
            return -1;
# endif 						/* }} */
        tspec.tv_sec    += (time_t)sec;
        tspec.tv_nsec   += nsec;

        e = pthread_cond_timedwait( &pcond->cond, &plock->mutex, &tspec );
    }

# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif

    return - !! e;
}

int
pnetThreadCondWait( PNET_THREAD_COND pcond, PNET_THREAD_LOCK plock )
{
    return pnetThreadCondWaitTimed( pcond, plock, -1, 0 );
}

/*----------------------------------------------------------------------*/
/* Specify what to do with a signal. SIGNUM is a well defined signal 	*/
/* number, while action is one of SIG_BLOCK, SIG_UNBLOCK, SIG_SETMASK	*/
/*----------------------------------------------------------------------*/
int
pnetThreadSignal( int signum, int action )
{
    int e;

# if defined HAVE_PTHREAD_H && ! defined PNET_APPLE

    sigset_t    set, oldset;

    if ( action == PNET_SIG_BLOCK )
	action = SIG_BLOCK;
    else if ( action == PNET_SIG_UNBLOCK )
	action = SIG_UNBLOCK;
    else
	return -1;

    sigemptyset( &set );
    sigaddset( &set, signum );

    e = pthread_sigmask( action, &set, &oldset );

# else
    e = -1;
# endif

    return - !! e;
}

/*----------------------------------------------------------------------*/
/* Send a signal to a thread.						*/
/*----------------------------------------------------------------------*/
int
pnetThreadKill( pnet_thread_id tid, int signum )
{
    int e;
# if defined HAVE_PTHREAD_H && ! defined PNET_APPLE
    e = pthread_kill( (pthread_t)tid, signum );
# else 
    e = -1;
# endif
    return - !! e;
}

/*----------------------------------------------------------------------*/
/* Thread specific data.						*/
/*----------------------------------------------------------------------*/

/*
 * Create a new thread key. Destructor is called whenever a thread exits
 * and the key value is not NULL. A new key always has a NULL associated 
 * with its value. A new key is visible to all threads in the process.
 */
PNET_THREAD_KEY
pnetThreadKeyCreate( PNET_THREAD_Destructor destructor )
{
    struct pnet_thread_key	* pkey;
    int e = 0;

    STDMALLOC( pkey, sizeof( *pkey ), NULL );

# ifdef HAVE_PTHREAD_H
    e =  pthread_key_create( &pkey->key, destructor );
# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif 

    if ( e )
	{ STDFREE( pkey ); return NULL; }

    return pkey;
}
/*
 * Delete this key. Can be called from insided destructors as it never
 * invokes destructors itself.
 */
int
pnetThreadKeyDelete( PNET_THREAD_KEY pkey )
{
    int e = 0;
# ifdef HAVE_PTHREAD_H
    e = pthread_key_delete( pkey->key );
# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif
    return - !! e;
}

/*
 * Associate a thread-specific value with a key obtained by calling
 * pnetThreadKeyCreate(). Different threads may bind different values to the
 * same key. May be called from thread-specific data destructors (may lead
 * to memory leaks or infinite loops, be warned.
 */
int
pnetThreadSetSpecific( PNET_THREAD_KEY pkey, const void * value )
{
    int e = 0;

# ifdef HAVE_PTHREAD_H
    e = pthread_setspecific( pkey->key, value );
# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif

    return - !! e;
}

/*
 * Get the thread-specific value associated with this key. NULL is returned
 * if no value was found for this key.
 */
void *
pnetThreadGetSpecific( PNET_THREAD_KEY pkey )
{
# ifdef HAVE_PTHREAD_H
    return pthread_getspecific( pkey->key );
# elif defined HAVE_WINDOWS_THREADS
    return NULL;
# endif
}

/*----------------------------------------------------------------------*/
/* Thread cancellation							*/
/*----------------------------------------------------------------------*/

int
pnetThreadSetCancelState( int state, int * oldstate )
{
    int e = 0;

# ifdef HAVE_PTHREAD_H

    /* map our values into pthread values. */

    if ( state == PNET_THREAD_CS_Enable )
	state = PTHREAD_CANCEL_ENABLE;
    else if ( state == PNET_THREAD_CS_Disable )
	state = PTHREAD_CANCEL_DISABLE;
    else
	return -1;

    e = pthread_setcancelstate( state, oldstate );

    if ( e )
	return -1;

    /* map pthread values back into our own ones. */

    if ( *oldstate == PTHREAD_CANCEL_ENABLE )
	*oldstate = PNET_THREAD_CS_Enable;
    else if ( *oldstate == PTHREAD_CANCEL_DISABLE )
	*oldstate = PNET_THREAD_CS_Disable;

# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif

    return - !! e;
}

int
pnetThreadSetCancelType( int type, int *oldtype )
{
    int e = 0;

# ifdef HAVE_PTHREAD_H

    /* map our values into pthread values. */

    if ( type == PNET_THREAD_CT_Deferred )
	type = PTHREAD_CANCEL_DEFERRED;
    else if ( type == PNET_THREAD_CT_Async )
	type = PTHREAD_CANCEL_ASYNCHRONOUS;
    else
	return -1;

    e = pthread_setcanceltype( type, oldtype );

    if ( e )
	return -1;

    /* map pthread values back into our own ones. */

    if ( *oldtype == PTHREAD_CANCEL_DEFERRED )
	*oldtype = PNET_THREAD_CT_Deferred;
    else if ( *oldtype == PTHREAD_CANCEL_ASYNCHRONOUS )
	*oldtype = PNET_THREAD_CT_Async;

# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif

    return - !! e;
}

int
pnetThreadCancel( pnet_thread_id tid )
{
    int e = 0;
# ifdef HAVE_PTHREAD_H
    e = pthread_cancel( tid );
# elif defined HAVE_WINDOWS_THREADS
    e = -1;
# endif

    return - !! e;
}
void
pnetThreadTestCancel( void )
{
# ifdef HAVE_PTHREAD_H
    pthread_testcancel();
# elif defined HAVE_WINDOWS_THREADS
# endif
}
