/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: pneticmp.c,v 1.23 2002/10/10 12:27:08 kingofgib Exp $
 */


/*----------------------------------------------------------------------*
 * filename:		pneticmp.c
 * created on:		Thu Jun  6 07:02:03 CEST 2002
 * created by:		teddykgb
 * project: 		Portable Network Library
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/* Manipulation of ICMP messages 					*/
/*----------------------------------------------------------------------*/

# include "local.h"
# include "pkt.h"

# ifdef HAVE_NETINET_ICMP6_H
#    include <netinet/icmp6.h>
# endif

/*----------------------------------------------------------------------*/
/* Send/Recv ICMP packets						*/
/*----------------------------------------------------------------------*/
static int
icmp_send_ip( PNetSocket *ps, PNetAddr* pa,
	      int type, int code, void * p, int plen )
{
    byte		buf[ 256 ];
    struct pnet_icmp* 	icmp;
    int			len = 0;

    icmp = (struct pnet_icmp*) buf;

    memset( icmp, 0, sizeof( icmp ) );

    if ( type == ICMP_ECHO )
    {
	icmp->icmp_type	= type;
	icmp->icmp_code	= 0;
	icmp->icmp_id	= sys_getpid();
	icmp->icmp_seq	= (int)p;
	gettimeofday( (struct timeval*)icmp->icmp_data, NULL );
	len 		= sizeof( struct timeval );
    }
    else if ( type == ICMP_UNREACH || type == ICMP_TIMXCEED )
    {
	icmp->icmp_type	= type;
	icmp->icmp_code	= code;
	icmp->icmp_void	= 0;
	len = PMIN( sizeof( buf ), (u_int) plen );
	memcpy( icmp->icmp_data, p, len );
    }
    else
    {
	TYPEERR("icmp_send_ip()\n");
	return -1;
    }

    len += sizeof( struct pnet_icmp );

    icmp->icmp_cksum = pnetIP_ChecksumFold( 
	pnetIP_ChecksumAdd( (pnet_ushort*) icmp, len, 0 ) );

    return pnetWriteTo( ps, pa, (char*) buf, len );
}
static int
icmp_send_ipv6( PNetSocket *ps, PNetAddr* pa,
		int type, int code, void *p, int plen )
{
    byte	buf[128];
    pnet_icmp6*	icmp6;
    int		len = 0;

    icmp6 = (pnet_icmp6*) buf;

    memset( icmp6, 0, sizeof( icmp6 ) );

    icmp6->icmp6_cksum	= 0;

    if ( type == ICMP_ECHO )
    {
	icmp6->icmp6_type= ICMP6_ECHO_REQUEST;	/* Translate type */
	icmp6->icmp6_code= 0;
	icmp6->icmp6_id	 = htons( (pnet_ushort)sys_getpid() );
	icmp6->icmp6_seq = htons( (pnet_ushort)((int)p) );
	gettimeofday( (struct timeval*)(icmp6 + 1), NULL );

	len = sizeof(struct timeval);
    }
    else if ( type == ICMP_UNREACH || type == ICMP_TIMXCEED )
    {
	icmp6->icmp6_type = type == ICMP_UNREACH ? ICMP6_DST_UNREACH :
			    ICMP6_TIME_EXCEEDED;
	icmp6->icmp6_code = code;
	len = PMIN( sizeof( buf ), (u_int) plen );
	memcpy( icmp6 + 1, p, len );
    }
    else
    {
	TYPEERR("icmp_send_ipv6()\n");
	return -1;
    }

    len += 8;	/* Account for 8 byte ICMP header */

    if ( pnetWriteTo( ps, pa, (char*) buf, len ) != len )
    {
	perr(E_FATAL,"icmp_send_ipv6() error sending %d bytes\n", len );
	return -1;
    }
    return 0;
}

int
pnetICMPSend( PNETSOCK ps, PNETADDR pa,
	int type, int code, void * p, int plen )
{
    DBG( dbg("pnetICMPSend(ps=%lX,ps=%lX,type=%d,code=%d,p=%lX,plen=%d)\n",
	      XX(ps),XX(pa),type,code,XX(p),plen) );

    if ( ps->fam == AF_INET )
	return icmp_send_ip  ( ps, pa, type, code, p, plen );

    if ( ps->fam == AF_INET6 )
	return icmp_send_ipv6( ps, pa, type, code, p, plen );
    
    FAMERR("pnetICMPSend()");

    return -1;
}
int
pnetICMPSendEchoReq( PNETSOCK ps, PNETADDR pa, int seq )
{
    return pnetICMPSend( ps, pa, ICMP_ECHO, 0, (int*)seq, 0 );
}
/*----------------------------------------------------------------------*/
/* p: internet header + 64 bits of original data datagram 		*/
/*----------------------------------------------------------------------*/
int
pnetICMPSendDestUnreach( PNETSOCK ps, PNETADDR pa,
			 int code, void *p, int plen  )
{
    return pnetICMPSend( ps, pa, ICMP_UNREACH, code, p, plen );
}
int
pnetICMPSendTimeX( PNETSOCK ps, PNETADDR pa,
		   int code, void *p, int plen )
{
    return pnetICMPSend( ps, pa, ICMP_TIMXCEED, code, p, plen );
}

/*----------------------------------------------------------------------*/
/* ICMPv6 Message Filters						*/
/*----------------------------------------------------------------------*/

PNET_ICMPv6_Filter
pnetICMP_FilterOpen( void )
{
# ifdef ICMP6_FILTER
    struct icmp6_filter	* filter;

    STDMALLOC( filter, sizeof( struct icmp6_filter ), NULL );


    ICMP6_FILTER_SETBLOCKALL( filter );


    return filter;
# endif
    return NULL;
}
void
pnetICMP_FilterClose( PNET_ICMPv6_Filter filter )
{
    STDFREE( filter );
}
void
pnetICMP_FilterSetPass( PNET_ICMPv6_Filter filter, int msg_type )
{
# ifdef ICMP6_FILTER
    ICMP6_FILTER_SETPASS( msg_type, filter );
# endif
}
void
pnetICMP_FilterSetBlock( PNET_ICMPv6_Filter filter, int msg_type )
{
# ifdef ICMP6_FILTER
    ICMP6_FILTER_SETBLOCK( msg_type, filter );
# endif
}
int
pnetICMP_FilterWillPass( PNET_ICMPv6_Filter filter, int msg_type )
{
# ifdef ICMP6_FILTER
    return ICMP6_FILTER_WILLPASS( msg_type, filter );
# endif
    return 0;
}
int
pnetICMP_FilterWillBlock( PNET_ICMPv6_Filter filter, int msg_type )
{
# ifdef ICMP6_FILTER
    return ICMP6_FILTER_WILLBLOCK( msg_type, filter );
# endif
    return 0;
}
int
pnetICMP_FilterInstall( PNETSOCK ps, PNET_ICMPv6_Filter filter )
{
# ifdef ICMP6_FILTER
    if ( setsockopt( ps->sd, IPPROTO_ICMPV6, ICMP6_FILTER, filter,
		    sizeof( struct icmp6_filter) ) )
	{ FATALERR( "pnetICMP_FilterInstall()" ); return -1; }

    pdbg(E_DBG1,"Installed ICMP filter on socket %lX(%d)\n",XX(ps),ps->sd);
# endif

    return 0;
}
/*----------------------------------------------------------------------*/
/* Get string representations of ICMP messages 				*/
/*----------------------------------------------------------------------*/

const char *
pnetICMP_TypeDesc( int proto, int type )
{
    if ( proto == PNET_IPPROTO_ICMP )
    {
	switch( type )
	{
	case ICMP_ECHOREPLY:		return "Echo reply";
	case ICMP_UNREACH:		return "Dest. unreachable";
	case ICMP_SOURCEQUENCH:		return "Source quench";
	case ICMP_REDIRECT:		return "Redirect";
	case ICMP_ECHO:			return "Echo request";
	case ICMP_ROUTERADVERT:		return "Router advert";
	case ICMP_ROUTERSOLICIT:	return "Router solicit";
	case ICMP_TIMXCEED:		return "Timer exceeded";
	case ICMP_PARAMPROB:		return "Param. problem";
	case ICMP_TSTAMP:		return "Timestamp request";
	case ICMP_TSTAMPREPLY:		return "Timestamp reply";
	case ICMP_IREQ:			return "Info request";
	case ICMP_IREQREPLY:		return "Info reply";
	case ICMP_MASKREQ:		return "Address mask request";
	case ICMP_MASKREPLY:		return "Address mask reply";
	}
	return "ICMP Other";
    }
    else if ( proto == PNET_IPPROTO_ICMPV6 )
    {
	switch( type )
	{ 
	case ICMP6_DST_UNREACH:     return "Dest. Unreach";
	case ICMP6_PACKET_TOO_BIG:  return "Pkt too big";
	case ICMP6_TIME_EXCEEDED:   return "Time exceeded";
	case ICMP6_PARAM_PROB:      return "Param. problem";
	case ICMP6_ECHO_REQUEST:    return "Echo request";
	case ICMP6_ECHO_REPLY:      return "Echo reply";
	case ICMP6_MEMBERSHIP_QUERY: return "Membership query";
	case ICMP6_MEMBERSHIP_REPORT: return "Membership report";
	case ICMP6_MEMBERSHIP_REDUCTION: return "Group membership termination";

	case ND_ROUTER_SOLICIT:	return "Router solicit";
	case ND_ROUTER_ADVERT:	return "Router advert";
	case ND_NEIGHBOR_SOLICIT:	return "Neighbor solicit";
	case ND_NEIGHBOR_ADVERT:	return "Neighbor advert";
	case ND_REDIRECT:		return "Redirect";

	case ICMP6_ROUTER_RENUMBERING: return "Router renumbering";
	}
	return "ICMPv6 Other";
    }

    PROTOERR( "pnetICMP_TypeDesc()" );
    return NULL;
}

const char *
pnetICMP_CodeDesc( int proto, int type, int code )
{
    if ( proto == PNET_IPPROTO_ICMP )
    {
        if ( type == ICMP_UNREACH )
        {
	    switch( code )
	    {
	    case ICMP_UNREACH_NET:	return "bad net";
	    case ICMP_UNREACH_HOST:	return "bad host";
	    case ICMP_UNREACH_PROTOCOL:	return "bad protocol";
	    case ICMP_UNREACH_PORT:	return "bad port";
	    case ICMP_UNREACH_NEEDFRAG:	return "DF caused drop";
	    case ICMP_UNREACH_SRCFAIL:	return "src route failed";
	    case ICMP_UNREACH_NET_UNKNOWN: return "unknown net";
	    case ICMP_UNREACH_HOST_UNKNOWN: return "unknown host";
	    case ICMP_UNREACH_NET_PROHIB: return "prohib. net";
	    case ICMP_UNREACH_HOST_PROHIB: return "prohib. host";
	    case ICMP_UNREACH_TOSNET:	return "bad tos for net";
	    case ICMP_UNREACH_TOSHOST:	return "bad tos for host";
	    case ICMP_UNREACH_FILTER_PROHIB:	return "admin. prohib.";
	    case ICMP_UNREACH_HOST_PRECEDENCE: return "host preced. violation";
	    case ICMP_UNREACH_PRECEDENCE_CUTOFF: return "host preced. cutoff";
	    }
	}
	else if ( type == ICMP_REDIRECT )
	{
	    switch( code )
	    {
	    case ICMP_REDIRECT_NET:	return "for net";
	    case ICMP_REDIRECT_HOST:	return "for host";
	    case ICMP_REDIRECT_TOSNET:	return "for tos and net";
	    case ICMP_REDIRECT_TOSHOST:	return "for tos and host";
	    }
	}
	else if ( type == ICMP_TIMXCEED )
	{
	    if ( code == ICMP_TIMXCEED_INTRANS )
		return "in transit";
	    if ( code == ICMP_TIMXCEED_REASS )
		return "in reassembly";
	}
	else if ( type == ICMP_PARAMPROB )
	{
	    if ( code == ICMP_PARAMPROB_ERRATPTR )
		return "err. at param. pointer";
	    if ( code == ICMP_PARAMPROB_OPTABSENT )
		return "req. opt. missing";
	    if ( code == ICMP_PARAMPROB_LENGTH )
		return "bad length";
	}
    }

    return "";
}
