/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: pnetaddr.c,v 1.44 2002/11/05 08:35:35 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:            pnetaddr.c
 * created on:          Sun May 19 13:06:19 CEST 2002
 * created by:          teddykgb
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

/*----------------------------------------------------------------------*/
/* Convert a SockAddr to presentation format				*/
/* This is a dirty, non-reentrant function, meant to be used for quick	*/
/* print out of a socket addr struct.					*/
/*----------------------------------------------------------------------*/

# ifdef HAVE_NET_IF_DL_H
# include <net/if_dl.h>
# endif

const char*
net_sockaddrtop(void * sa)
{
    static char buf[128];

    int fam = ((SockAddr*)sa)->sa_family;

    if (fam == AF_INET)
	inet_ntop(fam,&((InetAddr*)sa)->sin_addr,buf,sizeof(buf));
    else if (fam == AF_INET6)
	inet_ntop(fam,&((InetAddr6*)sa)->sin6_addr,buf,sizeof(buf));

# ifdef PNET_HAVE_LOCAL
    else if (fam == AF_LOCAL)
    {
	struct sockaddr_un * un = (struct sockaddr_un*) sa;

	snprintf(buf,sizeof(buf),"%s",un->sun_path);
    }
# endif 

# ifdef AF_LINK
    else if (fam == AF_LINK)
    {
	struct sockaddr_dl *sdl = (struct sockaddr_dl*)sa;
	byte* p = (byte*) LLADDR(sdl);
	char * pbuf = buf;

	/* we output "rl0: AABBCCDDEEFF"		*/
	memcpy(pbuf,sdl->sdl_data,sdl->sdl_nlen);
	pbuf += sdl->sdl_nlen;
	*pbuf++ = ':';
	*pbuf++ = ' ';

	memcpy(pbuf,p,sdl->sdl_alen);
	pbuf[sdl->sdl_alen] = 0;
    }
# endif
    else
    {
	FAMERR("net_sockaddrtop()");
	return "";
    }
    return buf;
}
/*----------------------------------------------------------------------*/
/* Convert a string consisting of a dotted IP address to a PNETADDR 	*/
/*----------------------------------------------------------------------*/
int
pnet_pton( int pfam, const char *buf, PNETADDR pa )
{
    int ret = 0;
    int fam;

    fam = pfam2fam(pfam);

    if (fam == AF_INET)
    {
	pa->pa_len = sizeof(InetAddr);
	ret = inet_pton(fam,buf,(void*)&pa->pa_in4addr.sin_addr);
    }
# ifdef PNET_HAVE_IPv6
    else if (fam == AF_INET6)
    {
	pa->pa_len = sizeof(InetAddr6);
	ret = inet_pton(fam,buf,(void*)&pa->pa_in6addr.sin6_addr);
    }
# endif
# ifdef PNET_HAVE_LOCAL
    else if (fam == AF_LOCAL)
    {
	struct sockaddr_un	* un	= &pa->pa_unaddr;

	un->sun_family	= fam;
	strncpy(un->sun_path,buf,strlen(buf));
	pa->pa_len	= SUN_LEN(un);
    }
# endif
    else
	{ FAMERR("pnet_pton()"); return -1; }

    if (ret < 0)		/* Address is not dotted format */
	{ return -1; }

    if (ret != 1)
	return -1;		/* Something else is wrong	*/

    pa->pa_family = fam;
    pa->init_addr = 1;
    return 0;
}
/*----------------------------------------------------------------------*/
/* Convert a PNetAddr to a string					*/
/* Again, as with net_sockaddrtop, a quick and dirty, non-reentrant way */
/* to print out a PNETADDR struct in a human readable format.		*/
/*----------------------------------------------------------------------*/
char *
net_ntop( PNETADDR pa )
{
    static char 	buf[128];
    return pnetAddrToString( pa, buf, sizeof( buf ) );
}
char *
pnet_ntop( PNETADDR pa, char * buf, int blen )
{
    return pnetAddrToString( pa, buf, blen );
}

const char *
pnet_inet_ntop( int pfam, pnet_byte * p_sin_addr, char * buf, int blen )
{
    return inet_ntop( pfam2fam( pfam ), p_sin_addr, buf, blen );
}
int
pnet_inet_pton( int pfam, pnet_byte * p_sin_addr, char * buf )
{
    return - ( inet_pton( pfam2fam( pfam) , buf, p_sin_addr ) != 1 );
}

char *
pnetAddrToString( PNETADDR pa, char *buf, pnet_uint len )
{
    int written;

    if ( ! pa )
	return NULL;

    if ( ! pa->init_addr )
    {
	snprintf( buf, len, "-- no addr --" );
	return buf;
    }
    
    if ( pa->pa_family == AF_INET )
    {
        if (! inet_ntop( AF_INET, &pa->pa_sin_addr, buf, len ) )
	    goto err;

	written = strlen( buf );

        if ( pa->init_port )
        {
	    if ( written + sizeof( ":XXXXX" ) > len )
		goto err;

            snprintf( buf + written, sizeof(":XXXXX"),
	    			     ":%5d", ntohs( pa->pa_sin_port ) );
        }
	return buf;
    }
    else if ( pa->pa_family == AF_INET6 )
    {
	const char * p = NULL;

	if (! inet_ntop( AF_INET6, &pa->pa_sin6_addr, buf, len ) )
	    goto err;

	written = strlen( buf );

	if ( pa->pa_scope_id )
	{
	    char tmp[11];

	    sprintf(tmp, "%u", pa->pa_scope_id );

	    if ( pnetAddrIsLinkLocal( pa ) || pnetAddrIsMCLinkLocal( pa ) )
		p = pnetIfIndexToName( pa->pa_scope_id );

	    if ( ! p )
		p = tmp;

	    if ( written + strlen( p ) + 1 > len )
		goto err;

	    buf[ written++ ] = pnet_ScopeSep;

	    strcpy( buf + written, p );

	    written += strlen( p );
	}

	if (pa->init_port)
	{
	    if ( written + sizeof( ":XXXXXX" ) > len )
	    	goto err;

	    snprintf( buf + written, sizeof( ":XXXXX" ),
	    		  ":%5d",ntohs(pa->pa_sin6_port) );
	}

	return buf;
    }
# ifdef PNET_HAVE_LOCAL
    else if ( pa->pa_family == AF_LOCAL)
    {
	struct sockaddr_un * un = &pa->pa_unaddr;

	memset( buf, 0, len );
	strncpy( buf, un->sun_path, len - 1 );

	return buf;
    }
# endif
	{ FAMERR("pnet_ntop()"); return NULL; }

    FAMERR("pnetAddrToString()");

    return NULL;

err:
    SPACEERR("pnetAddrToString()");
    return NULL;
}

/*----------------------------------------------------------------------*/
/* Set the sin_addr or sin6_addr member of a PNetAddr struct.		*/
/* Also, set the other members of the sockaddr struct properly.		*/
/* ptr should be a pointer to a struct in_addr (4 byte IPv4 address) for*/
/* IPv4 pnet_addr, and a pointer to a struct in6_addr for an IPv6 	*/
/* pnet_addr. The pfam parameter must be one of the PNET_IPv4/IPv6 etc  */
/* and not the AF_XXX constants. (These are not "exported" outside 	*/
/* pnet).								*/
/*----------------------------------------------------------------------*/
int
addr_set_ip_addr(PNETADDR pa,int fam,const void *ptr,int len)
{
    DBG( dbg("addr_set_ip_addr()\n"));
    memset(pa,0,sizeof(PNetAddr));

    if (fam == AF_INET)
    {
	pa->pa_len = sizeof(InetAddr);
	memcpy(&pa->pa_sin_addr,ptr,len);
    }
# ifdef PNET_HAVE_IPv6
    else if (fam == AF_INET6)
    {
	pa->pa_len = sizeof(InetAddr6);
	memcpy(&pa->pa_sin6_addr,ptr,len);
    }
# endif

# ifdef PNET_HAVE_LOCAL
    else if (fam == AF_LOCAL)
    {
	struct sockaddr_un	* un	= &pa->pa_unaddr;

	strncpy(un->sun_path,ptr,len);
	pa->pa_len = SUN_LEN(un);
    }
# endif
    else
	{ FAMERR("addr_set_ip_addr()"); return -1; }

    pa->pa_family = fam;
    pa->init_addr = 1;

    return 0;
}

void
pnetAddrCopy(PNETADDR tgt,PNETADDR src)
{
    memcpy( tgt, src, sizeof(PNetAddr) );
}
/*----------------------------------------------------------------------*/
/* Creates a new pnet_addr, and fills in the correctly resolved host	*/
/* IP address.								*/
/* The host can be a host name, such as "host.tsps1.freenet6.net'	*/
/* or an IP address, such as '10.0.0.1' or 'ff0e::1'. If the address	*/
/* is a link local address, it must be postfixed by a '%<interface>'	*/
/* which gives the name of the interface to use with this address.	*/
/*----------------------------------------------------------------------*/
PNETADDR
pnetAddrResolve( int pfam, const char * host )
{
    PNetAddr    tmp;
    PNetAddr *	pa;

    if ( net_gethostbyname( pfam2fam( pfam ), host, &tmp ) )
	{ return NULL; }

    STDMALLOC(pa,sizeof(PNetAddr),NULL);

    memcpy( pa, &tmp, sizeof( PNetAddr ) );

    return pa;
}
PNETADDR
pnetAddrMake( void )
{
    PNetAddr * pa;

    STDMALLOC( pa, sizeof(PNetAddr), NULL );

    return pa;
}
PNETADDR
pnetAddrNext( PNETADDR pa )
{
    if ( ! pa )
	return NULL;
    return  pa->pa_next;
}
int
pnetAddrToHostname( PNETADDR pa, char *buf, int blen )
{
    if ( ! pa->init_addr )
	return -1;
    return net_gethostbyaddr( pa->pa_family, pa, buf, blen );
}
void
pnetAddrFree( PNETADDR pa )
{
    PNETADDR tmp;

    while ( pa )
    {
	tmp = pa->pa_next;
	STDFREE( pa );
	pa = tmp;
    }
}

int
pnetAddrGetFamily( PNETADDR pa )
{
    if ( ! pa->init_addr )
	return -1;

    return fam2pfam( pa->pa_family );
}
int
pnetAddrGetPort( PNETADDR pa )
{
    return (int)ntohs( (pnet_ushort)addr_get_port( pa ) );
}
pnet_byte*
pnetAddrGetBinary( PNETADDR pa, int * pLen )
{
    if ( ! pa->init_addr )
	return NULL;

    if ( pa->pa_family == AF_INET )
    {
	if ( pLen )
	    *pLen = sizeof( pa->pa_s_addr );
	return (pnet_byte*) &pa->pa_s_addr;
    }
# ifdef PNET_HAVE_IPv6
    else if ( pa->pa_family == AF_INET6 )
    {
	if ( pLen )
	    *pLen = sizeof( pa->pa_s6_addr );
	return (pnet_byte*) &pa->pa_s6_addr;
    }
# endif
    return NULL;
}
int
pnetAddrLookupPort( PNETADDR pa, const char * type, const char * serv )
{
    int	port;

    if ( ( port = net_servicetoport( serv, type ) ) < 0 )
	return -1;

    addr_set_port( pa, port );

    return 0;
}

int
pnetAddrSetPort( PNETADDR pa, int port )
{
    return addr_set_port( pa, port );
}

int
net_servicetoport( const char * serv, const char * type )
{
    int		port;
    char 	c;

    if ( ! serv || ! *serv )
	return 0;

    if ( sscanf( serv, "%d%c", &port, &c) != 1 		&&
         net_getservbyname( serv, type, (int*) &port ) 	)
    {
        perr(E_FATAL,"net_servicetoport(): '%s': %s\n",serv,neterr());
        return -1;
    }

    return port;
}
/*----------------------------------------------------------------------*/
/* Some checks for IPv6 addresses.					*/
/*----------------------------------------------------------------------*/
int
pnetAddrIsUnspecified( PNETADDR pa )
{
    if ( pa->pa_family == AF_INET )
	return pa->pa_sin_addr.s_addr == 0x00000000;
# ifdef IN6_IS_ADDR_UNSPECIFIED 
    return pa->pa_family == AF_INET6 && 
	   IN6_IS_ADDR_UNSPECIFIED( &pa->pa_sin6_addr );
# endif
}
int
pnetAddrIsLoopback( PNETADDR pa )
{
    if ( pa->pa_family == AF_INET )
	return *(byte*)&pa->pa_sin_addr.s_addr == 0xFF;

# ifdef IN6_IS_ADDR_LOOPBACK 
    return pa->pa_family == AF_INET6 && 
	   IN6_IS_ADDR_LOOPBACK( &pa->pa_sin6_addr );
# endif
}
int
pnetAddrIsLinkLocal( PNETADDR pa )
{
# if defined IN6_IS_ADDR_LINKLOCAL  || defined PNET_WIN32
    return pa->pa_family == AF_INET6 && 
	   IN6_IS_ADDR_LINKLOCAL( &pa->pa_sin6_addr );
# endif
}
int
pnetAddrIsMCLinkLocal( PNETADDR pa )
{
# if defined IN6_IS_ADDR_MC_LINKLOCAL  || defined PNET_WIN32
    return pa->pa_family == AF_INET6 && 
	   IN6_IS_ADDR_MC_LINKLOCAL( &pa->pa_sin6_addr );
# endif
}
int
pnetAddrIsSiteLocal( PNETADDR pa )
{
# if defined IN6_IS_ADDR_SITELOCAL || defined PNET_WIN32
    return pa->pa_family == AF_INET6 && 
	   IN6_IS_ADDR_SITELOCAL( &pa->pa_sin6_addr );
# endif
}
int
pnetAddrIsV4Mapped( PNETADDR pa )
{
# if defined IN6_IS_ADDR_V4MAPPED || defined PNET_WIN32
    return pa->pa_family == AF_INET6 && 
	   IN6_IS_ADDR_V4MAPPED( &pa->pa_sin6_addr );
# endif
}
int
pnetAddrIsV4Compatible( PNETADDR pa )
{
# if defined IN6_IS_ADDR_V4COMPAT || defined PNET_WIN32
    return pa->pa_family == AF_INET6 && 
	   IN6_IS_ADDR_V4COMPAT( &pa->pa_sin6_addr );
# endif
}
/*----------------------------------------------------------------------*/
/* Return the length of a pnet_addr					*/
/*----------------------------------------------------------------------*/
int
addr_len(int fam)
{
    /* Note that we can't use a switch, cause AF_INET6 is mapped to 	*/
    /* AF_INET4 on platforms that don't support IPv6.			*/

    if (fam == AF_INET)
	return sizeof(InetAddr);
    if (fam == AF_INET6)
	return sizeof(InetAddr6);

# ifdef PNET_HAVE_LOCAL
    if (fam == AF_LOCAL)
	return sizeof(UnixAddr);
# endif

    return 0;
}
/*----------------------------------------------------------------------*/
/* Return the length of the IP address portion of an PNetAddr		*/
/*----------------------------------------------------------------------*/
int
addr_ip_len(int fam)
{
    if (fam == AF_INET)
	return sizeof(struct in_addr);
    if (fam == AF_INET6)
	return sizeof(struct in6_addr);
# ifdef PNET__HAVE_LOCAL
    if (fam == AF_LOCAL)
	return sizeof(UnixAddr);
# endif

    return 0;
}
/*----------------------------------------------------------------------*/
/* Set the port of a PNetAddr						*/
/*----------------------------------------------------------------------*/
int
addr_set_port(PNetAddr * pa,int port)
{
    PNETADDR 	pacurr;

    DBG(dbg("addr_set_port(pa=%lX,port=%lX)\n",XX(pa),XX(port)));

    pacurr = pa;

    for ( pacurr = pa; pacurr; pacurr = pacurr->pa_next )
    {
# ifdef PNET_HAVE_LOCAL
	/* No ports for LOCAL sockets */
	if (pacurr->pa_family == AF_LOCAL)
	    continue;
# endif
	if (pacurr->pa_family == AF_INET)
	    pacurr->pa_sin_port  = htons((pnet_ushort)port);
	else if (pa->pa_family == AF_INET6)
	    pacurr->pa_sin6_port = htons((pnet_ushort)port);
	else
	    { FAMERR("addr_set_port()"); continue; }
  
	pacurr->init_port = 1;
    }

    return 0;
}

int
addr_get_port( PNetAddr* pa )
{
    if ( ! pa->init_addr )
    	return -1;

    if ( pa->pa_family == AF_INET )
	return pa->pa_sin_port;
    else if ( pa->pa_family == AF_INET6 )
	return pa->pa_sin6_port;
    return -1;
}

/*----------------------------------------------------------------------*/
/* Make a pnet_addr from a given SockAddr				*/
/*----------------------------------------------------------------------*/
int
addr_from_sockaddr(PNetAddr *pa,SockAddr *sa)
{
    size_t len;

    pa->pa_family  = sa->sa_family;

# ifdef HAVE_SOCKADDR_SA_LEN
    len = sa->sa_len;
# else
    len = addr_len(sa->sa_family);
# endif

    LLOG( len );

    memcpy(&pa->pa_saddr,sa,len);

    pa->pa_len  = len;
    pa->init_addr = 1;
    pa->init_port = 1;

    return 0;
}
/*----------------------------------------------------------------------*/
/* Make a listen address. Port number should be an string containing	*/
/* an integer for IPv4 and IPv6; it should be a path for LOCAL		*/
/* sockets. The third argument can be an interface name, e.g. "rl0", 	*/
/* "eth0", or whatever interface name was obtained through		*/
/* pnetGetIfInfo(); or it can be the IP address of the given interface.	*/
/* The IP address can be IP or IPv6; in case of a IPv6 link local	*/
/* address, it should be postfixed by '%iface'---that is the interface  */
/* name or index on which we should access the local link.		*/
/*----------------------------------------------------------------------*/

int
addr_extract_scope( const char * addr, int no_prefix )
{
    const char * p;
    char   	 c;
    int    	 idx = -1;

    if ( ! addr )
	return -1;

    if ( ! no_prefix )
    {
	if ( ! ( p = strchr( addr, pnet_ScopeSep ) ) )
	    return -1;
	p++;
    }
    else 
	p = addr;

    if ( ! *p ) 
	goto end;

    if ( sscanf( p,"%d%c", &idx, &c ) != 1 )
    {
	idx = pnetIfNameToIndex( p );
	if ( idx == 0 )
	{
	    perr(E_FATAL,"Interface %s not found\n", p );
	    goto end;
	}
    }

end:
    LLOG( idx );
    return idx;
}
int
addr_make_listen( int fam, PNetAddr *pa,
		  const char * ifname, const char *sport )
{
    int 	ret = 0;

    DBG(dbg("addr_make_listen(fam=%s,pa=%lX,ifname=%s,sport=%s)\n",
	    net_aftoa(fam),XX(pa),ifname,sport));

    memset(pa,0,sizeof(PNetAddr));

    if (fam == AF_INET)
    {
        InetAddr * in = &pa->pa_in4addr;

	if( ! ifname )
            in->sin_addr.s_addr = INADDR_ANY;
	else
	{
	    if ( inet_pton( AF_INET, ifname, &in->sin_addr ) != 1 &&
	    	 net_get_if_ipaddr( AF_INET, ifname, (SockAddr*)in) )
	    {
		perr(E_FATAL,"addr_make_listen(AF_INET4): unknown "
			     "interface %s\n",ifname);
		return -1;
	    }
	}

	pa->pa_family 		= fam;
        pa->pa_len             = sizeof(InetAddr);
    }
    else if (fam == AF_INET6)
    {
        InetAddr6 * in = &pa->pa_in6addr;

        if ( ! ifname )
# ifdef HAVE_IN6ADDR_ANY
            in->sin6_addr       = in6addr_any;
# else
	    memset( &in->sin6_addr, 0, sizeof( in->sin6_addr ) );
# endif
	else
	{
	    char * p = NULL;
	    char * s = strdup( ifname );
	    int scope = addr_extract_scope( ifname, 0 );
	
	    /* Set the actual address 		*/
	    if ( (p = strchr( s, pnet_ScopeSep ) ) )
		*p = 0;

            if ( inet_pton( AF_INET6, s, &in->sin6_addr ) != 1 )
	    {
		/* We have an interface name. Fetch its address here */
	    	if ( net_get_if_ipaddr( AF_INET6, s, (SockAddr*)in) )
		{
		    perr(E_FATAL,"addr_make_listen(AF_INET6): unknown "
				 "interface %s\n",s);
		    free( s );
		    return -1;
		}
	    }

# ifdef HAVE_SOCKADDR_IN6_SIN6_SCOPE_ID
	    if ( scope > -1 )
		in->sin6_scope_id = scope;
# endif
	    if ( p )
		*p = pnet_ScopeSep;
	    free( s );
	}

	pa->pa_family		= fam;
	pa->pa_len		= sizeof(InetAddr6);
    }

# ifdef PNET_HAVE_LOCAL         /* { */

    else if (fam == AF_LOCAL)
    {
        struct sockaddr_un * un = &pa->pa_unaddr;

        /* Get rid of the old path */
        unlink(sport);

        memset(un,0,sizeof(struct sockaddr_un));
        un->sun_family          = AF_LOCAL;

        /* Don't overflow the buffer */
        strncpy(un->sun_path,sport,sizeof(un->sun_path) - 1);

        pa->pa_family           = AF_LOCAL;
	pa->pa_len              = SUN_LEN(un);

    }

# endif                         /* } */

    else
    {
        FAMERR("addr_make_listen()");
        ret = -1;
    }

    /* If strtol() returns zero, the kernel will choose a port on which */
    /* to bind this address. Is this correct? I guess it should be...	*/
    /* Or we can fail, asking for a port number to be specified, but	*/
    /* then, one will never be able to bind to an ephemeral port. 	*/
    /* The value of the ephemeral port can be obtained with 		*/
    /* sock_getsockname() once the kernel has bound this address.	*/

    if ( sport )
	addr_set_port(pa,strtol( sport, NULL, 0) );

    pa->init_addr = 1;

    return ret;
}
